{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 5.1 (Evaluators) of the OpenGL 1.2.1 specs.
-}

module GL_Evaluators (
   Order(..),
   Map1Target(..),
   marshalMap1Target,          -- internal use only
   Map1(..),
   Map2Target(..),
   marshalMap2Target,          -- internal use only
   Map2(..),
   EvalCoord(..), EvalCoordV(..),
   AutoNormal(..),
   MapGrid1(..), MapGrid2(..),
   evalMesh1, evalMesh2,
   evalPoint1, evalPoint2
) where

import Foreign          (Ptr)

import GL_BasicTypes    (GLenum, GLint, GLfloat, GLdouble, Capability(..))
import GL_Constants     (gl_MAP1_VERTEX_3, gl_MAP1_VERTEX_4, gl_MAP1_INDEX,
                         gl_MAP1_COLOR_4, gl_MAP1_NORMAL,
                         gl_MAP1_TEXTURE_COORD_1, gl_MAP1_TEXTURE_COORD_2,
                         gl_MAP1_TEXTURE_COORD_3, gl_MAP1_TEXTURE_COORD_4,
                         gl_MAP2_VERTEX_3, gl_MAP2_VERTEX_4, gl_MAP2_INDEX,
                         gl_MAP2_COLOR_4, gl_MAP2_NORMAL,
                         gl_MAP2_TEXTURE_COORD_1, gl_MAP2_TEXTURE_COORD_2,
                         gl_MAP2_TEXTURE_COORD_3, gl_MAP2_TEXTURE_COORD_4,
                         gl_AUTO_NORMAL)
import GL_Polygons      (PolygonMode, marshalPolygonMode)
import GL_VertexArray   (Stride)
import GL_VertexSpec    (Vertex1(..), Vertex2(..))
import GLU_Constants    (glu_MAP1_TRIM_2, glu_MAP1_TRIM_3)

---------------------------------------------------------------------------

newtype Order = Order GLint

---------------------------------------------------------------------------

-- Including GLU_MAP1_TRIM_2 and GLU_MAP1_TRIM_3 here is a little hack,
-- but it is very convenient. :-)
data Map1Target =
     Map1Vertex3
   | Map1Vertex4
   | Map1Index
   | Map1Color4
   | Map1Normal
   | Map1TextureCoord1
   | Map1TextureCoord2
   | Map1TextureCoord3
   | Map1TextureCoord4
   | Map1Trim2
   | Map1Trim3
   deriving (Eq,Ord)

marshalMap1Target :: Map1Target -> GLenum
marshalMap1Target Map1Vertex3       = gl_MAP1_VERTEX_3
marshalMap1Target Map1Vertex4       = gl_MAP1_VERTEX_4
marshalMap1Target Map1Index         = gl_MAP1_INDEX
marshalMap1Target Map1Color4        = gl_MAP1_COLOR_4
marshalMap1Target Map1Normal        = gl_MAP1_NORMAL
marshalMap1Target Map1TextureCoord1 = gl_MAP1_TEXTURE_COORD_1
marshalMap1Target Map1TextureCoord2 = gl_MAP1_TEXTURE_COORD_2
marshalMap1Target Map1TextureCoord3 = gl_MAP1_TEXTURE_COORD_3
marshalMap1Target Map1TextureCoord4 = gl_MAP1_TEXTURE_COORD_4
marshalMap1Target Map1Trim2         = glu_MAP1_TRIM_2
marshalMap1Target Map1Trim3         = glu_MAP1_TRIM_3

class Map1 a where
   map1 :: Map1Target
        -> (a, a) -> Stride -> Order
        -> Ptr b -> IO ()

instance Map1 GLfloat  where
   map1 t (u1, u2) = glMap1f (marshalMap1Target t) u1 u2

instance Map1 GLdouble where
   map1 t (u1, u2) = glMap1d (marshalMap1Target t) u1 u2

foreign import "glMap1f" unsafe glMap1f ::
      GLenum
   -> GLfloat  -> GLfloat  -> GLint -> Order
   -> Ptr a -> IO ()

foreign import "glMap1d" unsafe glMap1d ::
      GLenum
   -> GLdouble -> GLdouble -> GLint -> Order
   -> Ptr a -> IO ()

instance Capability Map1Target where
   marshalCapability = marshalMap1Target

---------------------------------------------------------------------------

data Map2Target =
     Map2Vertex3
   | Map2Vertex4
   | Map2Index
   | Map2Color4
   | Map2Normal
   | Map2TextureCoord1
   | Map2TextureCoord2
   | Map2TextureCoord3
   | Map2TextureCoord4
   deriving (Eq,Ord)

marshalMap2Target :: Map2Target -> GLenum
marshalMap2Target Map2Vertex3       = gl_MAP2_VERTEX_3
marshalMap2Target Map2Vertex4       = gl_MAP2_VERTEX_4
marshalMap2Target Map2Index         = gl_MAP2_INDEX
marshalMap2Target Map2Color4        = gl_MAP2_COLOR_4
marshalMap2Target Map2Normal        = gl_MAP2_NORMAL
marshalMap2Target Map2TextureCoord1 = gl_MAP2_TEXTURE_COORD_1
marshalMap2Target Map2TextureCoord2 = gl_MAP2_TEXTURE_COORD_2
marshalMap2Target Map2TextureCoord3 = gl_MAP2_TEXTURE_COORD_3
marshalMap2Target Map2TextureCoord4 = gl_MAP2_TEXTURE_COORD_4

class Map2 a where
   map2 :: Map2Target
        -> (a, a) -> Stride -> Order
        -> (a, a) -> Stride -> Order
        -> Ptr b -> IO ()

instance Map2 GLfloat  where
   map2 t (u1, u2) us uo (v1, v2) = glMap2f (marshalMap2Target t) u1 u2 us uo v1 v2

instance Map2 GLdouble where
   map2 t (u1, u2) us uo (v1, v2) = glMap2d (marshalMap2Target t) u1 u2 us uo v1 v2

foreign import "glMap2f" unsafe glMap2f ::
      GLenum
   -> GLfloat  -> GLfloat  -> GLint -> Order
   -> GLfloat  -> GLfloat  -> GLint -> Order
   -> Ptr a -> IO ()

foreign import "glMap2d" unsafe glMap2d ::
      GLenum
   -> GLdouble -> GLdouble -> GLint -> Order
   -> GLdouble -> GLdouble -> GLint -> Order
   -> Ptr a -> IO ()

instance Capability Map2Target where
   marshalCapability = marshalMap2Target

---------------------------------------------------------------------------

class EvalCoord a where
   evalCoord :: a -> IO ()

instance EvalCoord (Vertex1 GLfloat ) where
   evalCoord (Vertex1 u) = glEvalCoord1f u

instance EvalCoord (Vertex1 GLdouble) where
   evalCoord (Vertex1 u) = glEvalCoord1d u

instance EvalCoord (Vertex2 GLfloat ) where
   evalCoord (Vertex2 u v) = glEvalCoord2f u v

instance EvalCoord (Vertex2 GLdouble) where
   evalCoord (Vertex2 u v) = glEvalCoord2d u v

foreign import "glEvalCoord1f" unsafe glEvalCoord1f :: GLfloat              -> IO ()
foreign import "glEvalCoord1d" unsafe glEvalCoord1d :: GLdouble             -> IO ()
foreign import "glEvalCoord2f" unsafe glEvalCoord2f :: GLfloat  -> GLfloat  -> IO ()
foreign import "glEvalCoord2d" unsafe glEvalCoord2d :: GLdouble -> GLdouble -> IO ()

---------------------------------------------------------------------------

class EvalCoordV a where
   evalCoordV :: Ptr a -> IO ()

instance EvalCoordV (Vertex1 GLfloat ) where
   evalCoordV = glEvalCoord1fv

instance EvalCoordV (Vertex1 GLdouble) where
   evalCoordV = glEvalCoord1dv

instance EvalCoordV (Vertex2 GLfloat ) where
   evalCoordV = glEvalCoord2fv

instance EvalCoordV (Vertex2 GLdouble) where
   evalCoordV = glEvalCoord2dv

foreign import "glEvalCoord1fv" unsafe glEvalCoord1fv :: Ptr (Vertex1 GLfloat ) -> IO ()
foreign import "glEvalCoord1dv" unsafe glEvalCoord1dv :: Ptr (Vertex1 GLdouble) -> IO ()
foreign import "glEvalCoord2fv" unsafe glEvalCoord2fv :: Ptr (Vertex2 GLfloat ) -> IO ()
foreign import "glEvalCoord2dv" unsafe glEvalCoord2dv :: Ptr (Vertex2 GLdouble) -> IO ()

---------------------------------------------------------------------------

data AutoNormal = AutoNormal deriving (Eq,Ord)

instance Capability AutoNormal where
   marshalCapability AutoNormal = gl_AUTO_NORMAL

---------------------------------------------------------------------------

class MapGrid1 a where
   mapGrid1 :: GLint -> a -> a -> IO ()

instance MapGrid1 GLfloat  where
   mapGrid1 = glMapGrid1f

instance MapGrid1 GLdouble where
   mapGrid1 = glMapGrid1d

foreign import "glMapGrid1f" unsafe glMapGrid1f :: GLint -> GLfloat  -> GLfloat  -> IO ()
foreign import "glMapGrid1d" unsafe glMapGrid1d :: GLint -> GLdouble -> GLdouble -> IO ()

---------------------------------------------------------------------------

class MapGrid2 a where
   mapGrid2 :: GLint -> a -> a
            -> GLint -> a -> a -> IO ()

instance MapGrid2 GLfloat  where
   mapGrid2 = glMapGrid2f

instance MapGrid2 GLdouble where
   mapGrid2 = glMapGrid2d

foreign import "glMapGrid2f" unsafe glMapGrid2f ::
      GLint -> GLfloat  -> GLfloat
   -> GLint -> GLfloat  -> GLfloat  -> IO ()

foreign import "glMapGrid2d" unsafe glMapGrid2d ::
      GLint -> GLdouble -> GLdouble
   -> GLint -> GLdouble -> GLdouble -> IO ()

---------------------------------------------------------------------------

evalMesh1 :: PolygonMode -> (GLint, GLint) -> IO ()
evalMesh1 mode (i1, i2) = glEvalMesh1 (marshalPolygonMode mode) i1 i2

foreign import "glEvalMesh1" unsafe glEvalMesh1 ::
   GLenum -> GLint -> GLint -> IO ()

evalMesh2 :: PolygonMode -> (GLint, GLint) -> (GLint, GLint) -> IO ()
evalMesh2 mode (i1, i2) (j1, j2) = glEvalMesh2 (marshalPolygonMode mode) i1 i2 j1 j2

foreign import "glEvalMesh2" unsafe glEvalMesh2 ::
   GLenum -> GLint -> GLint -> GLint -> GLint -> IO ()

foreign import "glEvalPoint1" unsafe evalPoint1 :: GLint          -> IO ()
foreign import "glEvalPoint2" unsafe evalPoint2 :: GLint -> GLint -> IO ()
