{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 5.4 (Display Lists) of the OpenGL 1.2.1
specs.
-}

module GL_DisplayList (
   DisplayList(..), DisplayListMode(..),
   unmarshalDisplayListMode,   -- internal use only
   newList, endList,           -- deprecated, use newEndList
   newEndList, callList, callLists, listBase, genLists, isList, deleteLists
) where

import Foreign          (Ptr)
import Monad            (liftM)

import GL_Constants     (gl_COMPILE, gl_COMPILE_AND_EXECUTE)
import GL_BasicTypes    (GLboolean_, GLboolean, unmarshalGLboolean,
                         GLuint, GLenum, GLsizei)
import GL_VertexArray   (Type, marshalType)

---------------------------------------------------------------------------

-- exported with contructor
newtype DisplayList = DisplayList GLuint

data DisplayListMode = Compile | CompileAndExecute
  deriving (Eq,Ord)

marshalDisplayListMode :: DisplayListMode -> GLenum
marshalDisplayListMode Compile           = gl_COMPILE
marshalDisplayListMode CompileAndExecute = gl_COMPILE_AND_EXECUTE

unmarshalDisplayListMode :: GLenum -> DisplayListMode
unmarshalDisplayListMode mode
   | mode == gl_COMPILE             = Compile
   | mode == gl_COMPILE_AND_EXECUTE = CompileAndExecute
   | otherwise                      = error "unmarshalDisplayListMode"

{-# DEPRECATED newList, endList "use newEndList instead" #-}
newList :: DisplayList -> DisplayListMode -> IO ()
newList (DisplayList dl) = glNewList dl . marshalDisplayListMode

foreign import "glNewList" unsafe glNewList :: GLuint -> GLenum -> IO ()
foreign import "glEndList" unsafe endList :: IO ()

newEndList :: DisplayList -> DisplayListMode -> IO a -> IO a
newEndList l m command = do
   newList l m
   val <- command
   endList
   return val

callList :: DisplayList -> IO ()
callList (DisplayList dl) = glCallList dl

foreign import "glCallList" unsafe glCallList :: GLuint -> IO ()

callLists :: GLsizei -> Type -> Ptr a -> IO ()
callLists n = glCallLists n . marshalType

foreign import "glCallLists" unsafe glCallLists :: GLsizei -> GLenum -> Ptr a -> IO ()

foreign import "glListBase" unsafe listBase :: GLuint -> IO ()

genLists :: GLsizei -> IO [DisplayList]
genLists n = do
   first <- glGenLists n
   return $ if first == 0
               then []
               else [ DisplayList l | l <- [ first .. first + fromIntegral n - 1 ] ]

foreign import "glGenLists" unsafe glGenLists :: GLsizei -> IO GLuint

isList :: DisplayList -> IO GLboolean
isList (DisplayList dl) = liftM unmarshalGLboolean $ glIsList dl

foreign import "glIsList" unsafe glIsList :: GLuint -> IO GLboolean_

deleteLists :: [DisplayList] -> IO ()
deleteLists lists = sequence_ [ glDeleteLists dl 1 | DisplayList dl <- lists ]

foreign import "glDeleteLists" unsafe glDeleteLists :: GLuint -> GLsizei -> IO ()
