{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

#include "config.h"

module GL_BasicTypes (
   GLboolean, GLboolean_,
   marshalGLboolean, unmarshalGLboolean, boolToGLint,   -- internal use only
   GLbyte, GLubyte, GLshort, GLushort, GLint, GLuint, GLsizei,
   GLenum, GLbitfield,
   toBitfield, fromBitfield,                            -- internal use only
   GLfloat, GLclampf, GLdouble, GLclampd,
   WindowPosition(..), WindowSize(..),
   windowPosition, windowSize,                          -- internal use only
   Viewport(..),
   Gettable(..),
   Capability(..), enable, disable, isEnabled,
   peek1, peek2, peek3, peek4, poke1, poke2, poke3, poke4
) where

import Int
import Word
import Bits             (Bits((.|.),(.&.)))
import Foreign          (Ptr, Storable(..), castPtr)

import GL_Constants     (gl_FALSE, gl_TRUE)

-- We don't use the underlying numerical representation of booleans on
-- the Haskell side, Bool is much nicer to use.
type GLboolean  = Bool
type GLboolean_ = HTYPE_GLBOOLEAN

marshalGLboolean :: GLboolean -> GLboolean_
marshalGLboolean False = gl_FALSE
marshalGLboolean True  = gl_TRUE

unmarshalGLboolean :: GLboolean_ -> GLboolean
unmarshalGLboolean = (gl_FALSE /=)

boolToGLint :: Bool -> GLint
boolToGLint False = 0
boolToGLint True  = 1

type GLbyte     = HTYPE_GLBYTE
type GLubyte    = HTYPE_GLUBYTE
type GLshort    = HTYPE_GLSHORT
type GLushort   = HTYPE_GLUSHORT
type GLint      = HTYPE_GLINT
type GLuint     = HTYPE_GLUINT
type GLsizei    = HTYPE_GLSIZEI
type GLenum     = HTYPE_GLENUM

type GLbitfield = HTYPE_GLBITFIELD

-- convenience functions for conversion between lists of flags and a bitfield

toBitfield :: (Bits b, Num b) => (a -> b) -> [a] -> b
toBitfield marshal = foldl (.|.) 0 . map marshal

fromBitfield :: (Enum a, Bounded a, Bits b, Num b) => (a -> b) -> b -> [a]
fromBitfield marshal bitfield =
   [ c | c <- [minBound .. maxBound],  (bitfield .&. marshal c) /= 0 ]


type GLfloat    = HTYPE_GLFLOAT
type GLclampf   = HTYPE_GLCLAMPF
type GLdouble   = HTYPE_GLDOUBLE
type GLclampd   = HTYPE_GLCLAMPD

---------------------------------------------------------------------------

data WindowPosition = WindowPosition GLint GLint deriving (Eq,Ord)

instance Storable WindowPosition where
   sizeOf    ~(WindowPosition x y) = sizeOf x + sizeOf y
   alignment ~(WindowPosition x _) = alignment x
   peek = peek2 WindowPosition
   poke ptr (WindowPosition x y) = poke2 ptr x y

windowPosition :: Integral a => a -> a -> WindowPosition
windowPosition x y = WindowPosition (fromIntegral x) (fromIntegral y)

---------------------------------------------------------------------------

data WindowSize = WindowSize GLsizei GLsizei deriving (Eq,Ord)

instance Storable WindowSize where
   sizeOf    ~(WindowSize w h) = sizeOf w + sizeOf h
   alignment ~(WindowSize w _) = alignment w
   peek = peek2 WindowSize
   poke ptr (WindowSize w h) = poke2 ptr w h

windowSize :: Integral a => a -> a -> WindowSize
windowSize w h = WindowSize (fromIntegral w) (fromIntegral h)

---------------------------------------------------------------------------

data Viewport = Viewport WindowPosition WindowSize deriving (Eq,Ord)

-- (Un-)Marshaling a viewport is a little bit of a hack: Functions expecting
-- a pointer to a viewport simply use GLint* for it. But we don't care about
-- this and (un-)marshal the parts the way they are. This means that GLint
-- and GLsizei should better have the same size and the highest bit remains
-- unused.   :-P
instance Storable Viewport where
   sizeOf    ~(Viewport p s) = sizeOf p + sizeOf s
   alignment ~(Viewport p _) = alignment p
   peek ptr                  = do p <- peek        (castPtr ptr)
                                  s <- peekByteOff (castPtr ptr) (sizeOf p)
                                  return $ Viewport p s
   poke ptr (Viewport p s)   = do poke        (castPtr ptr)            p
                                  pokeByteOff (castPtr ptr) (sizeOf p) s

---------------------------------------------------------------------------

class Gettable param val where
   get :: param -> IO val

---------------------------------------------------------------------------

class Capability a where
   marshalCapability :: a -> GLenum

enable :: Capability a => a -> IO ()
enable = glEnable . marshalCapability

foreign import "glEnable" unsafe glEnable :: GLenum -> IO ()

disable :: Capability a => a -> IO ()
disable = glDisable . marshalCapability

foreign import "glDisable" unsafe glDisable :: GLenum -> IO ()

isEnabled :: Capability a => a -> IO Bool
isEnabled = glIsEnabled . marshalCapability

foreign import "glIsEnabled" unsafe glIsEnabled :: GLenum -> IO GLboolean

---------------------------------------------------------------------------
-- Utilities (a little bit verbose/redundant, but seems to generate better
-- code than mapM/zipWithM_)

{-# INLINE peek1 #-}
peek1 :: Storable a => (a -> b) -> Ptr c -> IO b
peek1 f ptr = do
   x <- peekElemOff (castPtr ptr) 0
   return $ f x

{-# INLINE peek2 #-}
peek2 :: Storable a => (a -> a -> b) -> Ptr c -> IO b
peek2 f ptr = do
   x <- peekElemOff (castPtr ptr) 0
   y <- peekElemOff (castPtr ptr) 1
   return $ f x y

{-# INLINE peek3 #-}
peek3 :: Storable a => (a -> a -> a -> b) -> Ptr c -> IO b
peek3 f ptr = do
   x <- peekElemOff (castPtr ptr) 0
   y <- peekElemOff (castPtr ptr) 1
   z <- peekElemOff (castPtr ptr) 2
   return $ f x y z

{-# INLINE peek4 #-}
peek4 :: Storable a => (a -> a -> a -> a -> b) -> Ptr c -> IO b
peek4 f ptr = do
   x <- peekElemOff (castPtr ptr) 0
   y <- peekElemOff (castPtr ptr) 1
   z <- peekElemOff (castPtr ptr) 2
   w <- peekElemOff (castPtr ptr) 3
   return $ f x y z w

{-# INLINE poke1 #-}
poke1 :: Storable b => Ptr a -> b -> IO ()
poke1 ptr x =
   pokeElemOff (castPtr ptr) 0 x

{-# INLINE poke2 #-}
poke2 :: Storable b => Ptr a -> b -> b -> IO ()
poke2 ptr x y = do
   pokeElemOff (castPtr ptr) 0 x
   pokeElemOff (castPtr ptr) 1 y

{-# INLINE poke3 #-}
poke3 :: Storable b => Ptr a -> b -> b -> b -> IO ()
poke3 ptr x y z = do
   pokeElemOff (castPtr ptr) 0 x
   pokeElemOff (castPtr ptr) 1 y
   pokeElemOff (castPtr ptr) 2 z

{-# INLINE poke4 #-}
poke4 :: Storable b => Ptr a -> b -> b -> b -> b -> IO ()
poke4 ptr x y z w = do
   pokeElemOff (castPtr ptr) 0 x
   pokeElemOff (castPtr ptr) 1 y
   pokeElemOff (castPtr ptr) 2 z
   pokeElemOff (castPtr ptr) 3 w
