{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to chapter 7 (Callback Registration) of the
GLUT3 API docs.
-}

module GLUT_CallbackUtils (
   PerWindowCallback(..), wFunc,
   GlobalCallback(..),    gFunc
) where

import Foreign		(FunPtr, nullFunPtr, freeHaskellFunPtr, castFunPtr)
import IORef		(IORef, newIORef, readIORef, writeIORef)
import IOExts		(unsafePerformIO)
import Maybe		(isJust)

import GLUT_Window	(Window, getWindow)

---------------------------------------------------------------------------
-- To free adjustor thunks, we have to track registration/unregistration
-- in a global table, mapping the callback to its last registered address.

data PerWindowCallback =
     DisplayCB         | OverlayDisplayCB  | ReshapeCB
   | KeyboardCB        | KeyboardUpCB      | MouseCB
   | MotionCB          | PassiveMotionCB   | EntryExitCB
   | VisibilityCB      | WindowStatusCB    | SpecialCB
   | SpecialUpCB       | SpaceballMotionCB | SpaceballRotateCB
   | SpaceballButtonCB | ButtonBoxCB       | DialsCB
   | TabletMotionCB    | TabletButtonCB    | JoystickCB
   deriving Eq

-- No timer callback here, because there can be many of them! See timerFunc.
data GlobalCallback = MenuStatusCB | IdleCB
   deriving Eq

data Callback = PerWindowCB PerWindowCallback Window
              | GlobalCB GlobalCallback
   deriving Eq

-- A simple association lists suffices, because most callbacks won't change,
-- and the one which do (e.g. idle callback) will be moved to the front of
-- the list automagically. NOTE: The () return type is a hack!
type CallbackTable = [(Callback, FunPtr ())]

-- The following pragma is necessary to make this hack/technique work reliably.
{-# notInline theCallbackTable #-}
theCallbackTable :: IORef CallbackTable
theCallbackTable = unsafePerformIO (newIORef [])

-- (un-)register a per-window callback
wFunc :: PerWindowCallback -> (a -> IO (FunPtr b)) -> (FunPtr b -> IO ()) -> Maybe a -> IO ()
wFunc cb makeCallback registerCallbackAtGLUT maybeAction = do
   currentWindow <- getWindow
   wgFunc (PerWindowCB cb currentWindow) makeCallback registerCallbackAtGLUT maybeAction

-- (un-)register a global callback
gFunc :: GlobalCallback -> (a -> IO (FunPtr b)) -> (FunPtr b -> IO ()) -> Maybe a -> IO ()
gFunc = wgFunc . GlobalCB

-- generic callback (un-)registration
wgFunc :: Callback -> (a -> IO (FunPtr b)) -> (FunPtr b -> IO ()) -> Maybe a -> IO ()
wgFunc cb makeCallback registerCallbackAtGLUT maybeAction = do
   -- create the callback if we are registering
   newCBPtr  <- maybe (return nullFunPtr) makeCallback maybeAction
   -- if there was already a registered callback for this event,
   -- free the adjustor thunk and remove it from the global table.
   oldCBTable <- readIORef theCallbackTable
   newCBTable <- maybe (return oldCBTable)
                       (\oldCBPtr -> do freeHaskellFunPtr (castFunPtr oldCBPtr)
                                        return [ ca | ca@(cb2,_) <- oldCBTable, cb2 /= cb ])
                       (lookup cb oldCBTable)
   -- update the global table
   writeIORef theCallbackTable
              (if isJust maybeAction
                  then (cb, castFunPtr newCBPtr) : newCBTable
                  else newCBTable)
   -- (un-)register event at GLUT
   registerCallbackAtGLUT newCBPtr
