<p align="center">
<a href="https://travis-ci.org/dgasmith/gau2grid">
  <img src="https://travis-ci.org/dgasmith/gau2grid.svg?branch=master" alt="Travis CI"/>
</a>

<a href="https://ci.appveyor.com/project/MolSSI/gau2grid">
  <img src="https://ci.appveyor.com/api/projects/status/d3l5nid8a2dww8dc?svg=true" alt="Appveyor"/>
</a>

<a href="https://codecov.io/gh/dgasmith/gau2grid">
  <img src="https://codecov.io/gh/dgasmith/gau2grid/branch/master/graph/badge.svg" alt="Codecov" />
</a>

<a href="https://anaconda.org/psi4/gau2grid">
  <img src="https://anaconda.org/psi4/gau2grid/badges/version.svg" />
</a>

<a href='https://gau2grid.readthedocs.io/en/latest/?badge=latest'>
    <img src='https://readthedocs.org/projects/gau2grid/badge/?version=latest' alt='Documentation Status' />
</a>
</p>

# gau2grid
A collocation code for computing gaussians on a grid of the form:
```
out_Lp = x^l y^m z^n \sum_i coeff_i e^(exponent_i * (|center - p|)^2)
```
Where the returned matrix dimension are the angular momentum (L) by number of requested points (p).

```python
import gau2grid
import numpy as np

# Build coordinates along the Z axis
>>> xyz = np.zeros((3, 5))
>>> xyz[2] = np.arange(5)

# Compute a 's' gaussian with a scaling and exponent of one at the origin
>>> ret = gau2grid.collocation(xyz, 0, [1], [1], [0, 0, 0])
>>> print(ret["PHI"])
[[  1.00000e+00   3.67879e-01   1.83156e-02   1.23409e-04   1.12535e-07]]

# Compute a 'p' gaussian with a scaling and exponent of one at the origin
>>> ret = gau2grid.collocation(xyz, 1, [1], [1], [0, 0, 0], spherical=False)
>>> print(ret["PHI"])
[[  0.00000e+00   0.00000e+00   0.00000e+00   0.00000e+00   0.00000e+00]
 [  0.00000e+00   0.00000e+00   0.00000e+00   0.00000e+00   0.00000e+00]
 [  0.00000e+00   3.67879e-01   3.66312e-02   3.70229e-04   4.50140e-07]]

# Note that the X and Y components are zero as they are orthogonal to our Z vector.
```

The returned matrix can be in either cartesian or regular solid harmonics. There are currently
three algorithms in which to compute these collocation matrices:
 - Optimize C: A autogenerated C library that optimizes for cache,
    vectorization, and sparsity. Fastest, requires compilation, found at
    `gau2grid.collocation`.
- Optimized/Generated NumPy: A exploratory tool to
    examine the sparsity in the gaussians. No compilation required, found at
    `gau2grid.np_gen.collocation`.
- NumPy Reference: A simple NumPy-based loop
    code. No compilation required, found at `gau2grid.ref.collocation`.

See the [documentation](https://gau2grid.readthedocs.io/en/latest/?badge=latest) for more information!

## Building Gau2Grid
The C library is built with CMake and has C no required dependancies other than
the standard library. A CMake and build example can found below:

```bash
cmake -H. -Bobjdir
cd objdir; make -j2
```

Several common CMake options are as follow:
 - `-DPYTHON_EXECUTABLE` - Path to the desired Python executable
 - `-DMAX_AM` - Maximum angular momentum to compile to, default 6
 - `-DCMAKE_INSTALL_PREFIX` - Installation directory
 - `-DCARTESIAN_ORDER` - The cartesian ordering of the basis functions (row)
 - `-DSPHERICAL_ORDER` - The spherical ordering of the basis functions (cca, gaussian)

## Python installation
The gau2grid program (without the optimized C library) can be installed using
the canonical `setup.py` script,
```
python setup.py install
```

# Authors
This code was inspired by a number of folks and quite a few provided excellent advice.

 - Daniel G. A. Smith - Code author
 - Rob M. Parrish - Author of the Psi4 section which contains the original equations
 - Lori A. Burns - CMake, building, and library linking
 - Andy C. Simmonett - RSH coefficients
 - Ben Pritchard - Generator and vectorization recommendations

