# -*- coding: utf-8 -*-
#
# The gameclock game engines
#
# This is where the game engines reside.
#
# (C) Anarcat 2011

import math
import gameclock.clock

class Game:
    """the game engine
    
    this regroups two clocks and handles turn switches

    to simplify the code logic, this is basically the "Blitz" mode
    """

    # the number of turns played
    turns = 0
    miliseconds = False
    mode = 'blitz'
    
    default_time = 2 * 60 * 1000 # in ms
    default_players = 2 # default number of players
    clock_type = gameclock.clock.ChessClock # the default game type. note that this cannot be FisherGame because of the way the UI is setup.

    # the clocks in the game
    #
    # in chess there are two clocks, but there can be more. this is simply a list
    cur_clock = None
    first_clock = None

    def __init__(self, clock_type, clocks, default_time, miliseconds, delay = None):
        self.miliseconds = miliseconds
        self.clock_type = clock_type
        self.default_time = default_time

        # the clock engines
        self.cur_clock = self.first_clock = p = self.clock_type(self, delay)
        for i in range(clocks-1):
            p.next = self.clock_type(self, delay)
            p = p.next

    def start(self):
        """start the game
        
        this basically starts the clock
        """
        self.cur_clock.start()
        self.cur_clock.update()

    def end_turn(self):
        """end the turn
        
        this is the handler for the main button. it will stop the
        active clock and start the other and switch the active clock
        """
        self.cur_clock.stop()
        # XXX: we might lose a few ms here
        self.cur_clock = self.next_clock()
        self.cur_clock.start()
        self.turns += 1

    def next_clock(self):
        return self.cur_clock.next or self.first_clock

    def pause(self):
        """pause the game
        
        this just pauses the current clock
        """
        self.cur_clock.pause()

    def switch_clock(self):
        """change the current clock"""
        self.cur_clock = self.next_clock()

    def get_turns(self):
        """the number of complete turns played
        
        a 'complete turn' is a turn played by each player

        this is 1-based, i.e. when the game starts, it's turn 1, after
        every player has played once, it's turn 2, etc.
        """
        return math.floor(self.turns/self.count_players()) + 1

    def count_players(self):
        p = self.first_clock
        count = 0
        while p:
            count += 1
            p = p.next
        return count

    def dead(self):
        p = self.first_clock
        while p:
            if p.dead:
                return True
            p = p.next
        return False

    def running(self):
        return self.cur_clock.running()

    def __str__(self):
        """make a better string representation of the objects
        
        we basically dump all variables and some functions
        """
        return "  game engine %s\n  turns: %f / %d miliseconds: %d\n  first %s\n  current %s" % ( object.__str__(self), self.turns, self.get_turns(), self.miliseconds, self.first_clock, self.cur_clock)
