;; cmail-feedmail.el --- Send mail with feedmail

;; Copyright (C) 1992-2000 Yukihiro Matsumoto

;; Author: Nakagawa Makoto <makoto.nakagawa@jp.compaq.com>
;; Keywords: mail
;; Created at: Sun May 28 18:00:00 JST 2000
;; $Date: 2002/07/20 23:09:58 $
;; $Revision: 1.2 $

;; This file is part of cmail (a mail utility for GNU Emacs)

;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or (at
;; your option) any later version.

;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;;; Code:
(eval-when-compile (require 'static)
		   (require 'advice)
		   (require 'cl) ; --> flet, dolist
		   (require 'cmail-vars))

;;
;; custom $B4XO"(B
;;
(defun cmail-feedmail-set-enable-queue (symbol value)
  ;; $B$3$N;~E@$G$O(B cmail-feedmail-enable-queue $B$ODj5A$5$l$F$$$J$$2DG=@-$"$j!#(B
  (unless (boundp 'cmail-feedmail-enable-queue)
    (setq cmail-feedmail-enable-queue (not value)))
  (unless (equal cmail-feedmail-enable-queue value)
    (cmail-feedmail-toggle-queueing)))

(cmail-i18n-defcustom cmail-feedmail-enable-queue nil
  ((ja_JP . "\
*feedmail $B$N(B queue $B$rMxMQ$9$k$+$I$&$+!#(B")
   (en_US . "\
*Use queue function of feedmail."))
  :set 'cmail-feedmail-set-enable-queue
  :type 'boolean
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)
(cmail-custom-add-init 'cmail-feedmail-set-enable-queue
		       'cmail-feedmail-enable-queue)

(cmail-i18n-defcustom cmail-mail-insert-date t
  ((ja_JP . "\
*date $B%X%C%@$r@8@.$9$k$+$I$&$+!#(B")
   (en_US . "\
*Insert date header."))
  :type 'boolean
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)

(cmail-i18n-defcustom cmail-feedmail-date-generator t
  ((ja_JP . "\
*date $B%X%C%@@8@.4o!#(Bt $B$J$i$P(B feedmail $B$N(B default $B$N$b$N$r;HMQ$9$k!#(B")
   (en_US . "\
*Date header generator."))
  :type '(choice (const t) function)
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)

(cmail-i18n-defcustom cmail-mail-insert-message-id t
  ((ja_JP . "\
*message-id $B%X%C%@$r@8@.$9$k$+$I$&$+!#(B")
   (en_US . "\
*Insert message-id header."))
  :type 'boolean
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)

(cmail-i18n-defcustom cmail-feedmail-message-id-generator t
  ((ja_JP . "\
*message-id $B%X%C%@@8@.4o!#(B")
   (en_US . "\
*Message-id header generator."))
  :type '(choice (const t) function)
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)

(cmail-i18n-defcustom cmail-feedmail-queue-path "~/Queue/"
  ((ja_JP . "\
*feedmail $BMQ$N%-%e!<$rJ];}$9$k%G%#%l%/%H%j!#(B $B:G8e$OI,$:%9%i%C%7%e(B/$B$G=*$o$k(B.")
   (en_US . "\
*Directory to store queued mail for feedmail. String must end with '/' (slash)."))
  :type '(directory)
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)

(cmail-i18n-defcustom cmail-feedmail-queue-queue-path "q"
  ((ja_JP . "\
*feedmail $BMQ$N%-%e!<!#Aw?.$NBP>]$H$J$k$b$N!#(B")
   (en_US . "\
*feedmail queue directory for outgoing mails."))
  :type '(directory)
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)

(cmail-i18n-defcustom cmail-feedmail-queue-draft-path "draft"
  ((ja_JP . "\
*feedmail $BMQ$N%-%e!<!#%I%i%U%H$N%a!<%k$,N/$^$k!#(B")
   (en_US . "\
*feedmail queue directory for draft mails."))
  :type '(directory)
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)

(cmail-i18n-defcustom cmail-feedmail-queue-fqm-suffix ".cmail-fqm"
  ((ja_JP . "\
*feedmail $B$N(B cmail $BMQ$N%-%e!<%U%!%$%k$N3HD%;R!#(B")
   (en_US . "\
*suffix for queued file created by cmail over feedmail."))
  :type '(string)
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)

(cmail-i18n-defcustom cmail-feedmail-slug-limit-lenght 21
  ((ja_JP . "\
*$B%-%e!<%U%!%$%kL>$N:GBgD9!#(B")
   (en_US . "\
*limit against the lenght of a queue file name."))
  :type 'number
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)

(cmail-i18n-defcustom cmail-feedmail-dates-when-queue nil
  ((ja_JP . "\
*$BAwIUMQ$X%-%e!<%$%s%0$7$?;~E@$G(B Date $B%U%#!<%k%I$rIU$1$k!#(B")
   (en_US . "\
*insert date field when queued to queue, not draft."))
  :type 'boolean
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)

(cmail-i18n-defcustom cmail-feedmail-mail-callback-when-queue nil
  ((ja_JP . "\
*cmail-mail-callback $B$,(B t $B$G$b!"AwIUMQ$X%-%e!<%$%s%0$7$?;~E@$G(B
Reply Mark $B$rIU$1$k!#(B")
   (en_US . "\
*set reply mark when queued, even if cmail-mail-callback is t."))
  :type 'boolean
  :group 'cmail-all-variables
  :group 'cmail-use-feedmail-group)

(defvar cmail-feedmail-queue-directory
  (concat cmail-feedmail-queue-path
	  cmail-feedmail-queue-queue-path))
(defvar cmail-feedmail-queue-draft-directory
  (concat cmail-feedmail-queue-path
	  cmail-feedmail-queue-draft-path))

;;
;; message resource
;;
(dolist (resource
	 `((feedmail-before-queue-prompt
	    ("Japanese" . "q)$B%-%e!<!"(Bd)$B%I%i%U%H!"(Bi)$BAw?.!'(B")
	    (t . "q)ueue, d)raft or send i)mmediately: "))
	   (feedmail-after-send-prompt
	    ("Japanese" . "$B%I%i%U%H%U%!%$%k(B %s $B$r>C$7$^$9$+!#(B")
	    (t . "Delete draft file: %s? "))))
  (cmail-add-resource resource))

;;
;; $BK\BN(B
;;
(defun cmail-toggle-use-feedmail ()
  (interactive)
  (if cmail-use-feedmail
      (cmail-nouse-feedmail)
    (cmail-use-feedmail)))

(defun cmail-use-feedmail ()
  (unless cmail-use-feedmail
    (require 'feedmail)
    (if (boundp 'send-mail-function)
	(setq si:cmail-send-mail-function send-mail-function))
    (add-hook 'cmail-mail-hook 'cmail-feedmail-localizer)
    
    (ad-enable-advice 'cmail-send 'around 'cmail-feedmail-send)
    (ad-activate 'cmail-send)
    
    (setq cmail-use-feedmail t)
    (run-hooks 'cmail-use-feedmail-hook)))

(defun cmail-nouse-feedmail ()
  (when cmail-use-feedmail
    (setq send-mail-function si:cmail-send-mail-function)
    (remove-hook 'cmail-mail-hook 'cmail-feedmail-localizer)

    (ad-disable-advice 'cmail-send 'around 'cmail-feedmail-send)
    (ad-activate 'cmail-send)
    
    (setq cmail-use-feedmail nil)
    (run-hooks 'cmail-nouse-feedmail-hook)))

(defun cmail-feedmail-toggle-queueing ()
  (interactive)
  (if cmail-feedmail-enable-queue
      (setq cmail-feedmail-enable-queue nil)
    (setq cmail-feedmail-enable-queue t)))

(defadvice cmail-send (around cmail-feedmail-send)
  (if (or (and feedmail-enable-queue feedmail-queue-runner-is-active)
	  (not feedmail-enable-queue))
      (let ((is-fqm (and buffer-file-name (feedmail-fqm-p buffer-file-name)))
	    (previous-buffer-file-name buffer-file-name)
	    ;; to avoid confirmation in mail-send when in draft queue
	    (buffer-file-name nil))
	ad-do-it
	(if (and is-fqm
		 (y-or-n-p
		  (cmail-format-resource1 'feedmail-after-send-prompt
			  (file-name-nondirectory previous-buffer-file-name))))
	    (delete-file previous-buffer-file-name)))
    (let ((feedmail-prompt-before-queue-user-alist
	   '((?i . cmail-feedmail-message-action-send)
	     (?I . cmail-feedmail-message-action-send-strong)))
	  (feedmail-ask-before-queue-prompt
	   (cmail-format-resource 'feedmail-before-queue-prompt)))
      (flet ((feedmail-tidy-up-slug (slug) (cmail-feedmail-tidy-up-slug slug)))
	(feedmail-send-it)))))

(defun cmail-feedmail-message-action-draft ()
  (interactive)
  (flet ((feedmail-tidy-up-slug (slug) (cmail-feedmail-tidy-up-slug slug)))
    (feedmail-message-action-draft)))

(defun cmail-feedmail-message-action-draft-strong ()
  (interactive)
  (let ((feedmail-queue-slug-maker nil)
	(feedmail-ask-for-queue-slug t))
    (flet ((feedmail-tidy-up-slug (slug) (cmail-feedmail-tidy-up-slug slug)))
      (feedmail-message-action-draft-strong))))

(defun cmail-feedmail-message-action-queue ()
  "Call \`feedmail-message-action-queue\'.  But insert date field when
\`cmail-feedmail-dates-when-queue\' is t.  Also execute mail-send-actions
when \`cmail-feedmail-mail-callback-when-queue\' is t."
  (interactive)
  (if cmail-feedmail-dates-when-queue
      (feedmail-fiddle-date nil))
  (let ((*mail-send-actions mail-send-actions)
	(cmail-feedmail-localizer-hook cmail-feedmail-localizer-hook))
    (flet ((feedmail-tidy-up-slug (slug) (cmail-feedmail-tidy-up-slug slug)))
      (if (and cmail-mail-callback cmail-feedmail-mail-callback-when-queue)
	  (remove-hook 'cmail-feedmail-localizer-hook
		       'cmail-feedmail-set-mail-send-actions-dumper))
      (feedmail-message-action-queue))
    (if (and cmail-mail-callback cmail-feedmail-mail-callback-when-queue)
	(setq mail-send-actions nil) ; this will do no harm even if feedmail-nuke-buffer-after-queue
	(while *mail-send-actions
	  (condition-case nil
	      (apply (car (car *mail-send-actions))
		     (cdr (car *mail-send-actions)))
	    (error))
	  (setq *mail-send-actions (cdr mail-send-actions))))))

(add-hook
 'cmail-derived-mail-mode-setter
 (function
  (lambda ()
    (define-key cmail-mail-mode-map "\C-x\C-s"
      'cmail-feedmail-message-action-draft)
    (define-key cmail-mail-mode-map "\C-x\C-w"
      'cmail-feedmail-message-action-draft-strong)
    (define-key cmail-mail-mode-map "\C-c\C-a"
      'cmail-feedmail-message-action-queue))))
(if running-xemacs
    (add-hook
     'cmail-mail-mode
     (function
      (lambda ()
	(add-menu-button '("Mail")
			 ["Dump to Draft" cmail-feedmail-message-action-draft t]
			 "Go to Cmail Summary Buffer")
	(add-menu-button '("Mail")
			 ["Dump to Draft as" cmail-feedmail-message-action-draft-strong t]
			 "Go to Cmail Summary Buffer")
	(add-menu-button '("Mail")
			 ["Dump to Queue" cmail-feedmail-message-action-queue t]
			 "Go to Cmail Summary Buffer")))
     'append)
  (add-hook
   'cmail-derived-mail-mode-setter
   (function
    (lambda ()
      ;; cmail-mail-mode is derived from mail-mode
      (define-key cmail-mail-menu-bar [queue]
	'("Dump to Queue" . cmail-feedmail-message-action-queue))
      (define-key cmail-mail-menu-bar [draft]
	'("Dump to Draft as" . cmail-feedmail-message-action-draft-strong))
      (define-key cmail-mail-menu-bar [draft]
	'("Dump to Draft" . cmail-feedmail-message-action-draft))))))

(setq auto-mode-alist
      (append
       `((,(concat "\\" cmail-feedmail-queue-fqm-suffix "$") .
	  (lambda ()
	    ;; $B<!$NF|$K$O(B cmail-nouse-feedmail $B$G(B
	    ;; $B$"$k>l9g$r9MN8$9$k!#(B
	    (unless (memq 'cmail-feedmail-localizer cmail-mail-hook)
	      (add-hook 'cmail-mail-hook 'cmail-feedmail-localizer nil t))
	    (funcall cmail-feedmail-queue-runner-mode-setter))))
       auto-mode-alist))

(defvar cmail-feedmail-queue-runner-mode-setter
  (function
   (lambda (&optional arg)
;	 (when (or (eq 'no-conversion last-coding-system-used)
;		   (string-match "\\`raw-text"
;				 (symbol-name last-coding-system-used)))
     ;; feedmail uses insert-file-contents-literally
     ;; $B$3$N$d$jJ}$G$$$$$N!)(B
     ;; insert-file-literally $B$H(B find-file $B$G(B undecided $B$N(B 2$B%1!<%9(B?
;	 (debug)
     (let ((coding-system
	    (car (find-coding-systems-region (point-min) (point-max)))))
       (princ (format "find-coding-system is %s \n" coding-system)
	      (get-buffer-create "*test*"))
       (when (or (eq coding-system 'undecided)
		 (string-match "\\`\\(raw-\\|x-c\\)text\\'"
			       (symbol-name coding-system)))
	 (princ (format "detect-coding-system is %s" coding-system)
		(get-buffer-create "*test*"))
	 (let ((coding-system
		(detect-coding-region (point-min) (point-max) t)))
	   (set-buffer-multibyte t)
	   (if (string-match "raw-text" (symbol-name coding-system))
	       ;; might not be detected
	       (setq coding-system default-buffer-file-coding-system))
	   (decode-coding-region (point-min) (point-max) coding-system)))
       (cmail-mail-mode)
       (run-hooks 'cmail-mail-hook)))))

(defun cmail-feedmail-message-action-send ()
  (let ((feedmail-enable-queue nil))
    (cmail-send)))
(defun cmail-feedmail-message-action-send-strong ()
  (let ((feedmail-enable-queue nil)
	(feedmail-confirm-outgoing nil))
    (cmail-send)))

(defun cmail-feedmail-localizer ()
  (dolist (variable
	   '(send-mail-function 
	     feedmail-enable-queue
	     feedmail-queue-directory
	     feedmail-queue-draft-directory
	     feedmail-queue-runner-message-sender
	     feedmail-queue-runner-mode-setter
	     feedmail-queue-fqm-suffix))
    (make-local-variable variable))
  (setq send-mail-function 'feedmail-send-it
	feedmail-enable-queue cmail-feedmail-enable-queue
	feedmail-queue-directory cmail-feedmail-queue-directory
	feedmail-queue-draft-directory cmail-feedmail-queue-draft-directory
	feedmail-queue-runner-message-sender 'cmail-send-and-exit
	feedmail-queue-runner-mode-setter cmail-feedmail-queue-runner-mode-setter
	feedmail-queue-fqm-suffix cmail-feedmail-queue-fqm-suffix)

  ;; message-id$B!"(Bdate $BN>%U%#!<%k%I$N@8@.$rB.$/$9$k!#(B
  (make-local-hook 'mail-send-hook)
  (add-hook 'mail-send-hook
	    (function
	     (lambda ()
	       (setq feedmail-date-generator
		     (if cmail-mail-insert-date
			 (if (or (not cmail-feedmail-date-generator)
				 (eq t cmail-feedmail-date-generator))
			     (` ("ignored" 
				 (, (feedmail-default-date-generator
				     feedmail-queue-runner-is-active))
				 create))) ; or relpace?
		       cmail-feedmail-date-generator))
	       (setq feedmail-message-id-generator
		     (if cmail-mail-insert-message-id
			 (if (or (not cmail-feedmail-message-id-generator)
				 (eq t cmail-feedmail-message-id-generator))
			     (` ("ignored"
				 (, (feedmail-default-message-id-generator
				     feedmail-queue-runner-is-active))
				 create))
			   cmail-feedmail-message-id-generator)))
	       (setq feedmail-x-mailer-line
		     (if cmail-mail-insert-x-mailer
			 (` ("ignored" 
			     (nil
			      "%s "
			      (, (concat "(via feedmail "
					 feedmail-patch-level
					 (if feedmail-queue-runner-is-active
					     " Q" " I")
					 (if feedmail-enable-spray "S" "")
					 ")")))
			     combine))
		       "")))) ; nil does not work for version 8
	    nil t) ; make it local
  (run-hooks 'cmail-feedmail-localizer-hook))

;;
;; mail-send-actions $B$N1JB32=(B
;;
(add-hook 'cmail-feedmail-localizer-hook
	  'cmail-feedmail-set-mail-send-actions-dumper)

(setq file-name-handler-alist
      (append file-name-handler-alist
	      (`(((,(concat "\\" cmail-feedmail-queue-fqm-suffix "$")) .
		 cmail-feedmail-file-name-handler)))))

(defun cmail-feedmail-set-mail-send-actions-dumper ()
  (make-local-variable 'write-region-annotate-functions)
  (setq write-region-annotate-functions
	(append write-region-annotate-functions
		'(cmail-feedmail-mail-send-actions-dumper))))

(defun cmail-feedmail-file-name-handler (operation &rest args)
  (let ((after-insert-file-functions
	 (append after-insert-file-functions
		 '(cmail-feedmail-mail-send-actions-loader)))
	(inhibit-file-name-handlers
	 (cons 'cmail-feedmail-file-name-handler
	       (and (eq inhibit-file-name-operation operation)
		    inhibit-file-name-handlers)))
	(inhibit-file-name-operation operation))
    (apply operation args)))

(defconst cmail-feedmail-mail-send-action-header "x-cmail-mail-send-action")
(defun cmail-feedmail-mail-send-actions-dumper (start end)
  (mapcar
   (lambda (action)
     (cons 1 (concat cmail-feedmail-mail-send-action-header ": "
		     (prin1-to-string action) "\n")))
   (reverse mail-send-actions)))

(defun cmail-feedmail-mail-send-actions-loader (bytes)
  (save-restriction
    (narrow-to-region (point) (+ (point) bytes))
    (when (cmail-get-field-values-in-mail-buffer cmail-feedmail-mail-send-action-header)
      (let (start)
	(re-search-forward
	 (concat "^" cmail-feedmail-mail-send-action-header) nil 'move)
	(goto-char (match-beginning 0))
	(setq start (point))
	(make-local-variable 'mail-send-actions)
	(while (looking-at
		(concat cmail-feedmail-mail-send-action-header ": "))
	  (goto-char (match-end 0))
	  (add-to-list 'mail-send-actions (read (current-buffer)))
	  (forward-line))
	(setq bytes (- bytes (- (point) start)))
	(delete-region (point) start)
	(goto-char (point-min))
	bytes))))

;; feedmail $B$N%G%#%U%)%k%H$O(B feedmail-buffer-to-binmail
(when (boundp 'send-mail-function)
  (let ((send-mail-function (symbol-name send-mail-function)))
    (string-match "\\`\\(.*\\)-send-it\\'" send-mail-function)
    (cond ((string= "sendmail" (match-string 1 send-mail-function))
	   (setq feedmail-buffer-eating-function 'feedmail-buffer-to-sendmail))
	  ((string= "smtpmail" (match-string 1 send-mail-function))
	   (setq feedmail-buffer-eating-function 'feedmail-buffer-to-smtpmail))
	  (t
	   ;; do nothing, leave it to default.
	   ))))

;; flim/smtpmail $B$r;H$&>l9g(B
(static-unless (and (require 'smtpmail nil t)
		    (fboundp 'smtpmail-via-smtp))
  (defun smtpmail-via-smtp (a p)
    "Function defined in mail/smtp which corresponds to smtp-via-smtp in flim/smtpmail."
    ;; feedmail-send-it-immediately $B$,(B error buffer $B$rI=<($7$F$/$l$k$O$:!#(B
    ;; feedmail-send-it-immediately
    ;; -> feedmail-give-it-to-buffer-eater
    ;;    -> funcall feedmail-bffer-eating-function
    ;;       -> feedmail-buffer-to-smtpmail
    ;;          -> smtpmail-via-smtp
    ;;
    ;; Older flim/smtp does not handle SMTP error properly,
    ;; or use smtp-send-buffer directory against newer flim
    (eq t (smtp-via-smtp user-mail-address a p))))

;; buffer-eating-function $B$K$h$j!"(Baddress $B$K$h$C$F(B smtp server $B$N@Z$jBX(B
;; $B$($k$3$H$,2DG=!)(B

;; user-mail-address $B$O(B p(repped) $B$+$i<hF@$9$k$N$,(B better?$B!#(B
;; mail/smtpmail $B$O(B user-mail-address $B$rJQ99$9$k!#(B

;; smtp-server $B$K$O4X?t$r;XDj$G$-$k!#(B(flim $B$N>l9g(B) mail/smtpmail $B$O(B 
;; smtpmail-smtp-server $B$rJQ99$9$k!#(B

(defun cmail-feedmail-queue-reminder (&optional what-event)
  (interactive "p")
  (let ((feedmail-queue-fqm-suffix cmail-feedmail-queue-fqm-suffix)
	(feedmail-queue-directory cmail-feedmail-queue-directory)
	(feedmail-queue-draft-directory cmail-feedmail-queue-draft-directory))
    (feedmail-queue-reminder what-event)))

(defun cmail-feedmail-run-the-queue (&optional arg)
  (interactive "p")
  (let ((feedmail-queue-fqm-suffix cmail-feedmail-queue-fqm-suffix)
	(feedmail-queue-directory cmail-feedmail-queue-directory)
	(feedmail-queue-draft-directory cmail-feedmail-queue-draft-directory)
	(feedmail-queue-runner-mode-setter
	 cmail-feedmail-queue-runner-mode-setter))
    (feedmail-run-the-queue arg)))

(eval-when-compile (require 'emu))
(defalias-maybe 'truncate-string-to-width 'truncate-string)
(defun cmail-feedmail-tidy-up-slug (slug)
  (require 'filename) ; -> replace-as-filename
  (let ((filename-limit-length cmail-feedmail-slug-limit-lenght)
	(filename-filters
	 (append filename-filters
		 (list
		  (function
		   (lambda (str)
		     (truncate-string-to-width str filename-limit-length)))))))
    (replace-as-filename slug)))

;;
;; Feedmail hooks for the draft list mode (experimental)
;;
(add-hook 'cmail-draft-list-mode-hook
	  'cmail-draft-list-mode-feedmail-setup)
(defun cmail-draft-list-mode-feedmail-setup ()
  (define-key cmail-draft-list-mode-map "r"
    'cmail-draft-feedmail-run-the-queue))

(eval-after-load "cmail-reply"
  '(setq cmail-get-draft-message-listing-func
	 (append cmail-get-draft-message-listing-func
		 '(cmail-draft-list-feedmail-draft
		   cmail-draft-list-feedmail))))

(defun cmail-draft-list-feedmail (msgnum)
  (if (file-directory-p cmail-feedmail-queue-directory)
      (cmail-draft-list-feedmail-dir
       cmail-feedmail-queue-directory
       "Feedmail Queue Files" msgnum)))

(defun cmail-draft-list-feedmail-draft (msgnum)
  (if (file-directory-p cmail-feedmail-queue-draft-directory)
      (cmail-draft-list-feedmail-dir
       cmail-feedmail-queue-draft-directory
       "Feedmail Draft Files" msgnum)))

(defun cmail-draft-list-feedmail-dir (dir label msgnum)
  (save-excursion
    (let ((fl (directory-files
	       dir nil
	       (concat "^[0-9][0-9][0-9]-.*"
		       (regexp-quote cmail-feedmail-queue-fqm-suffix)
		       "$")
	       t))
	  res1 res buf queue-file)
      (while fl
	(setq queue-file (car fl))
	(if queue-file
	    (progn
	      (setq buf (get-file-buffer (expand-file-name queue-file dir)))
	      (if buf (set-buffer buf)
		(find-file (expand-file-name queue-file dir)))
	      (save-excursion
		(goto-char (point-min))
		(setq res1
		      (append
		       res1
		       (list (list
			      msgnum
			      (or (cmail-get-field-values "To" ", ") "")
			      (or (cmail-get-field-values "Subject") "")
			      'cmail-draft-list-feedmail-commands
			      buf
			      (or (buffer-file-name) "")))))
		(setq msgnum (1+ msgnum))
		(if (null buf) (kill-buffer (current-buffer))))))
	(setq fl (cdr fl)))
      (setq res (cons label res1))
      (cons msgnum res))))

(defconst cmail-draft-list-feedmail-commands
  '((open . cmail-draft-list-feedmail-open)
    (delete . cmail-draft-list-feedmail-delete)))

(defun cmail-draft-list-feedmail-commands (operation message)
  (let ((func (cdr (assoc operation
			  cmail-draft-list-feedmail-commands))))
    (if func (apply func (list message)))))

(defun cmail-draft-list-feedmail-open (message)
  (if (nth 4 message) (switch-to-buffer (nth 4 message))
    (let ((draft-buffer (find-file-noselect (nth 5 message))))
      (cmail-draft-list-mode)
      (beginning-of-line)
      (bury-buffer)
      (switch-to-buffer draft-buffer))))

(defun cmail-draft-list-feedmail-delete (message)
  (if (nth 4 message) (kill-buffer (nth 4 message)))
  (if (nth 5 message) (delete-file (nth 5 message)))
  (cmail-draft-list-mode)
  (beginning-of-line))

(defun cmail-draft-feedmail-run-the-queue ()
  (interactive)
  (cmail-feedmail-run-the-queue)
  (cmail-draft-list-mode))

;;; @ End.
;;;

;;; cmail-feedmail.el ends here
