package Language::INTERCAL::Runtime::QuantumLibrary;

# Library functions for CLC-INTERCAL

# This file is part of CLC-INTERCAL.

# Copyright (C) 1999 Claudio Calvelli <lunatic@assurdo.com>, all rights reserved

# WARNING - do not operate heavy machinery while using CLC-INTERCAL

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use vars qw($VERSION @EXPORT @ISA);
$VERSION = '0.05';

require Exporter;
use Language::INTERCAL::Runtime::Library;

@EXPORT = qw(_qexit _qassign _qread _qwrite _qsubscript _qstash _qretrieve
	     _qowner _qfree _qimport _qlearns _qfinish _qvalue _qenter _qvfork);
@ISA = qw(Exporter);

sub _qimport ($$$$) {
    my ($regs, $shregs, $regname, $create) = @_;
    return if exists $regs->{$regname};
    my $rp = $regname;
    if ($regname =~ /^[\.:]/) {
	$rp =~ s/^\./,/;
	$rp =~ s/^:/;/;
	$rp =~ s/\s.*$//;
    }
    if (! exists $shregs->{$regname}) {
	return if ! $create;
	$regs->{$regname} = [0, 1, [], 0, 0, [], []];
	if (exists $regs->{$rp} && $regs->{$rp}[3]) {
	    # register is private to this thread, make a new copy for the shared
	    $shregs->{$regname} = [0, 1, [], 1, 0, [], []];
	    $regs->{$regname}[3] = 1;
	} else {
	    # register is shared, so make a pointer to it
	    $shregs->{$regname} = $regs->{$regname};
	}
	return;
    }
    if (exists $regs->{$rp} && $regs->{$rp}[3]) {
	# Element of a private array, and must have been created after priv
	$regs->{$regname} = [0, 1, [], 1, 0, [], []];
    } else {
	$regs->{$regname} = $shregs->{$regname};
    }
}

sub _qassign ($$$@) {
    my ($regs, $shregs, $regname, @data) = @_;
    _qimport($regs, $shregs, $regname, 1);
    _qimport($regs, $shregs, '@0', 1);
    _assign($regs, $regname, @data);
}

sub _qsubscript ($$$@) {
    my ($regs, $shregs, $regname, @data) = @_;
    _qimport($regs, $shregs, $regname, 0);
    _subscript($regs, $regname, @data);
}

sub _qread ($$$$$$@) {
    my ($regs, $shregs, $output, $roman, $io, $arrayio, @regname) = @_;
    for my $regname (@regname) {
	_qimport($regs, $shregs, $regname, 0);
    }
    _read($regs, $output, $roman, $io, $arrayio, @regname);
}

sub _qwrite ($$$$$@) {
    my ($regs, $shregs, $input, $io, $arrayio, @regname) = @_;
    for my $regname (@regname) {
	_qimport($regs, $shregs, $regname, 1);
    }
    _write($regs, $input, $io, $arrayio, @regname);
}

sub _qvalue ($$$$) {
    my ($regs, $shregs, $regname, $int) = @_;
    _qimport($regs, $shregs, $regname, 0);
    _qimport($regs, $shregs, '@0', 1);
    my $v = _value($regs, $regname, $int);
    $v;
}

sub _qstash ($$$) {
    my ($regs, $shregs, $regname) = @_;
    _qimport($regs, $shregs, $regname, 0);
    _stash($regs, $regname);
}

sub _qretrieve ($$$) {
    my ($regs, $shregs, $regname) = @_;
    _qimport($regs, $shregs, $regname, 0);
    _retrieve($regs, $regname);
}

sub _qowner ($$$@) {
    my ($regs, $shregs, $regname, @path) = @_;
    while (@path) {
	_qimport($regs, $shregs, $regname, 0);
	my $p = shift @path;
	die "512 $regname IS A FREE REGISTER\n"
	    if ! exists $regs->{$regname} || ! @{$regs->{$regname}[2]};
	die "512 $regname DOES NOT HAVE THAT MANY OWNERS\n"
	    if $p > @{$regs->{$regname}[2]};
	$regname = $regs->{$regname}[2][$p - 1];
    }
    $regname;
}

sub _qfree ($$$$) {
    my ($regs, $shregs, $slave, $master) = @_;
    _qimport($regs, $shregs, $slave, 0);
    _free($regs, $slave, $master);
}

sub _qlearns ($$$$$$$) {
    my ($regs, $shregs, $regname, $lectures, $subject, $stack, $return) = @_;
    _qimport($regs, $shregs, $regname, 0);
    die "822 $regname IS NOT A STUDENT\n"
	if ! exists $regs->{$regname} || ! @{$regs->{$regname}[6]};
    for my $class (@{$regs->{$regname}[6]}) {
	next if ! exists $lectures->{$class}{$subject};
	_qimport($regs, $shregs, $class, 1);
	if (! $regs->{$class}[3]) {
	    $regs->{$class} = _clone_array($regs->{$class});
	    $regs->{$class}[3] = 1;
	}
	my $belong = _clone_array($regs->{$class}[2]);
	unshift @{$regs->{$class}[2]}, $regname;
	push @$stack, [$return, $class, $belong];
	goto $lectures->{$class}{$subject};
    }
    die "823 #$subject NOT IN $regname\'S CURRICULUM\n";
}

sub _qfinish ($$$) {
    my ($regs, $shregs, $stack) = @_;
    die "801 NOT IN A LECTURE\n" if ! @$stack;
    my $v = pop @$stack;
    my ($return, $class, $belong) = @$v;
    _qimport($regs, $shregs, $class, 1);
    if (exists $regs->{$class}) {
	$regs->{$class}[2] = $belong;
    } else {
	$regs->{$class} = [0, 1, $belong, 0, 0, [], []];
    }
    goto $return;
}

sub _qexit ($$$$@) {
    my ($thread, $program_counter, $endprog, $splats, @rest) = @_;
    if (@$program_counter < 2) {
	die join("\n", @$splats, '') if @$splats;
	goto $endprog;
    }
    splice (@$program_counter, $thread, 1);
    $_[0] = 0 if $thread >= @$program_counter;
    while (@rest) {
	my $a = shift @rest;
	splice (@$a, $thread, 1);
    }
    goto $program_counter->[$_[0]];
}

sub _qenter ($$$$$$$@) {
    my ($thread, $regs, $reinstate, $program_counter, $next,
	$loop_bc, $loop_cb, @rest) = @_;
    my $new = @$program_counter;
    $program_counter->[$new] = $next;
    $regs->[$new] = {};
    for my $rp (keys %{$regs->[$thread]}) {
	$regs->[$new]{$rp} =
	    $regs->[$thread]{$rp}[3] ? _clone_value($regs->[$thread]{$rp})
				     : $regs->[$thread]{$rp};
    }
    for my $r ($reinstate, $loop_bc, $loop_cb) {
	$r->[$new] = [];
	for (my $stmt = 0; $stmt < @{$r->[$thread]}; $stmt++) {
	    $r->[$new][$stmt] =
		$r->[$thread][$stmt][1]
		    ? _clone_value($r->[$thread][$stmt])
		    : $r->[$thread][$stmt];
	}
    }
    while (@rest) {
	my $a = shift @rest;
	$a->[$new] = _clone_value($a->[$thread]);
    }
    $new;
}

sub _qvfork ($$$$$) {
    my ($regs, $shregs, $rp, $thread, $new_thread) = @_;
    _qimport($regs->[$thread], $shregs, $rp, 1);
    $regs->[$thread]{$rp} = _clone_array($regs->[$thread]{$rp});
    $regs->[$thread]{$rp}[3] = 1;
    $regs->[$new_thread]{$rp} = _clone_array($regs->[$thread]{$rp});
    $regs->[$thread]{$rp}[1] = 1;
    $regs->[$new_thread]{$rp}[1] = 0;
    # if this is a whole array, fork each value as well
    if ($rp =~ /^[,;]/) {
	$rp =~ s/^,/./;
	$rp =~ s/^;/:/;
	$rp .= ' ';
	for my $rr (keys %{$regs->[$thread]}) {
	    if (substr($rr, 0, length($rp)) eq $rp) {
		$regs->[$thread]{$rr} = _clone_array($regs->[$thread]{$rr});
		$regs->[$thread]{$rr}[3] = 1;
		$regs->[$new_thread]{$rr} = _clone_array($regs->[$thread]{$rr});
	    }
	}
    }
}

1;

__END__

=head1 NAME

Language::INTERCAL::Runtime::QuantumLibrary - Runtime library for CLC-INTERCAL

=head1 SYNOPSIS

    use Language::INTERCAL::Runtime::QuantumLibrary;
    use Language::INTERCAL::Runtime::Library;

    sub program {
        ...
    }

    program();

=head1 DESCRIPTION

I<Language::INTERCAL::Runtime::QuantumLibrary> provides the runtime library
and Quantum Emulator for CLC-INTERCAL's default back end (I<Perl>), as well
as the other perl back end (I<PerlText>); it is used when a quantum program
needs to run on non-quantum hardware. You should never need to access this
package directly, as the compiler does that automatically.

=head1 COPYRIGHT

This module is part of CLC-INTERCAL.

Copyright (c) 1999 by Claudio Calvelli E<lt>C<lunatic@assurdo.com>E<gt>,
all (f)rights reserved.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

=head1 SEE ALSO

L<Language::INTERCAL::Runtime::Library>, and a qualified psychiatrist.

