package Language::INTERCAL;

# Generic compiler functionality for CLC-INTERCAL

# This file is part of CLC-INTERCAL.

# Copyright (C) 1999 Claudio Calvelli <lunatic@assurdo.com>, all rights reserved

# WARNING - do not operate heavy machinery while using CLC-INTERCAL

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use FileHandle;
use Data::Dumper;
use File::Spec;
use Language::INTERCAL::Runtime::Library;

use vars qw($VERSION);
BEGIN {
    $VERSION = '0.05';
}

use vars qw(*fudge *fiddle);

*fudge = *fiddle = \&toggle;

my %toggle = (
    'debug' => 0,
    'mingle' => 1,
    'xor' => 1,
    'next' => 0,
    'roman' => 0,
    'width' => 79,
    'io' => 172,
    'bug' => 5,
    'ubug' => 0.005,
    'charset' => '',
    'parser' => 'CLCintercal',
    'arrayio' => 'CLC',
);

sub toggle {
    my $f;
    for $f (@_) {
	if ($f =~ /^(width|io|bug|ubug)\s*(?:=\s*)?((?:\d*\.)?\d+)/i) {
	    $toggle{$1} = $2;
	} elsif ($f =~ /^(charset|parser)\s*(?:=\s*)?/i) {
	    $toggle{$1} = $';
	} elsif ($f =~ /^\s*(?:=\s*)?([01])/ && exists $toggle{$`}) {
	    $toggle{$`} = $1;
	} elsif (exists $toggle{$f}) {
	    $toggle{$f} = ! $toggle{$f};
	} elsif ($toggle{'debug'}) {
	    print STDERR "Unknown toggle: '$f'\n";
	}
    }
}

sub import {
    goto \&compile if @_ > 1;
}

sub compile {
    my ($package, $filename, $line) = caller;
    my $syntax = 
	'013 SYNTAX IS "use|compile Language::INTERCAL <program>, <source> " '
      . "[, <list_filehandle>], ['optimise'] at $filename line $line\n";
    @_ < 3 and die $syntax;
    my $ignore = shift @_;
    my $prog_name = shift @_;
    my $source = '' . (shift @_);
    my @list_file = ();
    my $optimise = 0;
    my $quantum = 0;
    my $post = 0;
    my $dbhook = 0;
    while (@_) {
	my $i = shift @_;
	if (ref $i && (ref $i eq 'CODE' || UNIVERSAL::isa($i, 'GLOB'))) {
	    @list_file = ($i);
	} elsif ($i =~ /^opt(?:imi[sz]e)?$/i) {
	    $optimise = 1;
	} elsif ($i =~ /^nobug$/i) {
	    $toggle{'bug'} = 0;
	} elsif ($i =~ /^noubug$/i) {
	    $toggle{'ubug'} = 0;
	} elsif ($i =~ /^quantum$/i) {
	    $quantum = 1;
	} elsif ($i =~ /^post(?:process)?$/i) {
	    $post = 1;
	} elsif ($i =~ /^dbhook$/i) {
	    $dbhook = 1;
	} else {
	    die $syntax;
	}
    }
    my $ptree = parse Language::INTERCAL($source,
					 "(inline:$filename\:$line)",
					 @list_file);
    if ($quantum) { $ptree->{'flags'}{'quantum'} = 1 }
    if ($post) { $ptree->{'flags'}{'postprocess'} = 1 }
    if ($dbhook) { $ptree->{'flags'}{'dbhook'} = 1 }
    if ($optimise) {
	require Language::INTERCAL::Optimiser;
	Language::INTERCAL::Optimiser::first($ptree, $filename, $line);
	$ptree->{'flags'}{'optimise'} = 1;
	Language::INTERCAL::Optimiser::second($ptree, $filename, $line);
    }
    require Language::INTERCAL::Backend::Perl;
    Language::INTERCAL::Backend::Perl::backend(
			'Perl', $ptree, $package, $line, $prog_name);
}

sub name {
    my ($package, $filename, $line) = caller;
    @_ == 2 or die
	"013 SYNTAX IS \"name PARSE_TREE <name>\" at $filename line $line\n";
    my $self = shift;
    $self->{'name'} = shift;
    $self;
}

sub parse {
    my ($package, $filename, $line) = caller;
    my $syntax = '013 SYNTAX IS "parse Language::INTERCAL <program>, <file> " '
	       . "[, <list_filehandle>] at $filename line $line\n";
    @_ < 2 and die $syntax;
    my $class = shift @_;
    my $source = shift @_;
    my $srcfile = @_ ? shift @_ : "($filename\:$line)";
    my $list_file = undef;
    if (@_) {
	$list_file = shift @_;
    	@_ and die $syntax;
	ref($list_file) or die $syntax;
	ref($list_file) eq 'CODE' or
	ref($list_file) eq 'ARRAY' or
	    UNIVERSAL::isa($list_file, 'GLOB') or die $syntax;
    }
    my $chr = $toggle{'charset'};
    if ($chr eq '') {
	$chr = guess_charset($source);
	die "111 CANNOT GUESS CHARACTER SET\n" if $chr eq '';
    }
    $source = &{ convert_charset($chr, 'ASCII') }($source);
    my $p = 'Language::INTERCAL::Parser::' . $toggle{'parser'};
    eval "require $p";
    die $@ if $@;
    my $ptree;
    eval '$ptree = ' . $p . '::parse($source, $list_file, $srcfile, \\%toggle)';
    die $@ if $@;
    bless $ptree, $class;
}

sub backend {
    my ($package, $filename, $line) = caller;
    @_ >= 2 or die
	'013 SYNTAX IS "backend PARSE_TREE <type>, <arguments>" '
      . " at $filename line $line\n";
    my ($self, $type, @args) = @_;
    $@ = '';
    eval "require Language::INTERCAL::Backend::$type"
	if ! defined &{ "Language::INTERCAL::Backend::$type\::backend" };
    die "013 CANNOT FIND '$type' BACK END" . ($toggle{'debug'}?" ($@)":'') . "\n"
	if $@;
    if (exists $self->{'flags'}{'optimise'}) {
	require Language::INTERCAL::Optimiser;
	Language::INTERCAL::Optimiser::second($self, $filename, $line);
    }
    &{ "Language::INTERCAL::Backend::$type\::backend" }
		    ($type, $self, $package, $line, @args);
}

sub complete_name {
    my ($package, $filename, $line) = caller;
    @_ == 3 or die
	'013 SYNTAX IS "complete_name PARSE_TREE <backend>, <name>" '
      . " at $filename line $line\n";
    my ($self, $type, $name) = @_;
    $@ = '';
    eval "require Language::INTERCAL::Backend::$type"
	if ! defined &{ "Language::INTERCAL::Backend::$type\::backend" };
    die "013 CANNOT FIND '$type' BACK END" . ($toggle{'debug'}?" ($@)":'') . "\n"
	if $@;
    return $type
	if ! defined &{ "Language::INTERCAL::Backend::$type\::suffix" };
    $name .= &{ "Language::INTERCAL::Backend::$type\::suffix"}();
    $name;
}

sub optimise {
    my ($package, $filename, $line) = caller;
    @_ == 1 or die
	"013 SYNTAX IS \"optimise PARSE_TREE\" at $filename line $line\n";
    my $ptree = shift @_;
    require Language::INTERCAL::Optimiser;
    Language::INTERCAL::Optimiser::first($ptree, $filename, $line);
    $ptree->{'flags'}{'optimise'} = 1;
    $ptree;
}

sub link {
    my ($package, $filename, $line) = caller;
    my $syntax = "013 SYNTAX IS \"link PARSE_TREE [, PARSE_TREE]...\" "
	       . "at $filename (line $line)\n";
    my $makenew = 0;
    $makenew = shift if @_ && (! ref $_[0]) && $_[0] eq 'new';
    die $syntax if @_ == 0;

    # first, combine the flags together
    my %flags = ();
    my @unoptimised = ();
    my $ptree;
    for $ptree (@_) {
	die $syntax if ! ref $ptree
		    || ! UNIVERSAL::isa($ptree, 'Language::INTERCAL');
	my $f;
	my $o = 0;
	for $f (keys %{$ptree->{'flags'}}) {
	    if ($ptree->{'flags'}{$f}) {
		$flags{$f} = 1;
		$o = 1 if $f eq 'optimise';
	    }
	}
	push @unoptimised, $ptree if ! $o;
    }

    # if some parse tree are optimised, must optimise all the other ones...
    if (exists $flags{'optimise'}) {
	for $ptree (@unoptimised) {
	    print STDERR "link: optimising $ptree->{'name'}\n"
		if $ptree->{'toggle'}{'debug'};
	    optimise $ptree;
	}
    }

    # build the rest of the combined tree
    my @files = ();
    my %labels = ();
    my @join = qw(come_froms abstain reinstate fork
		  nexts gabstain greinstate gfork);
    my %join = map {($_, [])} @join;
    for $ptree (@_) {
	my $offset = @files;
	push @files, @{$ptree->{'files'}};
	my $label;
	for $label (keys %{$ptree->{'labels'}}) {
	    my $ldata = $ptree->{'labels'}{$label};
	    die "458 YOU LIKE LABEL ($ldata->[3]) A LOT "
	      . "($files[$ldata->[0]][0]:$ldata->[4] AND "
	      . "$files[$labels{$label}[0]][0]:$labels{$label}[4])\n"
		if exists $labels{$label};
	    $labels{$label} =
		[$ldata->[0], $ldata->[1] + $offset,
		 $ldata->[2], $ldata->[3], $ldata->[4], {}];
	}
	my $join;
	for $join (@join) {
	    my $j;
	    for $j (@{$ptree->{$join}}) {
		push @{$join{$join}},
		    [$j->[0], $j->[1] + $offset, $j->[2], $j->[3], $j->[4]];
	    }
	}
    }
    my $ptr = $_[0];
    if ($makenew) {
	my %a = %$ptr;
	$ptr = \%a;
    }
    $ptr->{'flags'} = \%flags;
    $ptr->{'files'} = \@files;
    $ptr->{'labels'} = \%labels;
    my $join;
    for $join (@join) {
	$ptr->{$join} = $join{$join};
    }

    # that's all
    $ptr;
}

sub save {
    my ($package, $filename, $line) = caller;
    @_ == 2 or die
	"013 SYNTAX IS \"save PARSE_TREE, NAME\" at $filename (line $line)\n";
    my ($self, $name) = @_;
    my $dumper = Data::Dumper->new([$self], ['IPT']);
    $dumper->Indent(0);
    $dumper->Purity(0);
    $dumper->Terse(1);
    $dumper->Deepcopy(1);
    my $dump = $dumper->Dump;
    my $out = FileHandle->new("> $name")
    	or die "012 $name: $! at $filename (line $line)\n";
    print $out "CLC-INTERCAL $VERSION IPT\n$dump\n";
    close $out;
    $self;
}

sub load {
    my ($package, $filename, $line) = caller;
    @_ == 2 or die
	"013 SYNTAX IS \"load Language::INTERCAL NAME\" at $filename (line $line)\n";
    my ($class, $name) = @_;
    my $close = 0;
    my $in = $name;
    if (ref $in && ! UNIVERSAL::isa($in, 'GLOB')) {
	$name = '';
    } else {
	$in = FileHandle->new("< $name")
	    or die "110 $name: $! at $filename (line $line)\n";
	$close = 1;
	$name = " IN $name";
    }
    my $magic = <$in>;
    die "666 PROBLEM WITH MAGIC$name at $filename (line $line)\n"
	unless $magic =~ /^CLC-INTERCAL (\d+\.\d+) IPT/ && $1 <= $VERSION;
    my $objver = $1;
    my $text = '';
    while (<$in>) { $text .= $_ }
    close $in if $close;
    $@ = '';
    my $res = eval $text;
    die "666 PROBLEM WITH COMPILER OBJECT$name at $filename (line $line)\n" if $@;
    if ($VERSION != $objver) {
	# might need to convert the opcodes
	# however, this is not required for now
    }
    $res;
}

sub iterate {
    (@_ >= 2 && @_ <= 6) or do {
	my ($package, $filename, $line) = caller;
	die "013 SYNTAX IS \"COMPILER_OBJECT->iterate(CODE [, BINIT [, BEXIT]] "
	  . "[, FINIT[, FEXIT]])\" AT $filename LINE $line\n";
    };
    my $ptree = shift @_;
    my $func = shift @_;
    my $binit = @_ ? shift @_ : sub {};
    my $bexit = @_ ? shift @_ : sub {};
    my $finit = @_ ? shift @_ : sub {};
    my $fexit = @_ ? shift @_ : sub {};
    my $fid;
    my $files = $ptree->{'files'};
    for ($fid = 0; $fid < @$files; $fid++) {
	my $file = $files->[$fid];
	&$finit($ptree, $fid, $file);
	my $bid;
	for ($bid = 1; $bid < @$file; $bid++) {
	    my $block = $file->[$bid];
	    &$binit($ptree, $fid, $bid, $block);
	    my $sid;
	    for ($sid = 0; $sid < @$block; $sid++) {
		&$func($ptree, $fid, $bid, $sid, $block->[$sid]);
	    }
	    &$bexit($ptree, $fid, $bid, $block);
	}
	&$fexit($ptree, $fid, $file);
    }
    $ptree;
}

sub unimport {
    my ($package, $filename, $line) = caller;
    die "444 $filename (line $line) attempted to remove INTERCAL!\n";
}

sub list_charsets {
    _list_modules(qw(Charset));
}

sub list_backends {
    _list_modules(qw(Language INTERCAL Backend));
}

sub list_parsers {
    _list_modules(qw(Language INTERCAL Parser));
}

sub _list_modules {
    my @modpath = @_;
    my %result = ();
    my $path;
    for $path (@INC) {
	my $dir = File::Spec->catdir($path, @modpath);
	opendir(DIR, $dir);
	while (defined (my $ent = readdir DIR)) {
	    my $name = $ent;
	    next unless $name =~ s/\.pm$//i;
	    next unless -f File::Spec->catfile($dir, $ent);
	    $result{$name} = 1;
	}
	closedir DIR;
    }
    sort keys %result;
}

sub guess_charset {
    my ($package, $filename, $line) = caller;
    @_ == 1 or die "013 SYNTAX IS guess_charset(TEXT) AT $filename LINE $line\n";
    my ($text) = @_;
    return 'ASCII' if $text =~ /PLEASE|DO|\(\d+\)/i;
    return 'EBCDIC' if $text =~ /[ח][ӓ][Ņ][][][Ņ]
				|[Ą][֖]
				|M[]+\]/x;
    $text =~ tr[\200-\377][\000-\177]; # remove bit 7
    $text =~ tr[\000-\077][\100-\177]; # remove bit 6
    $text =~ tr[\140-\177][\100-\137]; # remove bit 5
    return 'Baudot' if $text =~ /V[_\[]*S[_\[]*A[_\[]*C[_\[]*E[_\[]*A
				|I[_\[]*X
				|O[_\[]*[VWSAJPUGFX]+[_\[]*R/x;
    # maybe they have installed extra character sets?
    my $charset;
    for $charset (list_charsets()) {
	next if $charset eq 'EBCDIC';
	next if $charset eq 'Baudot';
	$@ = '';
	eval "require Charset::$charset";
	next if $@;
	my $a = '';
	eval "\$a = \L${charset}\E2ascii(\$text)";
	next if $@;
	return $charset if $text =~ /DO|PLEASE|\(\d+\)/i;
    }
    # nope, can't figure out
    '';
}

1;
