;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: wco -*-
;;; $Id: regression.lisp,v 1.5 2002/03/29 19:59:38 craig Exp $
;;;
;;; Copyright (c) 2000, 2001 onShore Development, Inc.
;;;
;;; Regression testing for WebCheckout
;;;
;;; Tests are keyed by class, and bugid. To run a test for a single bug:
;;;
;;; (regress [class] [bugid])
;;;
;;; To run all tests for a given class:
;;;
;;; (regress [class])
;;;
;;; To run all tests:
;;;
;;; (regress)
;;;
;;; Tests ought to be iterable, ought not to affect eachother when run
;;; serially, and generally assume that the database has been freshly
;;; :cold-boot'ed.

(in-package :odcl)

(defparameter *regression-funcs* (make-hash-table :test #'equal)
  "mapping of class + bugid to test function")


(defmacro defregression ((class &rest testids) &body body)
  "define a function FUNCTION-NAME with body BODY that tests all bugs
in BUGID-LIST"
  (let ((doc (when (and (listp body)
			(stringp (car body)))
               (car body))))
    `(let ((test-fun (lambda () ,@body)))
      (dolist (id ',testids)
        (setf (gethash `(,,class ,id) *regression-funcs*) (cons ,doc test-fun))))))

(defvar *regression-stats* nil)

(defun regress (&optional class bugid)
  "entry point for regression testing, called with no parameters to
run all tests"
  (cond (bugid
         (if-bind (function (gethash (list class bugid) *regression-funcs*))
             (funcall (cdr function))
             (cmsg "test ~s ~s not found" class bugid)))
        (class
         (let* ((all-keys (hashkeys *regression-funcs*))
                (class-keys (remove-if-not (lambda (k) (eq (car k) class)) all-keys)))
           (if (null class-keys)
               (progn
                 (cmsg "No tests found for class ~s" class)
                 (cmsg "Valid classes are: ~{~s~^, ~}" (remove-duplicates (mapcar #'car all-keys))))
               (progn
                 (cmsg "~%;; Running ~d tests for class ~s~%;;" (length class-keys) class)
                 (setq class-keys (sort class-keys (lambda (x y)
                                                     (let ((x (cadr x))
                                                           (y (cadr y)))
                                                       (cond ((numberp x)
                                                              (or (not (numberp y))
                                                                  (< x y)))
                                                             ((numberp y)
                                                              (not (or (not (numberp x))
                                                                       (< x y))))
                                                             (t
                                                              (string<= (string x) (string y))))))))
                 (dolist (test-key class-keys)
                   (%regress-with-handler test-key))))))
        (t
         (cmsg "Starting regression run")
         (let ((*regression-stats* (list (cons :running t)
                                         (cons :test-count 0)
                                         (cons :failure-count 0))))
           (mapc #'regress
                 (remove-duplicates (mapcar #'car (hashkeys *regression-funcs*))))
           (cmsg "Regression run complete, ~d tests run, ~d failures"
                 (get-alist :test-count *regression-stats*)
                 (get-alist :failure-count *regression-stats*))
           (if (< 0 (get-alist :failure-count *regression-stats*))
               (values nil *regression-stats*)
               (values t *regression-stats*))))))


(defun %regress-with-handler (bugid)
  (when *regression-stats*
    (let ((count (get-alist :test-count *regression-stats*)))
      (update-alist :test-count (1+ count) *regression-stats*)))
  (let ((function (gethash bugid *regression-funcs*))
        (time (get-utime)))
    (format t "~&;; ~a ~6a -" (car bugid) (cadr bugid))
    (force-output t)
    (handler-case
        (let ((*console-spam* nil))
          (funcall (cdr function))
          (format t "              "))
      (error ()
        (when *regression-stats*
          (let ((failures (get-alist :failure-count *regression-stats*)))
            (update-alist :failure-count (1+ failures) *regression-stats*)))
        (format t " ***FAILURE***")))
    (format t " ~d - ~a~%"
            (float (/ (ceiling (/ (- (get-utime) time) 100) 100) 100))
            (or (car function) ""))))
