/*--------------------------------------------------------------------*/
/*--- Callgrind                                                    ---*/
/*---                                                    ct_bbcc.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Callgrind, a Valgrind tool for call tracing.

   Copyright (C) 2002-2004, Josef Weidendorfer (Josef.Weidendorfer@gmx.de)

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "global.h"
#include "costs.h"

/*------------------------------------------------------------*/
/*--- BBCC operations                                      ---*/
/*------------------------------------------------------------*/

#define N_BBCC_INITIAL_ENTRIES  10437

/* BBCC table (key is BB/Context), per thread, resizable */
bbcc_hash current_bbccs;

void SK_(init_bbcc_hash)(bbcc_hash* bbccs)
{
   Int i;

   CT_ASSERT(bbccs != 0);

   bbccs->size    = N_BBCC_INITIAL_ENTRIES;
   bbccs->entries = 0;
   bbccs->table = (BBCC**) VG_(malloc)(bbccs->size * sizeof(BBCC*));

   for (i = 0; i < bbccs->size; i++) bbccs->table[i] = NULL;
}

void SK_(copy_current_bbcc_hash)(bbcc_hash* dst)
{
  CT_ASSERT(dst != 0);

  dst->size    = current_bbccs.size;
  dst->entries = current_bbccs.entries;
  dst->table   = current_bbccs.table;
}

bbcc_hash* SK_(get_current_bbcc_hash)()
{
  return &current_bbccs;
}

void SK_(set_current_bbcc_hash)(bbcc_hash* h)
{
  CT_ASSERT(h != 0);

  current_bbccs.size    = h->size;
  current_bbccs.entries = h->entries;
  current_bbccs.table   = h->table;
}

/*
 * Zero all costs of a BBCC
 */
void SK_(zero_bbcc)(BBCC* bbcc)
{
  Int i;
  jCC* jcc;

  CT_ASSERT(bbcc->cxt != 0);
  CT_DEBUG(1, "  zero_bbcc: BB 0x%x, Cxt %d "
	   "(fn '%s', rec %d)\n", 
	   bb_addr(bbcc->bb),
	   bbcc->cxt->base_number + bbcc->rec_index,
	   bbcc->cxt->fn[0]->name,
	   bbcc->rec_index);

  if ((bbcc->exe_counter ==0) &&
      (bbcc->ret_counter ==0)) return;

  for(i=0;i<bbcc->bb->cost_count;i++)
    bbcc->cost[i] = 0;
   
  for(jcc=bbcc->jcc_list; jcc; jcc=jcc->next_from)
    SK_(init_cost)( SK_(sets).full, jcc->cost );

  bbcc->exe_counter = 0;
  bbcc->ret_counter = 0;
}



void SK_(forall_bbccs)(void (*func)(BBCC*))
{
  BBCC *bbcc, *bbcc2;
  int i, j;
	
  for (i = 0; i < current_bbccs.size; i++) {
    if ((bbcc=current_bbccs.table[i]) == NULL) continue;
    while (bbcc) {
      /* every bbcc should have a rec_array */
      CT_ASSERT(bbcc->rec_array != 0);

      for(j=0;j<bbcc->cxt->fn[0]->separate_recursions;j++) {
	if ((bbcc2 = bbcc->rec_array[j]) == 0) continue;

	(*func)(bbcc2);
      }
      bbcc = bbcc->next;
    }
  }
}


/* All BBCCs for recursion level 0 are inserted into a
 * thread specific hash table with key
 * - address of BB structure (unique, as never freed)
 * - current context (includes caller chain)
 * BBCCs for other recursion levels are in bbcc->rec_array.
 *
 * The hash is used in setup_bb(), i.e. to find the cost
 * counters to be changed in the execution of a BB.
 */

static __inline__
UInt hash_bbcc(BB* bb, Context* cxt, UInt size)
{
   CT_ASSERT(bb != 0);
   CT_ASSERT(cxt != 0);

   return ((Addr)bb + (Addr)cxt) % size;
}
 

/* Lookup for a BBCC in hash.
 */ 
static
BBCC* lookup_bbcc(BB* bb, Context* cxt)
{
   BBCC* bbcc = bb->last_bbcc;
   UInt  hash;

   /* check LRU */
   if (bbcc->cxt == cxt) {
       if (!SK_(clo).separate_threads) {
	   /* if we don't dump threads separate, tid doesn't have to match */
	   return bbcc;
       }
       if (bbcc->tid == SK_(current_tid)) return bbcc;
   }

   SK_(stat).bbcc_lru_misses++;

   hash = hash_bbcc(bb, cxt, current_bbccs.size);
   bbcc = current_bbccs.table[hash];
   while (bbcc &&
	  (bb      != bbcc->bb ||
	   cxt     != bbcc->cxt)) {
       bbcc = bbcc->next;
   }
   
   CT_DEBUG(2,"  lookup_bbcc(BB 0x%x, Cxt %d, fn '%s'): 0x%p (tid %d)\n",
	    bb_addr(bb), cxt->base_number, cxt->fn[0]->name, 
	    bbcc, bbcc ? bbcc->tid : 0);

   CT_DEBUGIF(2)
     if (bbcc) SK_(print_bbcc)(-2,bbcc,False);

   return bbcc;
}


/* double size of hash table 1 (addr->BBCC) */
static void resize_bbcc_hash()
{
    Int i, new_size, conflicts1 = 0, conflicts2 = 0;
    BBCC** new_table;
    UInt new_hash;
    BBCC *curr_BBCC, *next_BBCC;

    new_size = 2*current_bbccs.size+3;
    new_table = (BBCC**) VG_(malloc)(new_size * sizeof(BBCC*));
 
    if (!new_table) return;
 
    for (i = 0; i < new_size; i++)
      new_table[i] = NULL;
 
    for (i = 0; i < current_bbccs.size; i++) {
	if (current_bbccs.table[i] == NULL) continue;
 
	curr_BBCC = current_bbccs.table[i];
	while (NULL != curr_BBCC) {
	    next_BBCC = curr_BBCC->next;

	    new_hash = hash_bbcc(curr_BBCC->bb,
				 curr_BBCC->cxt,
				 new_size);

	    curr_BBCC->next = new_table[new_hash];
	    new_table[new_hash] = curr_BBCC;
	    if (curr_BBCC->next) {
		conflicts1++;
		if (curr_BBCC->next->next)
		    conflicts2++;
	    }

	    curr_BBCC = next_BBCC;
	}
    }

    VG_(free)(current_bbccs.table);


    CT_DEBUG(0,"Resize BBCC Hash: %d => %d (entries %d, conflicts %d/%d)\n",
	     current_bbccs.size, new_size,
	     current_bbccs.entries, conflicts1, conflicts2);

    current_bbccs.size = new_size;
    current_bbccs.table = new_table;
    SK_(stat).bbcc_hash_resizes++;
}


static __inline
BBCC** new_recursion(int size)
{
    BBCC** bbccs;
    int i;
    
    bbccs = (BBCC**) VG_(malloc)(sizeof(BBCC*) * size);
    for(i=0;i<size;i++)
	bbccs[i] = 0;

    CT_DEBUG(3,"  new_recursion(size %d): 0x%p\n", size, bbccs);
    
    return bbccs;
}
  

/*
 * Allocate a new BBCC
 *
 * Uninitialized:
 * cxt, rec_index, rec_array, next_bbcc, next1, next2
 */
static __inline__ 
BBCC* new_bbcc(BB* bb)
{
   BBCC* new;
   Int i;

   new = (BBCC*)VG_(malloc)(sizeof(BBCC));
   new->bb  = bb;
   new->tid = SK_(current_tid);

   new->exe_counter = 0;
   new->ret_counter = 0;
   new->skipped = 0;
   new->cost = SK_(get_costarray)(bb->cost_count);
   for(i=0;i<bb->cost_count;i++)
     new->cost[i] = 0;

   /* Init pointer caches (LRU) */
   new->lru_next_bbcc = 0;
   new->lru_from_jcc  = 0;
   new->lru_to_jcc  = 0;
   
   new->jcc_list  = 0;

   SK_(stat).distinct_bbccs++;

   CT_DEBUG(3, "  new_bbcc(BB 0x%x): %p (now %d)\n", 
	    bb_addr(bb), new, SK_(stat).distinct_bbccs);

   return new;
}


/**
 * Inserts a new BBCC into hashes.
 * BBCC specific items must be set as this is used for the hash
 * keys:
 *  fn     : current function
 *  tid    : current thread ID
 *  from   : position where current function is called from
 *
 * Recursion level doesn't need to be set as this is not included
 * in the hash key: Only BBCCs with rec level 0 are in hashes.
 */
static
void insert_bbcc_into_hash(BBCC* bbcc)
{
    UInt hash;
    
    CT_ASSERT(bbcc->cxt != 0);

    CT_DEBUG(3,"+ insert_bbcc_into_hash(BB 0x%x, fn '%s')\n",
	     bb_addr(bbcc->bb), bbcc->cxt->fn[0]->name);

    /* check fill degree of hash and resize if needed (>90%) */
    current_bbccs.entries++;
    if (100 * current_bbccs.entries / current_bbccs.size > 90)
	resize_bbcc_hash();

    hash = hash_bbcc(bbcc->bb, bbcc->cxt, current_bbccs.size);
    bbcc->next = current_bbccs.table[hash];
    current_bbccs.table[hash] = bbcc;

    CT_DEBUG(3,"- insert_bbcc_into_hash: %d entries\n",
	     current_bbccs.entries);
}

static Char* mangled_cxt(Context* cxt, int rec_index)
{
    static Char mangled[FN_NAME_LEN];
    int i, p;

    if (!cxt) return "(no context)";

    p = VG_(sprintf)(mangled, "%s", cxt->fn[0]->name);
    if (rec_index >0)
	p += VG_(sprintf)(mangled+p, "'%d", rec_index +1);
    for(i=1;i<cxt->size;i++)
	p += VG_(sprintf)(mangled+p, "'%s", cxt->fn[i]->name);

    return mangled;
}


/* Create a new BBCC as a copy of an existing one,
 * but with costs set to 0 and jcc chains empty.
 *
 * This is needed when a BB is executed in another context than
 * the one at instrumentation time of the BB.
 *
 * Use cases:
 *  rec_index == 0: clone from a BBCC with differing tid/cxt
 *                  and insert into hashes
 *  rec_index >0  : clone from a BBCC with same tid/cxt and rec_index 0
 *                  don't insert into hashes
 */
static BBCC* clone_bbcc(BBCC* orig, Context* cxt, Int rec_index)
{
    BBCC*      new;

    CT_DEBUG(3,"+ clone_bbcc(BB 0x%x, rec %d, fn %s)\n",
	     bb_addr(orig->bb), rec_index, cxt->fn[0]->name);

    new  = new_bbcc(orig->bb);

    if (rec_index == 0) {

      /* hash insertion is only allowed if tid or cxt is different */
      CT_ASSERT((orig->tid != SK_(current_tid)) ||
		(orig->cxt != cxt));

      new->rec_index = 0;
      new->cxt = cxt;
      new->rec_array = new_recursion(cxt->fn[0]->separate_recursions);
      new->rec_array[0] = new;

      insert_bbcc_into_hash(new);
    }
    else {
      if (SK_(clo).separate_threads)
	CT_ASSERT(orig->tid == SK_(current_tid));

      CT_ASSERT(orig->cxt == cxt);
      CT_ASSERT(orig->rec_array);
      CT_ASSERT(cxt->fn[0]->separate_recursions > rec_index);
      CT_ASSERT(orig->rec_array[rec_index] ==0);

      /* new BBCC will only have differing recursion level */
      new->rec_index = rec_index;
      new->cxt = cxt;
      new->rec_array = orig->rec_array;
      new->rec_array[rec_index] = new;
    }

    /* update list of BBCCs for same BB */
    new->next_bbcc = orig->bb->bbcc_list;
    orig->bb->bbcc_list = new;


    CT_DEBUGIF(3)
      SK_(print_bbcc)(-2, new, False);

    CT_DEBUG(2,"- clone_BBCC(0x%p, %d) for BB 0x%x\n"
		"   orig %s\n"
		"   new  %s\n",
	     orig, rec_index, bb_addr(orig->bb),
	     mangled_cxt(orig->cxt, orig->rec_index),
	     mangled_cxt(new->cxt, new->rec_index));

    SK_(stat).bbcc_clones++;
 
    return new;
};



/* Get a pointer to the cost centre structure for given basic block
 * address. If created, the BBCC is inserted into the BBCC hash.
 * Also sets BB_seen_before by reference.
 *
 */ 
BBCC* SK_(get_bbcc)(BB* bb)
{
   BBCC* bbcc;

   VGP_PUSHCC(VgpCacheGetBBCC);

   CT_DEBUG(3, "+ get_bbcc(BB 0x%x)\n", bb_addr(bb));

   bbcc = bb->bbcc_list;

   if (!bbcc) {
     bbcc = new_bbcc(bb);

     /* initialize BBCC */
     bbcc->cxt       = 0;
     bbcc->rec_array = 0;
     bbcc->rec_index = 0;

     bbcc->next_bbcc = bb->bbcc_list;
     bb->bbcc_list = bbcc;
     bb->last_bbcc = bbcc;

     CT_DEBUGIF(3)
       SK_(print_bbcc)(-2, bbcc, False);
   }

   CT_DEBUG(3, "- get_bbcc(BB 0x%x): BBCC 0x%p\n",
		bb_addr(bb), bbcc);

   VGP_POPCC(VgpCacheGetBBCC);
   return bbcc;
}


/* Callgrind manages its own call stack for each thread.
 * When leaving a function, a underflow can happen when
 * Callgrind's tracing was switched on in the middle of
 * a run, i.e. when Callgrind was not able to trace the
 * Call.
 * This function tries to reconstruct the original call.
 * As we know the return address (the address following
 * the CALL instruction), we can detect the function
 * we return back to, but the original call site is unknown.
 * We suppose a call site at return address - 1.
 * (TODO: other heuristic: lookup info of instrumented BBs).
 */
static void handleUnderflow(BB* bb)
{
  /* RET at top of call stack */
  BBCC* source_bbcc;
  BB* source_bb;
  jCC* jcc;
  Bool seen_before;
  fn_node* caller;
  int fn_number, *pactive;
  call_entry* call_entry_up;

  CT_DEBUG(1,"  Callstack underflow !\n");

  /* we emulate an old call from the function we return to
   * by using (<return address> -1) */
  source_bb = SK_(get_bb)(bb_addr(bb)-1, 0, &seen_before);
  source_bbcc = SK_(get_bbcc)(source_bb);

  /* seen_before can be true if RET from a signal handler */
  if (!seen_before) {
    source_bbcc->exe_counter = SK_(current_state).collect ? 1 : 0;
  }
  else if (SK_(current_state).collect)
    source_bbcc->exe_counter++;
  
  /* Force a new top context, will be set active by push_cxt() */
  SK_(current_fn_stack).top--;
  SK_(current_state).cxt = 0;
  caller = SK_(get_fn_node)(bb);
  SK_(push_cxt)( caller );

  if (!seen_before) {
    /* set rec array for source BBCC: this is at rec level 1 */
    source_bbcc->rec_array = new_recursion(caller->separate_recursions);
    source_bbcc->rec_array[0] = source_bbcc;

    CT_ASSERT(source_bbcc->cxt == 0);
    source_bbcc->cxt = SK_(current_state).cxt;
    insert_bbcc_into_hash(source_bbcc);
  }
  CT_ASSERT(SK_(current_state).bbcc);

  /* correct active counts */
  fn_number = SK_(current_state).bbcc->cxt->fn[0]->number;
  pactive = SK_(get_fn_entry)(fn_number);
  (*pactive)--;

  /* This assertion is not correct for reentrant
   * signal handlers */
  /* CT_ASSERT(*pactive == 0); */

  SK_(current_state).nonskipped = 0; /* we didn't skip this function */
  /* back to current context */
  SK_(push_cxt)( SK_(current_state).bbcc->cxt->fn[0] );
  SK_(push_call_stack)(source_bbcc, SK_(current_state).bbcc,
		       (Addr)-1, False);
  call_entry_up = 
    &(SK_(current_call_stack).entry[SK_(current_call_stack).sp -1]);
  jcc = call_entry_up->jcc;
  /* assume this call is lasting since last dump or
   * for a signal handler since it's call */
  if (SK_(current_state).sig == 0)
    SK_(copy_cost)( SK_(sets).full, call_entry_up->enter_cost,
		    SK_(get_current_thread)()->lastdump_cost );
  else
    SK_(zero_cost)( SK_(sets).full, call_entry_up->enter_cost );
}


/*
 * Helper function called at start of each instrumented BB to setup
 * pointer to costs for current thread/context/recursion level
 */

__attribute__ ((regparm (1)))
void SK_(setup_bbcc)(BB* bb)
{
  BBCC *bbcc, *last_bbcc;
  Bool  call_emulation = False, delayed_push = False, skip = False;
  Addr esp;
  BB* last_bb;

  VGP_PUSHCC(VgpCacheSetup);

  CT_DEBUG(3,"+ setup_bbcc(BB 0x%x)\n", bb_addr(bb));

  esp = VG_(get_stack_pointer)();
  last_bbcc = SK_(current_state).bbcc;
  last_bb = last_bbcc ? last_bbcc->bb : 0;

  /* Manipulate JmpKind if needed, only using BB specific info */

  if (( SK_(current_state).jmpkind != JmpRet) &&
      ( SK_(current_state).jmpkind != JmpCall) && last_bb) {

    /* We simulate a JMP/Cont to be a CALL if
     * - jump is in another ELF object or section kind
     * - jump is to first instruction of a function (tail recursion)
     */
    if (bb->is_entry ||
	(last_bb->sect_kind != bb->sect_kind) ||
	(last_bb->obj->number != bb->obj->number)) {
      
      SK_(current_state).jmpkind = JmpCall;
      call_emulation = True;
      
      CT_DEBUG(1,"     JMP: %s to %s!\n",
	       last_bb->obj->name, bb->obj->name);
    }
  }

  if (SK_(current_state).jmpkind == JmpCall)
    skip = SK_(get_fn_node)(bb)->skip;

  CT_DEBUGIF(1) {
    VG_(printf)("%s %08x -> %08x, ESP %08x\n",
		(SK_(current_state).jmpkind == JmpNone)  ? "CONT" :
		(SK_(current_state).jmpkind == JmpCond)  ? "JCND" :
		(SK_(current_state).jmpkind == JmpRet)   ? "RET " :
		(SK_(current_state).jmpkind == JmpCall)  ? "CALL" :
		(SK_(current_state).jmpkind == JmpBoring)? "JMP " :
		(SK_(current_state).jmpkind == JmpSyscall)? "SYSC" : "CREQ",
		last_bb ? bb_jmpaddr(last_bb) : 0,
		bb_addr(bb), esp);
  }

  /* Handle CALL/RET and update context to get correct BBCC */
  
  if (SK_(current_state).jmpkind == JmpRet) {
    
    if ((SK_(current_call_stack).sp == 0) || 
	((SK_(current_fn_stack).top > SK_(current_fn_stack).bottom) &&
	 ( *(SK_(current_fn_stack).top-1)==0)) ) {

      /* On an empty call stack or at a signal separation marker,
       * a RETURN generates an call stack underflow.
       */	
      handleUnderflow(bb);
      SK_(pop_call_stack)();
    }
    else {
      if (SK_(unwind_call_stack)(esp)==0) {
	
	call_entry* call_entry_up = 
	  &(SK_(current_call_stack).entry[SK_(current_call_stack).sp-1]);
	
	/* Return without Call, i.e. a Jump.
	 * If this is a jump into another function,
	 * pretend a Return and a Call happening.
	 * TODO: Check for jump to same function
	 */
	CT_DEBUG(1,"     RET without call!\n");
	
	/* change source for delayed push */
	if (call_entry_up->jcc) {
	  SK_(current_state).bbcc = call_entry_up->jcc->from;
	  esp = call_entry_up->esp;
	  SK_(pop_call_stack)();
	}
	else {
	  CT_ASSERT(SK_(current_state).nonskipped != 0);
	}
	
	skip = SK_(get_fn_node)(bb)->skip;
	delayed_push = True;
	
      }
    }
  }
  else {
    SK_(unwind_call_stack)(esp);
    
    if (SK_(current_state).jmpkind == JmpCall) {
      delayed_push = True;

      if (call_emulation && SK_(current_call_stack).sp>0) {
	call_entry* call_entry_up = 
	  &(SK_(current_call_stack).entry[SK_(current_call_stack).sp-1]);
	
	esp = call_entry_up->esp;
      }
    }
  }
  
  /* Change new context if needed, taking delayed_push into account */
  if ((delayed_push && !skip) || (SK_(current_state).cxt == 0)) {
    SK_(push_cxt)(SK_(get_fn_node)(bb));
  }
  CT_ASSERT(SK_(current_fn_stack).top > SK_(current_fn_stack).bottom);
  
  /* If there is a fresh instrumented BBCC, assign current context */
  bbcc = SK_(get_bbcc)(bb);
  if (bbcc->cxt == 0) {
    CT_ASSERT(bbcc->rec_array == 0);
      
    bbcc->cxt = SK_(current_state).cxt;
    bbcc->rec_array = 
      new_recursion((*SK_(current_fn_stack).top)->separate_recursions);
    bbcc->rec_array[0] = bbcc;
      
    insert_bbcc_into_hash(bbcc);
  }
  else {
    /* get BBCC with current context */
    
    /* first check LRU of last bbcc executed */
    
    if (last_bbcc) {
      bbcc = last_bbcc->lru_next_bbcc;
      if (bbcc &&
	  ((bbcc->bb != bb) ||
	   (bbcc->cxt != SK_(current_state).cxt)))
	bbcc = 0;
    }
    else
      bbcc = 0;

    if (!bbcc)
      bbcc = lookup_bbcc(bb, SK_(current_state).cxt);
    if (!bbcc)
      bbcc = clone_bbcc(bb->bbcc_list, SK_(current_state).cxt, 0);
    
    bb->last_bbcc = bbcc;
  }

  /* save for fast lookup */
  if (last_bbcc)
    last_bbcc->lru_next_bbcc = bbcc;

  if ((*SK_(current_fn_stack).top)->separate_recursions >1) {
    UInt level, idx;
    fn_node* top = *(SK_(current_fn_stack).top);

    level = *SK_(get_fn_entry)(top->number);

    if (delayed_push && !skip) {
      if (SK_(clo).skip_direct_recursion) {
	/* do not increment rec. level if called from
	 * same function */
	if (!SK_(current_state).bbcc || 
	    (SK_(current_state).bbcc->cxt->fn[0] != bbcc->cxt->fn[0]))
	  level++;
      }
      else level++;
    }
    if (level> top->separate_recursions)
      level = top->separate_recursions;

    if (level == 0) {
      /* can only happen if instrumentation just was switched on */
      level = 1;
      *SK_(get_fn_entry)(top->number) = 1;
    }

    idx = level -1;
    if (bbcc->rec_array[idx])
      bbcc = bbcc->rec_array[idx];
    else
      bbcc = clone_bbcc(bbcc, SK_(current_state).cxt, idx);

    CT_ASSERT(bbcc->rec_array[bbcc->rec_index] == bbcc);
  }

  if (delayed_push) {
    if (!skip && SK_(current_state).nonskipped) {
      /* a call from skipped to nonskipped */
      SK_(current_state).bbcc = SK_(current_state).nonskipped;
    }
    SK_(push_call_stack)(SK_(current_state).bbcc,
			 bbcc, esp, skip);
  }
  
  if (SK_(current_state).jmpkind == JmpCond ||
      SK_(current_state).jmpkind == JmpBoring) {
    
    /* Handle conditional jumps followed, i.e. trace arcs
     * This uses JCC structures, too */
    
    jCC* jcc = SK_(get_jcc)(last_bbcc, bbcc);
    CT_ASSERT(jcc != 0);
    // Change from default, and check if already changed
    if (jcc->jmpkind == JmpCall)
      jcc->jmpkind = SK_(current_state).jmpkind;
    else
      CT_ASSERT(jcc->jmpkind == SK_(current_state).jmpkind);
    
    jcc->call_counter++;
    if (SK_(current_state).jmpkind == JmpCond)
      SK_(stat).jcnd_counter++;
    else
      SK_(stat).jump_counter++;
  }
  
  SK_(current_state).jmpkind = JmpNone;
  SK_(current_state).bbcc = bbcc;
  
  CT_DEBUGIF(1) {
    VG_(printf)("     ");
    SK_(print_bbcc_fn)(bbcc);
    VG_(printf)("\n");
  }
  
  CT_DEBUG(3,"- setup_bbcc (BB %p): Cost %p (Len %d), Instrs %d (Len %d)\n",
	   bb_addr(bb), bbcc->cost, bb->cost_count, 
	   bb->instr_count, bb->instr_len);
  CT_DEBUGIF(3)
    SK_(print_cxt)(-8, SK_(current_state).cxt, bbcc->rec_index);
  CT_DEBUG(3,"\n");
  
  if (SK_(current_state).collect && !SK_(current_state).nonskipped)
    bbcc->exe_counter++;
    
  (*SK_(cachesim).after_bbsetup)();

  SK_(stat).bb_executions++;

  VGP_POPCC(VgpCacheSetup);
}
