/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2005 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.freedesktop.cairo;

import org.gnu.glib.Struct;
import org.gnu.glib.Handle;

/**
 * TODO: error handling
 */
public class Pattern extends CairoObject {

    /**
     * Create a new cairo_pattern_t corresponding to a opaque color. The color
     * components are floating point numbers in the range 0 to 1. If the values
     * passed in are outside that range, they will be clamped.
     * 
     * @param red
     * @param green
     * @param blue
     */
    public Pattern(double red, double green, double blue) {
        super(cairo_pattern_create_rgb(red, green, blue));
    }

    /**
     * Create a new cairo_pattern_t corresponding to a translucent color. The
     * color components are floating point numbers in the range 0 to 1. If the
     * values passed in are outside that range, they will be clamped.
     * 
     * @param red
     * @param green
     * @param blue
     * @param alpha
     */
    public Pattern(double red, double green, double blue, double alpha) {
        super(cairo_pattern_create_rgba(red, green, blue, alpha));
    }

    Pattern(Handle hndl) {
        super(hndl);
    }

    protected void finalize() throws Throwable {
        cairo_pattern_destroy(getHandle());
        super.finalize();
    }

    /**
     * Sets the transformation matrix for this pattern.
     * 
     * @param matrix
     *            The transformation matrix.
     */
    public void setMatrix(Matrix matrix) {
        cairo_pattern_set_matrix(getHandle(), matrix.getHandle());
    }

    /**
     * Returns the current transform matrix of this pattern. Note that this
     * method returns a new matrix object and you must dispose it.
     * 
     * @return The transformation matrix for the pattern.
     */
    public Matrix getMatrix() {
        Handle hndl = cairo_pattern_get_matrix(getHandle());
        return new Matrix(hndl);
    }

    /*
     * Native calls
     */
    native static final private void cairo_pattern_destroy(Handle pat);

    native static final private void cairo_pattern_set_matrix(Handle pat,
            Handle matrix);

    native static final private Handle cairo_pattern_get_matrix(Handle pat);

    native static final private Handle cairo_pattern_create_rgb(double red,
            double green, double blue);

    native static final private Handle cairo_pattern_create_rgba(double red,
            double green, double blue, double alpha);

    // Used by SurfacePattern
    native static final protected Handle cairo_pattern_create_for_surface(
            Handle surface);

    native static final protected void cairo_pattern_set_extend(Handle pat,
            int extend);

    native static final protected int cairo_pattern_get_extend(Handle pat);

    native static final protected void cairo_pattern_set_filter(Handle pat,
            int filter);

    native static final protected int cairo_pattern_get_filter(Handle pat);

    // Used by Gradient
    native static final protected void cairo_pattern_add_color_stop_rgb(
            Handle pat, double offset, double red, double green, double blue);

    native static final protected void cairo_pattern_add_color_stop_rgba(
            Handle pat, double offset, double red, double green, double blue,
            double alpha);

    // Used by LinearGradient
    native static final protected Handle cairo_pattern_create_linear(double x0,
            double y0, double x1, double y1);

    // used by RadialGradient
    native static final protected Handle cairo_pattern_create_radial(
            double cx0, double yx0, double radius0, double cx1, double cy1,
            double radius1);
}
