/* $Id: HFQuadtree.h,v 1.11 2003/02/05 23:13:06 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Copyright (C) 1999 Amit J. Patel
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#ifndef ENGINE_QUADTREE_H
#define ENGINE_QUADTREE_H

#include <vector>
#include <map>

#include <Ark/Ark.h>
#include <Ark/ArkMath.h>
#include <Ark/ArkRay.h>
#include <Ark/ArkVBuffer.h>

#include <Modules/HeightField/ArkHeightField.h>
#include <Modules/HeightField/HFWorld.h>

namespace Ark
{
   class Patch;
   class QuadtreeNode;
 
   /**
    * Those are height-field specific entity data, a list of patches this
    * entity belongs to. This structure is created when the entity is added
    * to the world, it is updated when the entity moves, and destroyed when
    * the entity is removed of the world.
    * It is stored in the m_WorldData member of the entity.
    */
   class ARKHEIGHTFIELD_DLL_API HFEntityData
   {
       public:
	   BBox m_OldBbox;
	   std::vector< Patch* > m_Patches;
   };

   // Note: we use a map to ensure there no collision is computed
   //       twice.
   struct ARKHEIGHTFIELD_DLL_API Collider
   {
	 Entity *m_E1, *m_E2;

	 Collider (Entity *e1, Entity *e2)
	 {
	    assert (e1 != e2);

	    if (e1 > e2)
	    {
	       m_E1 = e1;
	       m_E2 = e2;
	    }
	    else
	    {
	       m_E1 = e2;
	       m_E2 = e1;
	    }
	 }
   };

   ARKHEIGHTFIELD_DLL_API bool operator < (const Collider &b, const Collider &c);
   ARKHEIGHTFIELD_DLL_API bool operator == (const Collider &b, const Collider &c);

   typedef std::map<Collider, bool> ColliderList;

   /**
    * Handles the common data for rendering quadtree patches.
    */
   class QuadtreeRenderManager;

   /** 
    *
    */
   class ARKHEIGHTFIELD_DLL_API QuadtreeNode
   {
      public:
	 QuadtreeNode();
	 virtual ~QuadtreeNode ();

	 virtual void Build(HeightField* hf, int mx, int mz, int size, QuadtreeRenderManager* manager);
	 void Render(const Frustum& view, Visibility vis);
	 Patch* FindPatch(scalar x, scalar z);
	 virtual void InvalidatePatches(const BBox& box);
	 virtual void CollectColliders (ColliderList &list);
	 inline const BBox& GetBoundingBox() const { return m_BBox; };


	 bool RayTrace (const Ray &ray,
			std::vector<int>& potentialtris,
			EntityList& potentialents);

      protected:
	 QuadtreeNode *m_Children[4];
	 BBox m_BBox;

	 uchar m_Vis;
   };

   /**
    * A quadtree...
    */
   class ARKHEIGHTFIELD_DLL_API Quadtree
   {
	 HeightField *m_Heightfield;
	 QuadtreeRenderManager* m_RenderManager;
	 QuadtreeNode *m_Rootnode;
	 int m_Size;

      public:
	 Quadtree (HeightField *heightfield);
	 ~Quadtree ();
	 
	 void Render(Renderer &renderer, const Camera& camera, const Color& fogColor);

	 /**
	  * Mark the patches in the specified area as "invalidate", which
	  * means their topology has changed since the quadtree has been
	  * created.
	  */
	 void Invalidate(scalar minx, scalar minz, scalar maxx, scalar maxz);

	 void SetMaterials (const MaterialList& materials);
	 
	 // Trace a ray through the world trying to detect a collision...
	 bool RayTrace (const Ray &ray, Collision &collision, bool test_ents);

	 // Remove an entity from the quadtree's patches...
	 void RemoveEntity (Entity *ent, HFEntityData *data);

	 // Add/update an entity in the quadtree's patches.
	 void UpdateEntity (Entity *ent, HFEntityData *data);

	 /**
	  * Make a list of potential colliders.
	  * Note: we use a map to ensure there no collision is computed
	  * twice.
	  */
	 void GetColliders (ColliderList &list);
   };


} // namespace Ark


#endif // QUADTREE_HPP
