/*
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the ArkRPG Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef WIN32
 #include <windows.h>
#endif

#include <Client/MessageView.h>
#include <Client/UICutText.h>

namespace Client
{

   Ark::String MsgGetText(const Ark::String &msg)
   {
       Ark::String::size_type pos = msg.find("|", 0);

      if (pos != Ark::String::npos)
	 return msg.substr(pos + 1, Ark::String::npos);
      else 
	 return msg;
   }

   Ark::String MsgGetKeyword(const Ark::String &msg)
   {
       Ark::String::size_type pos = msg.find("|", 0);

      if (pos != Ark::String::npos)
	 return msg.substr(0, pos);
      else 
	 return msg;
   }

   MessageLabel::MessageLabel () : m_Image (NULL)
   {}
   
   MessageLabel::MessageLabel (const Box &box,
			       Ark::Entity *my,
			       const Ark::String &msg,
			       Ark::Entity *dest) :
      m_Msg(MsgGetText(msg)),
      m_Reply(MsgGetKeyword(msg)),
      m_Image (NULL),
      m_MyEntity (my),
      m_DestEntity (dest),
      m_Box(box)
   {
   }

   MessageLabel::MessageLabel (const Box &box,
			       const Ark::TexturePtr& image) : 
      m_Msg (""),
      m_Image (image),
      m_MyEntity (NULL),
      m_DestEntity (NULL),
      m_Box(box)
   {
   }

   MessageLabel::~MessageLabel()
   {
   }
		       
   void
   MessageLabel::Render(UIRenderer *ui)
   {
      if (m_Image)
      {
	 ui->SetTexture (m_Image);
	 ui->DrawTexturedRectangle (m_Box.m_MinX, m_Box.m_MinY,
				    m_Box.m_MaxX, m_Box.m_MaxY);
      }
      else
      {
	 Ark::FontPtr font;
	 if (m_DestEntity)
	    font = ui->GetFont ("arial1-1-ffff00");
	 else
	    font = ui->GetFont ("arial1-1-ffffff");

	 CutTextDraw ctd;
	 CutText(ui, font.Get(), m_Box.m_MinY, m_Box.m_MinX,
		 m_Box.m_MinY, m_Box.m_MaxX, m_Box.m_MaxY,
		 m_Msg, ctd);
      }
   }

   bool
   MessageLabel::HandleMButton (bool down, int b, int x, int y)
   {
      if (m_DestEntity == NULL || m_Image)
	 return false;

      if (m_Box.IsInside(x,y))
      {
	 // Send the answer and remove message from stack
	 m_MyEntity->AddMessage (m_DestEntity, m_Reply);
	 m_MyEntity->m_Messages.erase (m_MyEntity->m_Messages.begin());

	 return true;
      }

      return false;
   }
   
   ////////////////////////////////////////////////////////////////////////


   MessageView::MessageView (UIRenderer *ui) : 
   	 m_PlayerEntity (0),
	 m_Style (ui)
   {
      Ark::Config cfg;
      cfg.Load("{game}/data/misc/messageview.cfg");

      m_PaddingX = cfg.GetInt("messageView::PaddingX", 15);
      m_PaddingY = cfg.GetInt("messageView::PaddingY", 15);

      m_FontQuestion = cfg.GetStr("messageView::FontQuestion",
				  "arial1-1-ffffff");
      m_FontAnswer = cfg.GetStr("messageView::FontAnswer",
				"arial1-1-ffff00");

      int expectedW = cfg.GetInt("messageView::ExpectedWidth", 800);
      const float scrfactor = scalar(ui->width) / scalar(expectedW);

      m_MinX = int(scrfactor*cfg.GetInt("messageView::MinX", 200));
      m_MaxX = int(scrfactor*cfg.GetInt("messageView::MaxX", 600));
      m_MinY = int(scrfactor*cfg.GetInt("messageView::MinY", 20));
      m_MaxY = int(scrfactor*cfg.GetInt("messageView::MaxY", 600));
   }

   MessageView::~MessageView ()
   {}

   void
   MessageView::AppendChild (UIRenderer *ui,
			     const Ark::String &text,
			     Ark::FontPtr font,
			     int startx,
			     Ark::Entity *ent,
			     Ark::Entity *dest)
   {
      CutBoxCompute ca;

      CutText(ui, font.Get(), m_Box.m_MaxY, m_MinX+startx,
	      m_Box.m_MaxY, m_MaxX, m_MaxY, MsgGetText(text), ca);
      m_Children.push_back (MessageLabel (ca.GetBox(), ent, text, dest));
      m_Box.AddBox(ca.GetBox());
   }

   void
   MessageView::UpdateChildren (UIRenderer *ui, Ark::Entity *ent)
   {

      if (ent->m_Messages.empty()) return;
	    
      Ark::EntityMessage &msg = ent->m_Messages[0];

      m_Box.MakeEmpty();
      m_Box.Add(m_MinX, m_MinY);

      // Add message
      AppendChild(ui, msg.m_Message, ui->GetFont (m_FontQuestion), 0,
		  ent, NULL);

      // Add answers
      for (size_t i = 0; i < msg.m_AnswerList.size(); i++)
      {
	 AppendChild(ui, msg.m_AnswerList[i], ui->GetFont (m_FontAnswer), 20,
		     ent, msg.m_Entity);
      }
      
      m_HasTimeout = msg.m_AnswerList.size()==0;
   }
   
   void
   MessageView::Render(UIRenderer *ui, Ark::Entity *ent)
   {
      if (ent == NULL || ent->m_Messages.empty())
	 return;

#define DEF_TIMEOUT 2.5
      if (m_Children.empty())
      {
	 UpdateChildren (ui, ent);
	 m_Timer.Update();
      }

      if (m_HasTimeout && m_Timer.GetDelta() > DEF_TIMEOUT)
      {
	 if (m_PlayerEntity && !m_PlayerEntity->m_Messages.empty())
	 {
	    m_PlayerEntity->m_Messages.erase
	       (m_PlayerEntity->m_Messages.begin());
	 }
	 m_Children.resize(0);
      }

      if (m_Children.empty()) return;

      m_Style.RenderSkin(ui, m_Box.m_MinX-m_PaddingX, m_Box.m_MinY-m_PaddingY,
			 m_Box.GetWidth()+2*m_PaddingX,
			 m_Box.GetHeight()+m_PaddingY);

      // Render my children.
      std::vector<MessageLabel>::iterator i;
      for (i = m_Children.begin(); i != m_Children.end(); i++)
	 i->Render (ui);

      m_PlayerEntity = ent;
   }

   bool
   MessageView::HandleMButton (bool down, int b, int x, int y)
   {
      if (m_PlayerEntity == NULL ||
	  m_PlayerEntity->m_Messages.empty())
	 return false;

      if (down == false || b != 1)
	 return false;

	    
      std::vector<MessageLabel>::iterator i;
	    
      for (i = m_Children.begin(); i != m_Children.end(); i++)
      {
	 if (i->HandleMButton (down, b, x, y))
	 {
	    m_Children.resize(0);
	    return true;
	 }
      }

      return false;
   }

}

