/* $Id: ArkSocket.h,v 1.6 2002/10/11 01:10:05 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_SOCKET_H
#define ARK_SOCKET_H

#include <Ark/Ark.h>
#include <Ark/ArkString.h>
#include <Ark/ArkBuffer.h>


namespace Ark
{
   struct ARK_DLL_API NetAddress
   {
	 uint8  ip[4];
	 uint16 port;
	 uint16 pad;
   };

   class SocketP;

   //  =======================================================================
   /// A class handling socket connections (UDP or TCP).
   //  =======================================================================
   class ARK_DLL_API Socket
   {
	 /**
	  * Create a socket but don't connect/bind it to anything yet. It
	  * is only used in the accept function yet.
	  */
	 Socket ()
	 {m_Priv = NULL;}
	 
      public:
	 /**
	  * Create a socket and connect it to \c host:port, where host is
	  * the IP address or the hostname of the remote host, and port is
	  * a port number. \c protocol can be "udp" or "tcp".
	  */
	 Socket (const String &host,
		 const String &port,
		 const String &protocol);

	 /**
	  * Create a new server socket. This server will listen on the given
	  * port, on the machine which runs the program. Clients asking for
	  * connections are put on an accept queue ; the program can accept
	  * a connection to a client by calling the Accept() function.
	  *
	  * If the queue grows bigger than \c backlog, then clients trying
	  * to connect will receive an ECONNREFUSED error.
	  */
	 Socket (const String &port,
		 const String &protocol, int backlog = 5);
	 
	 /// Destroy the given socket
	 ~Socket ();
	
	 /**
	  * If a client wants to connect, this function will create a new
	  * socket for it. If wait is TRUE, it blocks until a connection is
	  * available, else it  returns NULL if there is no waiting socket.
	  */
	 Socket *Accept (bool wait = false);
	 
	 /// Close the socket, unref all packets, etc.
	 void Close ();
	 
	 /**
	  * Returns true if this socket is dead (ie the connection has been
	  * closed, and so no data will be received/sent any more).
	  */
	 bool IsDead ();
	 
	 /**
	  * Make the socket blocking/non-blocking (depending on 'val').
	  * It means that data will be received by ReceiveData() only if
	  * they already are in the socket input buffer. It also disables
	  * the Nagle algorithm, which buffers packets for up to .2 seconds :
	  * that is bad for realtime network games..
	  */
	 void SetBlocking (bool val);
	 
	 /**
	  * Wait that the socket can receive data, using the select() call
	  * \param ms is the timeout (ie the maximum time we will wait for
	  * data) ; if it is -1, then this call will block until there is data.
	  *
	  * \return true if data are available.
	  */
	 bool WaitForData (int ms = -1);
	 
	 /**
	  * Updates the socket input buffer. If the socket is in blocking
	  * mode, the function will return only when a packet has been fully
	  * readen, else it checks if there's data in the input buffer; if
	  * yes it processes them.
	  *
	  * \return True if there was no error while getting data from
	  * network (ie the socket hasn't been closed)
	  */
	 bool ReceiveData ();

	 /**
	  * Flushes the socket output buffer. If the socket is in blocking
	  * mode, the function will return only when all the packet in the
	  * out packet list are  sent, else it sends them until the socket
	  * output buffer is full.
	  *
	  * \return True if there was no error sending data to network.
	  */
	 bool SendData();

	 /**
	  * Returns the oldest message received but not evaluated
	  * (can be nil). You should destroy the message after you have used
	  * it.
	  */
	 std::istringstream *GetPacket ();

	 /**
	  * Put message on the list of messages to be send. It is referenced,
	  * so you should call Packet::Unref after calling SendPacket(),
	  * except if you want to send the packet to another socket or if you
	  * want to keep it. You should *NEVER* destroy it (because in this
	  * case the reference counter wouldnt be checked, while the socket
	  * may still be using your packet).
	  *
	  * \return True if the packet was successfully added to the out
	  * queue.
	  */
	 bool SendPacket (Packet *packet);

private:
	 friend class SocketP;
	 SocketP *m_Priv;
   };

/* namespace Ark */
}

#endif

