/*
    Copyright (C) 2006-2009 Fons Adriaensen <fons@kokkinizita.net>
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include "adconf.h"


static const char *shdr_a  = "#            id      dist     azim     elev     conn";
static const char *shdr_b  = "#-----------------------------------------------------------------------";
static const char *mhdr10a = "#            W        X        Y";
static const char *mhdr10b = "#-----------------------------------";
static const char *mhdr11a = "#            W        X        Y        Z";
static const char *mhdr11b = "#--------------------------------------------";
static const char *mhdr20a = "#            W        X        Y        U        V";
static const char *mhdr20b = "#-----------------------------------------------------";
static const char *mhdr21a = "#            W        X        Y        Z        U        V";
static const char *mhdr21b = "#--------------------------------------------------------------";
static const char *mhdr22a = "#            W        X        Y        Z        R        S        T        U        V";
static const char *mhdr22b = "#-----------------------------------------------------------------------------------------";
static const char *mhdr30a = "#            W        X        Y        U        V        P        Q";
static const char *mhdr30b = "#-----------------------------------------------------------------------";
static const char *mhdr31a = "#            W        X        Y        Z        U        V        P        Q";
static const char *mhdr31b = "#--------------------------------------------------------------------------------";
static const char *jackp   = "system:playback";


Speaker::Speaker (void)
{
    reset ();
}


void Speaker::reset (void)
{
    _r = 1.0f;
    _a = 0.0f;
    _e = 0.0f;
    _x = 1.0f;
    _y = 0.0f;
    _z = 0.0f;
    *_label = 0;
    *_jackp = 0;
}


Matrow::Matrow (void)
{
}


void Matrow::reset (void)
{
    _w = _x = _y = _z = _r = _s = _t = _u = _v = _p = _q = 0;
}


AD_conf::AD_conf (void)
{
    _state     = 0;
    _fname [0] = 0;
    reset ();
}


void AD_conf::reset (void)
{
    int i;

    _fform = 2;
    _descr [0] = 0;
    _dec.h_ord = 1;
    _dec.v_ord = 0;
    _dec.nband = 1;
    _dec.nspkr = 4;
    _dec.scale = FUMA;
    _opt.scale = FUMA;
    _opt.nfeff = NONE;
    _opt.delay = OFF;
    _opt.level = OFF;
    _opt.xfreq = 500;
    _opt.ratio = 0;
    for (i = 0; i < 4; i++)
    {
	_lfgain [i] = 1.0f;
	_hfgain [i] = 1.0f;
    }
    for (i = 0; i < MAXOP; i++)
    {
	_speakers [i].reset ();
	_lfmatrix [i].reset ();
	_hfmatrix [i].reset ();
    }
}


int AD_conf::save (const char *file)
{
    FILE          *F;
    Speaker       *S;
    char          *p;
    time_t        t;
    int           i;

    strcpy (_fname, file);
    p = strrchr (_fname, '.');
    if (!p) strcat (_fname, ".ambdec");
    if (! (F = fopen (_fname, "w"))) 
    {
	fprintf (stderr, "Can't open '%s' for writing\n", _fname);
        return 1;
    } 

    t = time (0);
    fprintf (F, "# AmbDec configuration\n");
    fprintf (F, "# Written by AmbDec-%s at %s\n", VERSION, ctime (&t));
    fprintf (F, "/description      %s\n\n", _descr);
    fprintf (F, "/version          %d\n\n", _fform);
    fprintf (F, "/dec/hor_order    %d\n", _dec.h_ord); 
    fprintf (F, "/dec/ver_order    %d\n", _dec.v_ord); 
    fprintf (F, "/dec/freq_bands   %d\n", _dec.nband); 
    fprintf (F, "/dec/speakers     %d\n", _dec.nspkr); 
    switch (_dec.scale)
    {
    case N3D:	fprintf (F, "/dec/coeff_scale  n3d\n");  break;
    case SN3D: 	fprintf (F, "/dec/coeff_scale  sn3d\n"); break;
    case FUMA:	fprintf (F, "/dec/coeff_scale  fuma\n"); break;
    }
    fprintf (F, "\n"); 
    switch (_opt.scale)
    {
    case N3D: 	fprintf (F, "/opt/input_scale  n3d\n");  break;
    case SN3D: 	fprintf (F, "/opt/input_scale  sn3d\n"); break;
    case FUMA:	fprintf (F, "/opt/input_scale  fuma\n"); break;
    }
    switch (_opt.nfeff)
    {
    case NONE: 	 fprintf (F, "/opt/nfeff_comp   none\n");   break;
    case INPUT:	 fprintf (F, "/opt/nfeff_comp   input\n");  break;
    case OUTPUT: fprintf (F, "/opt/nfeff_comp   output\n"); break;
    }
    switch (_opt.delay)
    {
    case OFF: 	 fprintf (F, "/opt/delay_comp   off\n"); break;
    case ON:	 fprintf (F, "/opt/delay_comp   on\n");  break;
    }
    switch (_opt.level)
    {
    case OFF: 	 fprintf (F, "/opt/level_comp   off\n"); break;
    case ON:	 fprintf (F, "/opt/level_comp   on\n");  break;
    }
    fprintf (F, "/opt/xover_freq   %4.0lf\n", _opt.xfreq); 
    fprintf (F, "/opt/xover_ratio  %4.1lf\n", _opt.ratio); 

    fprintf (F, "\n/speakers/{\n"); 
    fprintf (F, "%s\n%s\n", shdr_a, shdr_b); 
    for (i = 0, S = _speakers; i < _dec.nspkr; i++, S++)
    {
	fprintf (F, "add_spkr   %3s  %8.3lf %8.1lf %8.1lf    %s\n", S->_label, S->_r, S->_a, S->_e, S->_jackp);  
    }
    fprintf (F, "/}\n\n");

    if (_dec.nband == 1) save_matrix (F, "matrix", _lfmatrix, _lfgain);
    else
    {
	save_matrix (F, "lfmatrix", _lfmatrix, _lfgain);
	save_matrix (F, "hfmatrix", _hfmatrix, _hfgain);
    }

    fprintf (F, "\n/end\n");
    fclose (F);
    return 0;
}


void AD_conf::save_matrix (FILE *F, const char *pref, Matrow *matr, float *gain)
{
    int i;
 
    fprintf (F, "/%s/{\n", pref);

    switch (_dec.h_ord)
    {
    case 1:
        fprintf (F, "order_gain    %8.5lf %8.5lf\n", gain [0], gain [1]);
	break;
    case 2:
        fprintf (F, "order_gain    %8.5lf %8.5lf %8.5lf\n", gain [0], gain [1], gain [2]);
	break;
    case 3:
        fprintf (F, "order_gain    %8.5lf %8.5lf %8.5lf %8.5lf\n", gain [0], gain [1], gain [2], gain [3]);
	break;
    }	

    switch ((_dec.h_ord << 4) | _dec.v_ord)
    {
    case 0x10:
	fprintf (F, "%s\n%s\n", mhdr10a, mhdr10b);
	for (i = 0; i < _dec.nspkr; i++, matr++)
	{
	    fprintf (F, "add_row   %8.5lf %8.5lf %8.5lf\n",
		     matr->_w, matr->_x, matr->_y); 
	}
	break;
    case 0x11:
	fprintf (F, "%s\n%s\n", mhdr11a, mhdr11b);
	for (i = 0; i < _dec.nspkr; i++, matr++)
	{
	    fprintf (F, "add_row   %8.5lf %8.5lf %8.5lf %8.5lf\n",
		     matr->_w, matr->_x, matr->_y, matr->_z); 
	}
	break;
    case 0x20:
	fprintf (F, "%s\n%s\n", mhdr20a, mhdr20b);
	for (i = 0; i < _dec.nspkr; i++, matr++)
	{
	    fprintf (F, "add_row   %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf\n",
		     matr->_w, matr->_x, matr->_y, matr->_u, matr->_v); 
	}
	break;
    case 0x21:
	fprintf (F, "%s\n%s\n", mhdr21a, mhdr21b);
	for (i = 0; i < _dec.nspkr; i++, matr++)
	{
	    fprintf (F, "add_row   %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf\n",
		     matr->_w, matr->_x, matr->_y, matr->_z, matr->_u, matr->_v); 
	}
	break;
    case 0x22:
	fprintf (F, "%s\n%s\n", mhdr22a, mhdr22b);
	for (i = 0; i < _dec.nspkr; i++, matr++)
	{
	    fprintf (F, "add_row   %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf\n",
		     matr->_w, matr->_x, matr->_y, matr->_z, matr->_r, matr->_s, matr->_t, matr->_u, matr->_v); 
	}
	break;
    case 0x30:
	fprintf (F, "%s\n%s\n", mhdr30a, mhdr30b);
	for (i = 0; i < _dec.nspkr; i++, matr++)
	{
	    fprintf (F, "add_row   %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf\n",
		     matr->_w, matr->_x, matr->_y, matr->_u, matr->_v, matr->_p, matr->_q); 
	}
	break;
    case 0x31:
	fprintf (F, "%s\n%s\n", mhdr31a, mhdr31b);
	for (i = 0; i < _dec.nspkr; i++, matr++)
	{
	    fprintf (F, "add_row   %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf %8.5lf\n",
		     matr->_w, matr->_x, matr->_y, matr->_z, matr->_u, matr->_v, matr->_p, matr->_q); 
	}
	break;
    }

    fprintf (F, "/}\n\n");
}


int AD_conf::load (const char *file)
{
    FILE          *F;
    Speaker       *S;
    Matrow        *R;
    float         *G;
    char          *p, *q;
    char          buff [1024];
    int           line, stat, n, r;
    int           i1, nsp, nlf, nhf, dim, ord, *cnt;
    float         f1, f2, f3;
    char          s1 [64];

    strcpy (_fname, file);
    p = strrchr (_fname, '.');
    if (!p) strcat (_fname, ".ambdec");
    if (! (F = fopen (_fname, "r"))) 
    {
	fprintf (stderr, "Can't open '%s' for writing\n", _fname);
        return 1;
    } 

    reset ();
    stat = 0;
    line = 0;
    dim = 0;
    ord = 0;
    nsp = 0;
    nlf = 0;
    nhf = 0;
    cnt = 0;
    S = 0;
    R = 0;
    G = 0;

    while (! stat && fgets (buff, 1024, F))
    {
        line++;
        p = buff; 
        while (isspace (*p)) p++;
        if (*p == '#' || *p < ' ') continue;
        q = p;
        while ((*q >= ' ') && !isspace (*q)) q++;
        *q++ = 0;   
        while ((*q >= ' ') && isspace (*q)) q++;

        stat = EXTRA;  
        if (!strcmp (p, "/version"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%d%n", &i1, &n) != 1) stat = ARGS;
	    else
	    {
		q += n;
		if ((i1 < 1) && (i1 > 2))
		{
		    fprintf (stderr, "Line %d: unknown version %d.\n", line, i1);
		    stat = ERROR;
		}
		else _fform = i1;
	    }
	}
	else if (!strcmp (p, "/description"))
	{
	    q [strlen (q) - 1] = 0;
	    strcpy (_descr, q);
	    stat = 0;
	}
        else if (!strcmp (p, "/dec/hor_order"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%d%n", &i1, &n) != 1) stat = ARGS;
	    else
	    {
		q += n;
		if ((i1 < 1) || (i1 > 3))
		{
		    fprintf (stderr, "Line %d: illegal horizontal order %d.\n", line, i1);
		    stat = ERROR;
		}
		else _dec.h_ord = i1;
	    }
	}
        else if (!strcmp (p, "/dec/ver_order"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%d%n", &i1, &n) != 1) stat = ARGS;
	    else
	    {
		q += n;
		if ((i1 < 0) || (i1 > _dec.h_ord) || (i1 + _dec.h_ord > 4)) // HACK
		{
		    fprintf (stderr, "Line %d: illegal vertical order %d.\n", line, i1);
		    stat = ERROR;
		}
		else _dec.v_ord = i1;
	    }
	}
        else if (!strcmp (p, "/dec/dimension"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%d%n", &i1, &n) != 1) stat = ARGS;
	    else
	    {
		q += n;
		if ((i1 != 2) && (i1 != 3))
		{
		    fprintf (stderr, "Line %d: illegal decoder dimension %d.\n", line, i1);
		    stat = ERROR;
		}
		else
		{
                    dim = i1;
                    if (ord)
                    {
	                _dec.h_ord = ord;
	                _dec.v_ord = (dim == 3) ? ord : 0;
		    }
		}
	    }
	}
        else if (!strcmp (p, "/dec/amb_order"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%d%n", &i1, &n) != 1) stat = ARGS;
	    else
	    {
		q += n;
		if ((i1 != 1) && (i1 != 2))
		{
		    fprintf (stderr, "Line %d: illegal decoder order %d.\n", line, i1);
		    stat = ERROR;
		}
		else
		{
                    ord = i1;
                    if (dim)
                    {
	                _dec.h_ord = ord;
	                _dec.v_ord = (dim == 3) ? ord : 0;
		    }
		}
	    }
	}
        else if (!strcmp (p, "/dec/speakers"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%d%n", &i1, &n) != 1) stat = ARGS;
	    else
	    {
		q += n;
		if ((i1 < 4) || (i1 > MAXOP))
		{
		    fprintf (stderr, "Line %d: illegal number of speakers %d.\n", line, i1);
		    stat = ERROR;
		}
		else _dec.nspkr = i1;
	    }
	}
        else if (!strcmp (p, "/dec/freq_bands"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%d%n", &i1, &n) != 1) stat = ARGS;
	    else
	    {
		q += n;
		if ((i1 != 1) && (i1 != 2))
		{
		    fprintf (stderr, "Line %d: illegal number of frequency bands %d.\n", line, i1);
		    stat = ERROR;
		}
		else _dec.nband = i1;
	    }
	}
        else if (!strcmp (p, "/dec/coeff_scale"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%s%n", s1, &n) != 1) stat = ARGS;
  	    q += n;
	    if      (!strcmp (s1, "n3d"))   _dec.scale = N3D;
	    else if (!strcmp (s1, "sn3d"))  _dec.scale = SN3D;
	    else if (!strcmp (s1, "fuma"))  _dec.scale = FUMA;
	    else if (!strcmp (s1, "norm"))  _dec.scale = N3D;   // deprecated
	    else if (!strcmp (s1, "fmset")) _dec.scale = FUMA;  // deprecated
	    else
	    {
		fprintf (stderr, "Line %d: illegal matrix scale '%s'.\n", line, s1);
  	        stat = ERROR;
	    }
	}
        else if (!strcmp (p, "/opt/input_scale"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%s%n", s1, &n) != 1) stat = ARGS;
  	    q += n;
	    if      (!strcmp (s1, "n3d"))   _opt.scale = N3D;
	    else if (!strcmp (s1, "sn3d"))  _opt.scale = SN3D;
	    else if (!strcmp (s1, "fuma"))  _opt.scale = FUMA;
	    else if (!strcmp (s1, "norm"))  _opt.scale = N3D;   // deprecated
	    else if (!strcmp (s1, "fmset")) _opt.scale = FUMA;  // deprecated
	    else
	    {
		fprintf (stderr, "Line %d: illegal input scale '%s'.\n", line, s1);
   	        stat = ERROR;
	    }
	}
        else if (!strcmp (p, "/opt/nfeff_comp"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%s%n", s1, &n) != 1) stat = ARGS;
  	    q += n;
	    if      (!strcmp (s1, "none"))   _opt.nfeff = NONE;
	    else if (!strcmp (s1, "input"))  _opt.nfeff = INPUT;
	    else if (!strcmp (s1, "output")) _opt.nfeff = OUTPUT;
	    else
	    {
		fprintf (stderr, "Line %d: illegal NF compensation option '%s'.\n", line, s1);
 	        stat = ERROR;
	    }
	}
        else if (!strcmp (p, "/opt/delay_comp"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%s%n", s1, &n) != 1) stat = ARGS;
  	    q += n;
	    if      (!strcmp (s1, "off")) _opt.delay = OFF;
	    else if (!strcmp (s1, "on"))  _opt.delay = ON;
	    else
	    {
		fprintf (stderr, "Line %d: illegal delay compensation option '%s'.\n", line, s1);
  	        stat = ERROR;
	    }
	}
        else if (!strcmp (p, "/opt/level_comp"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%s%n", s1, &n) != 1) stat = ARGS;
  	    q += n;
	    if      (!strcmp (s1, "off")) _opt.level = OFF;
	    else if (!strcmp (s1, "on"))  _opt.level = ON;
	    else
	    {
		fprintf (stderr, "Line %d: illegal level compensation option '%s'.\n", line, s1);
 	        stat = ERROR;
	    }
	}
        else if (!strcmp (p, "/opt/xover_freq"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%f%n", &f1, &n) != 1) stat = ARGS;
	    else
	    {
		q += n;
		if ((f1 < 50) || (f1 > 5000))
		{
		    fprintf (stderr, "Line %d: illegal crossover frequency %4.0lf.\n", line, f1);
		    stat = ERROR;
		}
		else _opt.xfreq = f1;
	    }
	}
        else if (!strcmp (p, "/opt/xover_ratio"))
	{
	    if (R || S) stat = SCOPE;
	    if (sscanf (q, "%f%n", &f1, &n) != 1) stat = ARGS;
	    else
	    {
		q += n;
		if ((f1 < -30) || (f1 > 30))
		{
		    fprintf (stderr, "Line %d: illegal HF band gain %4.1lf.\n", line, f1);
		    stat = ERROR;
		}
		else _opt.ratio = f1;
	    }
	}
        else if (!strcmp (p, "/speakers/{"))
	{
            if (R || S) stat = SCOPE;
	    if (nsp) 
	    {
		fprintf (stderr, "Line %d: repeated speaker definition block\n", line);
 	        stat = ERROR;
	    }
	    else S = _speakers;
	}
        else if (!strcmp (p, "add_spkr"))
	{
	    if (!S) stat = SCOPE;
	    if (nsp == _dec.nspkr)
	    {
		fprintf (stderr, "Line %d: too many speaker definitions.\n", line);
		stat = ERROR;
	    }
	    else
	    {
		r = sscanf (q, "%s%f%f%f%n", s1, &f1, &f2, &f3, &n);
		q += n;
		if (r < 4) stat = ARGS;
		else if (strlen (s1) > 3)
		{    
		    fprintf (stderr, "Line %d: speaker name '%s' is too long.\n", line, s1);
		    stat = ERROR;
		}
		else if (f1 < 0.5f) 
		{
		    fprintf (stderr, "Line %d: illegal speaker distance.\n", line);
		    stat = ERROR;
		}
		else if ((f2 < -360) || (f2 > 360))
		{
		    fprintf (stderr, "Line %d: illegal speaker azimuth.\n", line);
		    stat = ERROR;
		}
		else if ((f3 < -90) || (f3 > 90))
		{
		    fprintf (stderr, "Line %d: illegal speaker elevation.\n", line);
		    stat = ERROR;
		}
		else
		{
		    strcpy (S->_label, s1);
		    S->_r = f1;
		    S->_a = f2;
		    S->_e = f3;
		    if (sscanf (q, "%s%n", s1, &n) == 1)
		    {
			strcpy (S->_jackp, s1);
			q += n;
		    }
		    S++;
		    nsp++;
		}
	    }
	}
        else if (!strcmp (p, "/matrix/{"))
	{
	    if (R || S) stat = SCOPE;
	    if (_dec.nband == 2)
	    {
		fprintf (stderr, "Line %d: 'matrix' definition in dual band decoder.\n", line);
		stat = ERROR;
	    }
	    if (nlf)
	    {
		fprintf (stderr, "Line %d: repeated matrix definition.\n", line);
		stat = ERROR;
	    }
            else
	    {
		R = _lfmatrix;
		G = _lfgain;
		cnt = &nlf;
	    }
	}
	else if (!strcmp (p, "/lfmatrix/{"))
	{
	    if (R || S) stat = SCOPE;
	    if (_dec.nband == 1)
	    {
		fprintf (stderr, "Line %d: 'lfmatrix' definition in single band decoder.\n", line);
		stat = ERROR;
	    }
	    if (nlf)
	    {
		fprintf (stderr, "Line %d: repeated matrix definition.\n", line);
		stat = ERROR;
	    }
            else
	    {
		R = _lfmatrix;
		G = _lfgain;
		cnt = &nlf;
	    }
	}
        else if (!strcmp (p, "/hfmatrix/{"))
	{
	    if (R || S) stat = SCOPE;
	    if (_dec.nband == 1)
	    {
		fprintf (stderr, "Line %d: 'hfmatrix' definition in single band decoder.\n", line);
		stat = ERROR;
	    }
	    if (nhf)
	    {
		fprintf (stderr, "Line %d: repeated matrix definition.\n", line);
		stat = ERROR;
	    }
            else
	    {
		R = _hfmatrix;
		G = _hfgain;
		cnt = &nhf;
	    }
	}
        else if (!strcmp (p, "order_gain"))
	{
	    switch (_dec.h_ord) 
	    {
	    case 1:
	        if (sscanf (q, "%f%f%n\n", G + 0, G + 1, &n) != 2) stat = ARGS;
 	        else q += n;
		break;
	    case 2:
	        if (sscanf (q, "%f%f%f%n\n", G + 0, G + 1, G + 2, &n) != 3) stat = ARGS;
 	        else q += n;
		break;
	    case 3:
	        if (sscanf (q, "%f%f%f%f%n\n", G + 0, G + 1, G + 2, G + 3, &n) != 4) stat = ARGS;
 	        else q += n;
		break;
	    }
	}
        else if (!strcmp (p, "add_row"))
	{
	    if (!R) stat = SCOPE;
	    if (*cnt == _dec.nspkr)
	    {
		fprintf (stderr, "Line %d: too many matrix rows.\n", line);
		stat = ERROR;
	    }
            else
	    {
		stat = read_matrow (&q, R);
	        if (!stat)
	        {
		    R++;
		    (*cnt)++;
	        }
	    }		
	}
        else if (!strcmp (p, "/}"))
	{
	    if      (S) S = 0;
	    else if (R) R = 0;
	    else stat = SCOPE;
	}
        else if (!strcmp (p, "/end"))
	{
	    if (R || S) stat = SCOPE;
	}
        else stat = COMM;

        if (stat == EXTRA)
	{
            while (isspace (*q)) q++;
            if (*q < ' ') stat = 0;
	}		     

        switch (stat)
	{
        case COMM:
	    fprintf (stderr, "Line %d: unknown command '%s'\n", line, p);   
            break;
        case ARGS:
	    fprintf (stderr, "Line %d: missing arguments in '%s' command\n", line, p);   
            break;
        case EXTRA:
	    fprintf (stderr, "Line %d: extra arguments in '%s' command\n", line, p);   
            break;
        case SCOPE:
	    fprintf (stderr, "Line %d: command '%s' in wrong scope\n", line, p);   
            break;
	}
    }

    fclose (F);
    return stat;
}


int AD_conf::read_matrow (char **q, Matrow *R)
{
    int n, r;

    switch ((_dec.h_ord << 4) | _dec.v_ord)
    {
    case 0x10:
	r = sscanf (*q, "%f%f%f%n",
                    &R->_w, &R->_x, &R->_y, &n); 
	if (r != 3) return ARGS;
	break;
    case 0x11:
	r = sscanf (*q, "%f%f%f%f%n",
                    &R->_w, &R->_x, &R->_y, &R->_z, &n); 
	if (r != 4) return ARGS;
	break;
    case 0x20:
	r = sscanf (*q, "%f%f%f%f%f%n",
		    &R->_w, &R->_x, &R->_y, &R->_u, &R->_v, &n); 
	if (r != 5) return ARGS;
	break;
    case 0x21:
	r = sscanf (*q, "%f%f%f%f%f%f%n",
		    &R->_w, &R->_x, &R->_y, &R->_z, &R->_u, &R->_v, &n); 
	if (r != 6) return ARGS;
	break;
    case 0x22:
	r = sscanf (*q, "%f%f%f%f%f%f%f%f%f%n",
		    &R->_w, &R->_x, &R->_y, &R->_z, &R->_r, &R->_s, &R->_t, &R->_u, &R->_v, &n); 
	if (r != 9) return ARGS;
	break;
    case 0x30:
	r = sscanf (*q, "%f%f%f%f%f%f%f%n",
		    &R->_w, &R->_x, &R->_y, &R->_u, &R->_v, &R->_p, &R->_q, &n); 
	if (r != 7) return ARGS;
	break;
    case 0x31:
	r = sscanf (*q, "%f%f%f%f%f%f%f%f%n",
		    &R->_w, &R->_x, &R->_y, &R->_z, &R->_u, &R->_v, &R->_p, &R->_q, &n); 
	if (r != 8) return ARGS;
	break;
    }
    *q += n;
    return 0;
} 


void AD_conf::default_label (void)
{
    for (int i = 0; i < _dec.nspkr; i++) sprintf (_speakers [i]._label, "%d", i + 1);
}


void AD_conf::default_jackp (void)
{
    for (int i = 0; i < _dec.nspkr; i++) sprintf (_speakers [i]._jackp, "%s_%d", jackp, i + 1);
}

