/*
 * OSS compatible sequencer driver
 *
 * registration of device and proc
 *
 * Copyright (C) 1998,99 Takashi Iwai <tiwai@suse.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define SND_MAIN_OBJECT_FILE
#include "../../../include/driver.h"
#include "../../../include/minors.h"
#include "seq_oss_device.h"
#include "seq_oss_synth.h"

/*
 * module option
 */
#ifdef SND_SEQ_OSS_DEBUG
MODULE_PARM(seq_oss_debug, "i");
MODULE_PARM_DESC(seq_oss_debug, "debug option");
int seq_oss_debug = 0;
#endif


/*
 * prototypes
 */
static int register_device(void);
static void unregister_device(void);
static int register_proc(void);
static void unregister_proc(void);

static int dev_open(unsigned short minor, int cardnum, int device, struct file *file);
static int dev_release(unsigned short minor, int cardnum, int device, struct file *file);
static long dev_read(struct file *file, char *buf, long count);
static long dev_write(struct file *file, const char *buf, long count);
static int dev_ioctl(struct file *file, unsigned int cmd, unsigned long arg);
static unsigned int dev_poll(struct file *file, poll_table * wait);
#ifdef CONFIG_PROC_FS
static void info_read(snd_info_buffer_t *buf, void *private);
#endif


/*
 * module interface
 */

MODULE_AUTHOR("Takashi Iwai <tiwai@suse.de>");
MODULE_DESCRIPTION("OSS-compatible sequencer module");

static int __init alsa_seq_oss_init(void)
{
	int rc;
	static snd_seq_dev_ops_t ops = {
		snd_seq_oss_synth_register,
		snd_seq_oss_synth_unregister,
	};

	if ((rc = register_device()) < 0)
		return rc;
	if ((rc = register_proc()) < 0) {
		unregister_device();
		return rc;
	}
	if ((rc = snd_seq_oss_create_client()) < 0) {
		unregister_proc();
		unregister_device();
		return rc;
	}

	if ((rc = snd_seq_device_register_driver(SND_SEQ_DEV_ID_OSS, &ops,
						 sizeof(snd_seq_oss_reg_t))) < 0) {
		snd_seq_oss_delete_client();
		unregister_proc();
		unregister_device();
		return rc;
	}

	/* success */
	snd_seq_oss_synth_init();
	return 0;
}

static void __exit alsa_seq_oss_exit(void)
{
	snd_seq_device_unregister_driver(SND_SEQ_DEV_ID_OSS);
	snd_seq_oss_delete_client();
	unregister_proc();
	unregister_device();
}

module_init(alsa_seq_oss_init)
module_exit(alsa_seq_oss_exit)

/*
 * ALSA minor device interface
 */

static DECLARE_MUTEX(register_mutex);

static int
dev_open(unsigned short minor, int cardnum, int device, struct file *file)
{
	int level, rc;

	if (minor == SND_MINOR_OSS_MUSIC)
		level = SND_SEQ_OSS_MODE_MUSIC;
	else
		level = SND_SEQ_OSS_MODE_SYNTH;

	down(&register_mutex);
	rc = snd_seq_oss_open(file, level);
	up(&register_mutex);

	return rc;
}

static int
dev_release(unsigned short minor, int cardnum, int device, struct file *file)
{
	seq_oss_devinfo_t *dp;

	if ((dp = file->private_data) == NULL)
		return 0;

	snd_seq_oss_flush_write(dp);

	down(&register_mutex);
	snd_seq_oss_release(dp);
	up(&register_mutex);

	return 0;
}

static long
dev_read(struct file *file, char *buf, long count)
{
	seq_oss_devinfo_t *dp;
	dp = file->private_data;
	snd_debug_check(dp == NULL, -EIO);
	return snd_seq_oss_read(dp, buf, count);
}


static long
dev_write(struct file *file, const char *buf, long count)
{
	seq_oss_devinfo_t *dp;
	dp = file->private_data;
	snd_debug_check(dp == NULL, -EIO);
	return snd_seq_oss_write(dp, buf, count, file);
}

static int
dev_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	seq_oss_devinfo_t *dp;
	dp = file->private_data;
	snd_debug_check(dp == NULL, -EIO);
	return snd_seq_oss_ioctl(dp, cmd, arg);
}


static unsigned int
dev_poll(struct file *file, poll_table * wait)
{
	seq_oss_devinfo_t *dp;
	dp = file->private_data;
	snd_debug_check(dp == NULL, 0);
	return snd_seq_oss_poll(dp, file, wait);
}

/*
 * registration of sequencer minor device
 */

static snd_minor_t seq_oss_reg = {
	comment:	"sequencer",
	read:		dev_read,
	write:		dev_write,
	open:		dev_open,
	release:	dev_release,
	poll:		dev_poll,
	ioctl:		dev_ioctl,
};

static int __init
register_device(void)
{
	int rc;

	down(&register_mutex);
	if ((rc = snd_register_oss_device(SND_OSS_DEVICE_TYPE_SEQUENCER,
					  NULL, 0,
					  &seq_oss_reg,
					  SND_SEQ_OSS_DEVNAME)) < 0) {
		snd_printk("sequencer: can't register device seq\n");
		up(&register_mutex);
		return rc;
	}
	if ((rc = snd_register_oss_device(SND_OSS_DEVICE_TYPE_MUSIC,
					  NULL, 0,
					  &seq_oss_reg,
					  SND_SEQ_OSS_DEVNAME)) < 0) {
		snd_printk("sequencer: can't register device music\n");
		snd_unregister_oss_device(SND_OSS_DEVICE_TYPE_SEQUENCER, NULL, 0);
		up(&register_mutex);
		return rc;
	}
	debug_printk(("seq_oss: device registered\n"));
	up(&register_mutex);
	return 0;
}

static void
unregister_device(void)
{
	down(&register_mutex);
	debug_printk(("seq_oss: device unregistered\n"));
	if (snd_unregister_oss_device(SND_OSS_DEVICE_TYPE_MUSIC, NULL, 0) < 0)
		
		snd_printk("sequencer: error unregister device music\n");
	if (snd_unregister_oss_device(SND_OSS_DEVICE_TYPE_SEQUENCER, NULL, 0) < 0)
		snd_printk("sequencer: error unregister device seq\n");
	up(&register_mutex);
}

/*
 * /proc interface
 */

#ifdef CONFIG_PROC_FS

static snd_info_entry_t *info_entry;

static void
info_read(snd_info_buffer_t *buf, void *private)
{
	down(&register_mutex);
	snd_iprintf(buf, "OSS sequencer emulation version %s\n", SND_SEQ_OSS_VERSION_STR);
	snd_seq_oss_system_info_read(buf);
	snd_seq_oss_synth_info_read(buf);
	snd_seq_oss_midi_info_read(buf);
	up(&register_mutex);
}

#endif /* CONFIG_PROC_FS */

static int __init
register_proc(void)
{
#ifdef CONFIG_PROC_FS
	snd_info_entry_t *entry;

	entry = snd_info_create_entry(NULL, SND_SEQ_OSS_PROCNAME);
	if (entry == NULL)
		return -ENOMEM;

	entry->type = SND_INFO_ENTRY_TEXT;
	entry->subtype = SND_INFO_ENTRY_SEQUENCER;
	entry->private_data = NULL;
	entry->t.text.read_size = 1024;
	entry->t.text.read = info_read;
	if (snd_info_register(entry) < 0) {
		snd_info_free_entry(entry);
		return -ENOMEM;
	}
	info_entry = entry;
#endif
	return 0;
}

static void __exit
unregister_proc(void)
{
#ifdef CONFIG_PROC_FS
	if (info_entry)
		snd_info_unregister(info_entry);
	info_entry = NULL;
#endif
}

EXPORT_NO_SYMBOLS;
MODULE_LICENSE("GPL");
