/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Memory allocation routines.
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../include/driver.h"
#include "../include/info.h"

/*
 *  page allocation
 */

#ifdef CONFIG_SND_DEBUG_MEMORY
static long snd_alloc_pages;
static long snd_alloc_kmalloc;
static long snd_alloc_vmalloc;

static snd_info_entry_t *snd_memory_info_entry;
#endif /* CONFIG_SND_DEBUG_MEMORY */

void *snd_malloc_pages(unsigned long size, int *res_pg, int gfp_flags)
{
	int pg;
	void *buf;

	for (pg = 0; PAGE_SIZE * (1 << pg) < size; pg++);
	buf = (void *) __get_free_pages(gfp_flags, pg);
	if (buf != NULL) {
		mem_map_t *page = virt_to_page(buf);
		mem_map_t *last_page = virt_to_page(buf) + (1 << pg);
		while (page < last_page)
			set_bit(PG_reserved, &(page++)->flags);
#ifdef CONFIG_SND_DEBUG_MEMORY
		snd_alloc_pages += 1 << pg;
#endif
	}
	if (res_pg)
		*res_pg = pg;
	return buf;
}

void snd_free_pages(void *ptr, unsigned long size)
{
	int pg;
	mem_map_t *page, *last_page;

	if (ptr == NULL)
		return;
	for (pg = 0; PAGE_SIZE * (1 << pg) < size; pg++);
	page = virt_to_page(ptr);
	last_page = virt_to_page(ptr) + (1 << pg);
	while (page < last_page)
		clear_bit(PG_reserved, &(page++)->flags);
	free_pages((unsigned long) ptr, pg);
#ifdef CONFIG_SND_DEBUG_MEMORY
	snd_alloc_pages -= 1 << pg;
#endif
}

void snd_memory_init(void)
{
#ifdef CONFIG_SND_DEBUG_MEMORY
	snd_alloc_pages = 0;
	snd_alloc_kmalloc = 0;
	snd_alloc_vmalloc = 0;
#endif
}

void snd_memory_done(void)
{
#ifdef CONFIG_SND_DEBUG_MEMORY
	if (snd_alloc_pages > 0)
		snd_printk("snd_memory_done - snd_alloc_pages = %li\n", snd_alloc_pages);
	if (snd_alloc_kmalloc > 0)
		snd_printk("snd_memory_done - snd_alloc_kmalloc = %li\n", snd_alloc_kmalloc);
	if (snd_alloc_vmalloc > 0)
		snd_printk("snd_memory_done - snd_alloc_vmalloc = %li\n", snd_alloc_vmalloc);
#endif
}


#ifdef CONFIG_SND_DEBUG_MEMORY

typedef struct snd_malloc_head {
	unsigned long magic;
	size_t size;
	long data[0];
} snd_malloc_head_t;

#define KMALLOC_MAGIC 0x87654321

#define snd_malloc_entry(obj) (snd_malloc_head_t*)((char*)obj - (unsigned long)((snd_malloc_head_t*)0)->data)

void *snd_kmalloc(size_t size, int flags)
{
	snd_malloc_head_t *result;

	result = (snd_malloc_head_t *)kmalloc(size + sizeof(snd_malloc_head_t), flags);
	if (result) {
		result->size = size;
		result->magic = KMALLOC_MAGIC;
		snd_alloc_kmalloc += size;
		return result->data;
	}
	return NULL;
}

void _snd_kfree(const void *obj)
{
	snd_malloc_head_t *head;

	if (obj == NULL) {
		snd_printk("kfree NULL\n");
		return;
	}
	head = snd_malloc_entry(obj);
	if (head->magic != KMALLOC_MAGIC) {
		snd_printk("kfree bad\n");
		return;
	}
	snd_alloc_kmalloc -= head->size;
	kfree(head);
}

/*
 * malloc with magic
 */
void *_snd_magic_kcalloc(size_t size, int flags, unsigned int magic)
{
	unsigned long *ptr;
	ptr = snd_kcalloc(size + sizeof(unsigned long), flags);
	if (ptr == NULL)
		return NULL;
	*ptr = magic;
	return ptr + 1;
}

void *_snd_magic_kmalloc(size_t size, int flags, unsigned int magic)
{
	unsigned long *ptr;
	ptr = snd_kmalloc(size + sizeof(unsigned long), flags);
	if (ptr == NULL)
		return NULL;
	*ptr = magic;
	return ptr + 1;
}

void _snd_magic_kfree(void *_ptr)
{
	unsigned long *ptr = _ptr;
	if (ptr) {
		*--ptr = 0;
		_snd_kfree(ptr);
	}
	return;
}

#define VMALLOC_MAGIC 0x87654320

void *snd_vmalloc(unsigned long size)
{
	snd_malloc_head_t *result;

	result = (snd_malloc_head_t *)vmalloc(size + sizeof(snd_malloc_head_t));
	if (result) {
		result->size = size;
		result->magic = VMALLOC_MAGIC;
		snd_alloc_vmalloc += size;
		return result->data;
	}
	return NULL;
}

void snd_vfree(void *obj)
{
	snd_malloc_head_t *head;

	if (obj == NULL) {
		snd_printk("vfree NULL\n");
		return;
	}
	head = snd_malloc_entry(obj);
	if (head->magic != VMALLOC_MAGIC) {
		snd_printk("vfree bad\n");
		return;
	}
	snd_alloc_vmalloc -= head->size;
	vfree(head);
}

static void snd_memory_info_read(snd_info_buffer_t * buffer, void *private_data)
{
	snd_iprintf(buffer, "pages  : %li bytes (%li pages)\n", snd_alloc_pages * PAGE_SIZE, snd_alloc_pages);
	snd_iprintf(buffer, "kmalloc: %li bytes\n", snd_alloc_kmalloc);
	snd_iprintf(buffer, "vmalloc: %li bytes\n", snd_alloc_vmalloc);
}

int __init snd_memory_info_init(void)
{
	snd_info_entry_t *entry;

	entry = snd_info_create_entry(NULL, "meminfo");
	if (entry) {
		entry->t.text.read_size = 256;
		entry->t.text.read = snd_memory_info_read;
		if (snd_info_register(entry) < 0) {
			snd_info_free_entry(entry);
			entry = NULL;
		}
	}
	snd_memory_info_entry = entry;
	return 0;
}

int __exit snd_memory_info_done(void)
{
	if (snd_memory_info_entry)
		snd_info_unregister(snd_memory_info_entry);
	return 0;
}

#endif /* CONFIG_SND_DEBUG_MEMORY */


void *snd_kcalloc(size_t size, int flags)
{
	void *result;
	
	result = snd_kmalloc(size, flags);
	if (result != NULL)
		memset(result, 0, size);
	return result;
}

char *snd_kmalloc_strdup(const char *string, int flags)
{
	size_t len;
	char *res;

	if (!string)
		return NULL;
	len = strlen(string) + 1;
	if ((res = snd_kmalloc(len, flags)) == NULL)
		return NULL;
	memcpy(res, string, len);
	return res;
}


/*
 *  DMA allocation routines
 */

static int snd_dma_is_multi(snd_dma_t * dma, char * owner)
{
	if (!dma->multi)
		return 0;
	if (dma->multi_match[0] == NULL)
		return 1;
	if (dma->areas) {
		if (dma->areas->owner == NULL ||
		    strcmp(dma->areas->owner, owner))
			return 0;
	}
	if (!strcmp(dma->multi_match[0], owner) ||
	    (dma->multi_match[1] && !strcmp(dma->multi_match[1], owner)))
	    	return 1;
	return 0;
}

int snd_dma_malloc(snd_card_t * card, snd_dma_t * dma, char *owner, snd_dma_area_t ** rarea)
{
	unsigned char *buf;
	size_t size;
	int pg, add_area = 0;
	snd_dma_area_t *area;

	*rarea = NULL;
	snd_debug_check(card == NULL || dma == NULL, -ENXIO);
	snd_debug_check(dma->type == SND_DMA_TYPE_HARDWARE, -EINVAL);
	down(&dma->mutex);
	if (!snd_dma_is_multi(dma, owner) && dma->areas) {
		area = dma->areas;
		if (area->owner) {
			if (area->mmaped) {
				area->mmap_free = 0;
				up(&dma->mutex);
				return 0;
			} else {
				up(&dma->mutex);
				return -EAGAIN;
			}
		}
	} else {
		area = (snd_dma_area_t *)snd_kcalloc(sizeof(*area), GFP_KERNEL);
		if (area == NULL) {
			up(&dma->mutex);
			return -ENOMEM;
		}
		add_area = 1;
	}
	area->dma = dma;
	area->mmap_free = 0;
	buf = NULL;
	size = dma->rsize;
	pg = 0;
	while (size >= PAGE_SIZE) {
		int gfp_flags = GFP_KERNEL;
		if (dma->type == SND_DMA_TYPE_ISA ||
		    dma->type == SND_DMA_TYPE_PCI_16MB)
			gfp_flags |= GFP_DMA;
		buf = snd_malloc_pages(size, &pg, gfp_flags);
		if (! (gfp_flags & GFP_DMA) && buf &&
		    dma->addressbits < sizeof(unsigned long) * 8) {
			/* check the returned address matches the address mask */
			unsigned long addr = virt_to_bus(buf);
			addr >>= dma->addressbits;
			if (addr) {
				/* the area is over address mask.
				 * try low memory area now.
				 */
				snd_free_pages(buf, size);
				gfp_flags |= GFP_DMA;
				buf = snd_malloc_pages(size, &pg, gfp_flags);
			}
		}
		if (buf)
			break;
		size >>= 1;
	}
	if (buf) {
		area->size = (1 << pg) * PAGE_SIZE;
		area->buf = buf;
		area->owner = owner;
		if (add_area) {
			area->next = dma->areas;
			dma->areas = area;
		}
		*rarea = area;
	} else {
		if (add_area)
			_snd_kfree(area);
	}
	up(&dma->mutex);
	return buf == NULL ? -ENOMEM : 0;
}

static void snd_dma_free1(snd_card_t * card, snd_dma_t * dma, snd_dma_area_t * area)
{
	if (area->mmaped) {	/* wait */
		area->mmap_free = 1;
		return;
	}
	area->mmap_free = 0;
	area->owner = NULL;
	if (area->buf) {
#ifdef CONFIG_SND_ISA
		/* free DMA buffer */
		if (dma->type == SND_DMA_TYPE_ISA &&
		    dma->areas == area && area->next == NULL)
			snd_dma_disable(dma->dma);
#endif /* CONFIG_SND_ISA */
		snd_free_pages(area->buf, area->size);
		area->buf = NULL;
		area->size = 0;
		if (dma->areas == area) {
			dma->areas = area->next;
		} else {
			snd_dma_area_t *a = dma->areas;
			while (a->next && a->next != area)
				a = a->next;
			if (a->next == NULL) {
				snd_printk("snd_dma_free1: INTERNAL ERROR\n");
			} else {
				a->next = area->next;
			}
		}
		_snd_kfree(area);
	}
}

void snd_dma_free(snd_card_t * card, snd_dma_area_t * area)
{
	snd_dma_t *dma;

	if (card == NULL || area == NULL)
		return;
	dma = area->dma;
	if (dma == NULL)
		return;
	if (dma->type == SND_DMA_TYPE_HARDWARE)
		return;
	// snd_printk("snd_dma_free: dmanum = %i, mmaped = %i\n", dma->dma, area->mmaped);
	down(&dma->mutex);
	snd_dma_free1(card, dma, area);
	up(&dma->mutex);
}
