/*
 * OSS compatible sequencer driver
 *
 * Timer control routines
 *
 * Copyright (C) 1998,99 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "seq_oss_timer.h"
#include "seq_oss_event.h"
#include "../../../include/seq_oss_legacy.h"


/*
 */
#define DEFAULT_PPQ	96

/*
 */
static void timer_reset(seq_oss_timer_t *rec);

/*
 */
static DECLARE_MUTEX(register_mutex);
static int ntimers = 0;


/*
 * start queue (system timer)
 */
static void
timer_start(seq_oss_devinfo_t *dp)
{
	snd_seq_oss_start_timer(DEFAULT_PPQ, (DEFAULT_PPQ * 1000000L) / SND_SEQ_OSS_CTRLRATE);
}

/*
 * create and register a new timer.
 * if queue is not started yet, start it.
 */
seq_oss_timer_t *
snd_seq_oss_timer_new(seq_oss_devinfo_t *dp)
{
	seq_oss_timer_t *rec;

	rec = snd_kmalloc(sizeof(*rec), GFP_KERNEL);
	if (rec == NULL)
		return NULL;

	rec->dp = dp;
	snd_seq_oss_timer_init(rec);
	down(&register_mutex);
	ntimers++;
	if (ntimers == 1)
		timer_start(rec->dp);
	up(&register_mutex);

	return rec;
}


/*
 * delete timer.
 * if no more timer exists, stop the queue.
 */
void
snd_seq_oss_timer_delete(seq_oss_timer_t *rec)
{
	down(&register_mutex);
	snd_kfree(rec);
	ntimers--;
	if (ntimers == 0)
		snd_seq_oss_stop_timer();
	up(&register_mutex);
}


/*
 * get current tick of system timer
 */
abstime_t
snd_seq_oss_timer_sys_tick(seq_oss_devinfo_t *dp)
{
	snd_seq_queue_status_t status;
	memset(&status, 0, sizeof(status));
	status.queue = dp->queue;
	snd_seq_oss_control(dp, SND_SEQ_IOCTL_GET_QUEUE_STATUS, &status);
	if (status.running)
		return status.tick;
	else
		return 0;
}


/*
 * initialize timer values
 */
void
snd_seq_oss_timer_init(seq_oss_timer_t *rec)
{
	rec->cur_tick = 0;
	rec->tick_offset = snd_seq_oss_timer_sys_tick(rec->dp);
	rec->realtime = 0;
}


/*
 * reset timer offset
 */
static void
timer_reset(seq_oss_timer_t *rec)
{
	abstime_t curtick = rec->cur_tick + rec->tick_offset;
	abstime_t systick = snd_seq_oss_timer_sys_tick(rec->dp);
	if (curtick > systick)
		systick = curtick;
	rec->tick_offset = systick;
	rec->cur_tick = 0;
	rec->realtime = 0;
}


/*
 * process one timing event
 * return 1 : event proceseed -- skip this event
 *        0 : not a timer event -- enqueue this event
 */
int
snd_seq_oss_process_timer_event(seq_oss_timer_t *rec, evrec_t *ev)
{
	abstime_t parm = ev->t.time;

	if (ev->t.code == SND_OSS_EV_TIMING) {
		switch (ev->t.cmd) {
		case SND_OSS_TMR_WAIT_REL:
			parm += rec->cur_tick;
			rec->realtime = 0;
			/* continue to next */
		case SND_OSS_TMR_WAIT_ABS:
			if (parm == 0) {
				rec->realtime = 1;
			} else if (parm >= rec->cur_tick) {
				rec->realtime = 0;
				rec->cur_tick = parm;
			}
			return 1;	/* skip this event */
			
		case SND_OSS_TMR_START:
			timer_reset(rec);
			return 1;

		case SND_OSS_TMR_STOP:
		case SND_OSS_TMR_CONTINUE:
		case SND_OSS_TMR_TEMPO:
			return 1;
		}
	} else if (ev->s.code == SND_OSS_SEQ_SYNCTIMER) {
		/* reset timer */
		timer_reset(rec);
		return 1;

	} else if (ev->s.code == SND_OSS_SEQ_WAIT) {
		/* time = from 1 to 3 bytes */
		parm = (ev->echo >> 8) & 0xffffff;
		if (parm > rec->cur_tick) {
			/* set next event time */
			rec->cur_tick = parm;
			rec->realtime = 0;
		}
		return 1;
	}

	return 0;
}
