/*
 * OSS compatible sequencer driver
 *
 * synth device handlers
 *
 * Copyright (C) 1998,99 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "seq_oss_synth.h"

/*
 * constants
 */
#define SND_SEQ_OSS_MAX_SYNTH_NAME	30
#define MAX_SYSEX_BUFLEN	128


/*
 * definition of synth info records
 */

/* sysex buffer */
struct seq_oss_synth_sysex_t {
	int len;
	int skip;
	unsigned char buf[MAX_SYSEX_BUFLEN];
};

/* synth info */
struct seq_oss_synth_t {
	int seq_device;

	/* for synth_info */
	int synth_type;
	int synth_subtype;
	int nr_voices;

	char name[SND_SEQ_OSS_MAX_SYNTH_NAME];
	snd_seq_oss_callback_t oper;

	int opened;

	void *private;

	seq_oss_synth_sysex_t sysex;
};


/*
 * device table
 */
static int max_synth_devs = 0;
static seq_oss_synth_t *synth_devs[SND_SEQ_OSS_MAX_SYNTH_DEVS];

static DECLARE_MUTEX(register_mutex);

/*
 * prototypes
 */
static seq_oss_synth_t *get_synthdev(seq_oss_devinfo_t *dp, int dev);
static void reset_channels(seq_oss_synthinfo_t *info);


/*
 * registration of the synth device
 */
int
snd_seq_oss_synth_register(snd_card_t *card, int device, char *name, char *id,
			   void *arg, int size, snd_seq_dev_entry_t *result)
{
	int i;
	seq_oss_synth_t *rec;
	snd_seq_oss_reg_t *reg = arg;

	if ((rec = snd_kcalloc(sizeof(*rec), GFP_KERNEL)) == NULL) {
		snd_printk("sequencer: can't malloc synth info\n");
		return -ENOMEM;
	}
	rec->seq_device = -1;
	rec->synth_type = reg->type;
	rec->synth_subtype = reg->subtype;
	rec->nr_voices = reg->nvoices;
	rec->oper = reg->oper;
	rec->private = reg->private_data;
	rec->opened = 0;

	/* copy and truncate the name of synth device */
	strncpy(rec->name, name, sizeof(rec->name));
	rec->name[sizeof(rec->name)-1] = 0;

	/* registration */
	down(&register_mutex);
	for (i = 0; i < max_synth_devs; i++) {
		if (synth_devs[i] == NULL)
			break;
	}
	if (i >= max_synth_devs) {
		if (max_synth_devs >= SND_SEQ_OSS_MAX_SYNTH_DEVS) {
			snd_printk("sequencer: no more synth slot\n");
			up(&register_mutex);
			snd_kfree(rec);
			return -ENOMEM;
		}
		max_synth_devs++;
	}
	rec->seq_device = i;
	synth_devs[i] = rec;
	debug_printk(("seq_oss: synth %s registered %d\n", rec->name, i));
	up(&register_mutex);
	*result = rec;
	return 0;
}


int
snd_seq_oss_synth_unregister(snd_card_t *card, int device, void *arg,
			     int size, snd_seq_dev_entry_t entry)
{
	int index;
	seq_oss_synth_t *rec = (seq_oss_synth_t*)entry;

	down(&register_mutex);
	for (index = 0; index < max_synth_devs; index++) {
		if (synth_devs[index] == rec)
			break;
	}
	if (index >= max_synth_devs) {
		snd_printk("sequencer: can't unregister synth\n");
		up(&register_mutex);
		return -EINVAL;
	}

	snd_kfree(rec);
	synth_devs[index] = NULL;

	if (index == max_synth_devs - 1) {
		for (index--; index >= 0; index--) {
			if (synth_devs[index])
				break;
		}
		max_synth_devs = index + 1;
	}
	up(&register_mutex);
	return 0;
}


/*
 * set up synth tables
 */

void
snd_seq_oss_synth_setup(seq_oss_devinfo_t *dp)
{
	int i;
	seq_oss_synth_t *rec;
	seq_oss_synthinfo_t *info;

	down(&register_mutex);
	dp->max_synthdev = max_synth_devs;
	dp->synth_opened = 0;
	memset(dp->synths, 0, sizeof(dp->synths));
	for (i = 0; i < dp->max_synthdev; i++) {
		if ((rec = synth_devs[i]) == NULL)
			continue;
		if (! rec->oper.open || ! rec->oper.close)
			continue;
		info = &dp->synths[i];
		info->arg.app_index = dp->port;
		info->arg.file_mode = dp->file_mode;
		info->arg.seq_mode = dp->seq_mode;
		info->arg.event_passing = SND_SEQ_OSS_PROCESS_EVENTS;
		info->opened = 0;
		if (rec->oper.open(&info->arg, rec->private) < 0)
			continue;
		info->nr_voices = rec->nr_voices;
		if (info->nr_voices > 0) {
			info->ch = snd_kmalloc(sizeof(seq_oss_chinfo_t) * info->nr_voices, GFP_KERNEL);
			reset_channels(info);
		}
		debug_printk(("seq_oss: synth %d assigned\n", i));
		info->opened++;
		rec->opened++;

		/* reset sysex buffer */
		rec->sysex.len = 0;
		rec->sysex.skip = 0;

		dp->synth_opened++;
	}
	up(&register_mutex);
}


void
snd_seq_oss_synth_cleanup(seq_oss_devinfo_t *dp)
{
	int i;
	seq_oss_synth_t *rec;
	seq_oss_synthinfo_t *info;

	down(&register_mutex);
	for (i = 0; i < dp->max_synthdev; i++) {
		info = &dp->synths[i];
		if (! info->opened || (rec = synth_devs[i]) == NULL ||
		    ! rec->opened)
			continue;
		debug_printk(("seq_oss: synth %d closed\n", i));
		rec->oper.close(&info->arg);
		if (info->ch)
			snd_kfree(info->ch);
		rec->opened--;
	}
	up(&register_mutex);
	dp->synth_opened = 0;
	dp->max_synthdev = 0;
}

/*
 * return synth device information pointer
 */
static seq_oss_synth_t *
get_synthdev(seq_oss_devinfo_t *dp, int dev)
{
	seq_oss_synth_t *rec;
	if (dev < 0 || dev >= dp->max_synthdev)
		return NULL;
	if (! dp->synths[dev].opened || (rec = synth_devs[dev]) == NULL)
		return NULL;
	if (! rec->opened)
		return NULL;
	return rec;
}


/*
 * reset note and velocity on each channel.
 */
static void
reset_channels(seq_oss_synthinfo_t *info)
{
	int i;
	if (! info->ch || ! info->nr_voices)
		return;
	for (i = 0; i < info->nr_voices; i++) {
		info->ch[i].note = -1;
		info->ch[i].vel = 0;
	}
}


/*
 * reset synth device:
 * call reset callback.  if no callback is defined, send a heartbeat
 * event to the corresponding port.
 */
void
snd_seq_oss_synth_reset(seq_oss_devinfo_t *dp, int dev)
{
	seq_oss_synth_t *rec;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return;
	if (rec->oper.reset) {
		rec->oper.reset(&dp->synths[dev].arg);
	} else {
		snd_seq_event_t ev;
		memset(&ev, 0, sizeof(ev));
		snd_seq_oss_fill_addr(dp, &ev, dp->synths[dev].arg.addr.client,
				     dp->synths[dev].arg.addr.port);
		ev.type = SND_SEQ_EVENT_HEARTBEAT;
		snd_seq_oss_dispatch(dp, &ev, 0, 0);
	}
	reset_channels(&dp->synths[dev]);
}


/*
 * load a patch record:
 * call load_patch callback function
 */
int
snd_seq_oss_synth_load_patch(seq_oss_devinfo_t *dp, int dev, int fmt,
			    const char *buf, int p, int c)
{
	seq_oss_synth_t *rec;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return -ENXIO;

	if (! rec->oper.load_patch)
		return -ENXIO;

	return rec->oper.load_patch(&dp->synths[dev].arg, fmt, buf, p, c);
}

/*
 * check if the device is valid synth device
 */
int
snd_seq_oss_synth_is_valid(seq_oss_devinfo_t *dp, int dev)
{
	if (get_synthdev(dp, dev))
		return 1;
	return 0;
}


/*
 * receive OSS 6 byte sysex packet:
 * the full sysex message will be sent if it reaches to the end of data
 * (0xff).
 */
int
snd_seq_oss_synth_sysex(seq_oss_devinfo_t *dp, int dev, unsigned char *buf, snd_seq_event_t *ev)
{
	int i, send;
	unsigned char *dest;
	seq_oss_synth_t *rec;
	seq_oss_synth_sysex_t *sysex;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return -ENXIO;

	sysex = &rec->sysex;

	send = 0;
	dest = sysex->buf + sysex->len;
	/* copy 6 byte packet to the buffer */
	for (i = 0; i < 6; i++) {
		if (buf[i] == 0xff) {
			send = 1;
			break;
		}
		dest[i] = buf[i];
		sysex->len++;
		if (sysex->len >= MAX_SYSEX_BUFLEN) {
			sysex->len = 0;
			sysex->skip = 1;
			break;
		}
	}

	if (sysex->len && send) {
		if (sysex->skip) {
			sysex->skip = 0;
			sysex->len = 0;
			return -EINVAL; /* skip */
		}
		/* copy the data to event record and send it */
		ev->flags = SND_SEQ_EVENT_LENGTH_VARIABLE;
		if (snd_seq_oss_synth_addr(dp, dev, ev))
			return -EINVAL;
		ev->data.ext.len = sysex->len;
		ev->data.ext.ptr = snd_seq_ext_malloc(sysex->len, 0);
		sysex->len = 0;
		if (ev->data.ext.ptr == NULL)
			return -ENOMEM;
		memcpy(ev->data.ext.ptr, sysex->buf, sysex->len);
		return 0;
	}

	return -EINVAL; /* skip */
}

/*
 * fill the event source/destination addresses
 */
int
snd_seq_oss_synth_addr(seq_oss_devinfo_t *dp, int dev, snd_seq_event_t *ev)
{
	seq_oss_synth_t *rec;
	if ((rec = get_synthdev(dp, dev)) == NULL)
		return -EINVAL;
	snd_seq_oss_fill_addr(dp, ev, dp->synths[dev].arg.addr.client,
			      dp->synths[dev].arg.addr.port);
	return 0;
}


/*
 * OSS compatible ioctl
 */
int
snd_seq_oss_synth_ioctl(seq_oss_devinfo_t *dp, int dev, unsigned int cmd, unsigned long addr)
{
	seq_oss_synth_t *rec;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return -ENXIO;
	if (! rec->oper.ioctl)
		return -ENXIO;
	return rec->oper.ioctl(&dp->synths[dev].arg, cmd, addr);
}


/*
 * send OSS raw events - SEQ_PRIVATE and SEQ_VOLUME
 */
int
snd_seq_oss_synth_raw_event(seq_oss_devinfo_t *dp, int dev, unsigned char *data, snd_seq_event_t *ev)
{
	seq_oss_synth_t *rec;
	int rc;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return -ENXIO;
	ev->type = SND_SEQ_EVENT_OSS;
	memcpy(ev->data.raw8.d, data, 8);
	rc = snd_seq_oss_synth_addr(dp, dev, ev);
	return rc;
}


/*
 * create OSS compatible synth_info record
 */
int
snd_seq_oss_synth_make_info(seq_oss_devinfo_t *dp, int dev, oss_synth_info_t *inf)
{
	seq_oss_synth_t *rec;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return -ENXIO;
	inf->synth_type = rec->synth_type;
	inf->synth_subtype = rec->synth_subtype;
	inf->nr_voices = rec->nr_voices;
	inf->device = dev;
	strncpy(inf->name, rec->name, sizeof(inf->name));
	
	return 0;
}

/*
 * proc interface
 */
void
snd_seq_oss_synth_info_read(snd_info_buffer_t *buf)
{
	int i;
	seq_oss_synth_t *rec;

	down(&register_mutex);
	snd_iprintf(buf, "\nNumber of synth devices: %d\n", max_synth_devs);
	for (i = 0; i < max_synth_devs; i++) {
		snd_iprintf(buf, "\nsynth %d: ", i);
		if ((rec = synth_devs[i]) == NULL) {
			snd_iprintf(buf, "*empty*\n");
			continue;
		}
		snd_iprintf(buf, "%s\n", rec->name);
		snd_iprintf(buf, "  type 0x%x\n", rec->synth_type);
		snd_iprintf(buf, "  subtype 0x%x\n", rec->synth_subtype);
		snd_iprintf(buf, "  voices %d\n", rec->nr_voices);
		snd_iprintf(buf, "  ioctl %s\n", enabled_str((long)rec->oper.ioctl));
		snd_iprintf(buf, "  load_patch %s\n", enabled_str((long)rec->oper.load_patch));
	}
	up(&register_mutex);
}

