/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <io.h>
#include <dir.h>
#include <dos.h>
#include <sys/stat.h>

#define DIR_SEPARATOR "\\"

// P R O T O T Y P E S
//////////////////////////////////////////////////////

void GetPreviousDir(char* path);
void AddAllWildcard(char* path);
void GetIndividualFile(char* dest, char* path, char* file);
int  xcopy_file(const char *src_filename, const char *dest_filename);
char *addsep(char *);
int dir_exists(const char *);
void SplitPath(const char* path, char* drive, char* dir, char* fname, char* ext);
char *strmcpy(char *dest, const char *src, const unsigned int maxlen);
char *strmcat(char *dest, const char *src, const unsigned int maxlen);
void error(const char *);
int copy_file(const char *src_filename, const char *dest_filename);
void build_filename(char *, const char *, const char *);
int makedir(char *);

/*-------------------------------------------------------------------------*/
/* Deletes a directory tree.                            		   */
/*-------------------------------------------------------------------------*/

int DelTree(const char* path)
{
    int found;
    char temppath[MAXPATH];
    char temppath1[MAXPATH];
    char origpath[MAXPATH];
    struct ffblk info;
    
    strcpy(temppath, path);
    strcpy(origpath, path);
    GetPreviousDir(origpath);
    
    /* Two state machine: */    
    
    /* As long as the complete tree is not deleted */
    AddAllWildcard(origpath);
    AddAllWildcard(temppath);
    while (strcmp(temppath, origpath) != 0)
    {
          /* 1) as long as there are still sub directories, enter the first one */
	  found = findfirst(temppath, &info, FA_DIREC) == 0;
	  while (found &&
		 (info.ff_name[0] == '.' ||
		  (info.ff_attrib & FA_DIREC) == 0))
	  {
	     found = findnext(&info) == 0;
	  }

	  if (found)
	  {
	     GetIndividualFile(temppath1, temppath, info.ff_name);
	     strcpy(temppath, temppath1);
	     AddAllWildcard(temppath);
	  }

          /* 2) if there are no more sub directories in this directory, delete all the files 
                and leave this directory, and delete that directory */
	  else
          {
	      found = findfirst(temppath, &info, FA_RDONLY|FA_HIDDEN|FA_SYSTEM|FA_ARCH) == 0;
	      while (found)
	      {
		  GetIndividualFile(temppath1, temppath, info.ff_name);
		  if (chmod(temppath1, S_IREAD|S_IWRITE) == -1)
		      return 0;
	      
		  if (unlink(temppath1) == -1)
		      return 0;
	      
		  found = findnext(&info) == 0;
	      }
	
	      GetPreviousDir(temppath);	/* Strip *.* */
	
	      if (chmod(temppath, S_IREAD|S_IWRITE) == -1)
	         return 0;
	
	      if (rmdir(temppath) == -1)
	         return 0;
	    
	      GetPreviousDir(temppath);       /* Leave directory */
	      AddAllWildcard(temppath);
	}
    }
    
    return 1;
}

/*-------------------------------------------------------------------------*/
/* Strips the last file from the path.                       		   */
/*-------------------------------------------------------------------------*/

void GetPreviousDir(char* path)
{
    char drive[MAXDRIVE], dir[MAXDIR], fname[MAXFILE], ext[MAXEXT];

    if (path[strlen(path)-1] == '\\')
       path[strlen(path)-1] = '\0';

    SplitPath(path, drive, dir, fname, ext);
    
    strcpy(path, drive);
    strcat(path, dir);
}

/*-------------------------------------------------------------------------*/
/* Adds a *.* wildcard to a path                           		   */
/*-------------------------------------------------------------------------*/

void AddAllWildcard(char* path)
{
    addsep(path);
    strcat(path, "*.*");
}

/*-------------------------------------------------------------------------*/
/* Gets the path name, returned by a search in the wildcard enabled path   */
/*-------------------------------------------------------------------------*/

void GetIndividualFile(char* dest, char* path, char* file)
{
    strcpy(dest, path);
    GetPreviousDir(dest);
    addsep(dest);
    strcat(dest, file);
}

/*-------------------------------------------------------------------------*/
/* Searchs through the source directory (and its subdirectories) and calls */
/* function "xcopy_file" for every found file.                             */
/*-------------------------------------------------------------------------*/
int CopyTree(const char *src_pathname,
             const char *src_filename,
             const char *dest_pathname,
             const char *dest_filename) 
{
  char filepattern[MAXPATH],
       new_src_pathname[MAXPATH],
       new_dest_pathname[MAXPATH],
       src_path_filename[MAXPATH],
       dest_path_filename[MAXPATH],
       tmp_filename[MAXFILE + MAXEXT],
       tmp_pathname[MAXPATH];
  struct ffblk fileblock;
  int fileattrib,
      done;

  /* copy files in subdirectories  */
  strmcpy(filepattern, src_pathname, sizeof(filepattern));
  strmcat(filepattern, src_filename, sizeof(filepattern));
  done = findfirst(filepattern, &fileblock, FA_DIREC);
  while (!done) 
  {
    if (fileblock.ff_attrib == FA_DIREC &&
        strcmp(fileblock.ff_name, ".") != 0 &&
        strcmp(fileblock.ff_name, "..") != 0) 
    {
        /* build source pathname */
        strmcpy(new_src_pathname, src_pathname, sizeof(new_src_pathname));
        strmcat(new_src_pathname, fileblock.ff_name, sizeof(new_src_pathname));
        strmcat(new_src_pathname, DIR_SEPARATOR, sizeof(new_src_pathname));

        /* build destination pathname */
        strmcpy(new_dest_pathname, dest_pathname, sizeof(new_dest_pathname));
        strmcat(new_dest_pathname, fileblock.ff_name, sizeof(new_dest_pathname));
        strmcat(new_dest_pathname, DIR_SEPARATOR, sizeof(new_dest_pathname));

	CopyTree(new_src_pathname, "*.*",
		 new_dest_pathname, "*.*");
    }

    done = findnext(&fileblock);
  }

  fileattrib = FA_RDONLY+FA_ARCH+FA_HIDDEN+FA_SYSTEM;

  /* find first source file */
  strmcpy(filepattern, src_pathname, sizeof(filepattern));
  strmcat(filepattern, src_filename, sizeof(filepattern));
  done = findfirst(filepattern, &fileblock, fileattrib);

  /* check if destination directory must be created */
  if (!dir_exists(dest_pathname))
  {
    strmcpy(tmp_pathname, dest_pathname, sizeof(tmp_pathname));
    if (makedir(tmp_pathname) != 0) 
    {
      error("Unable to create directory");
      return 0;
    }
  }

  /* copy files */
  while (!done) 
  {
      /* build source filename including path */
      strmcpy(src_path_filename, src_pathname, sizeof(src_path_filename));
      strmcat(src_path_filename, fileblock.ff_name, sizeof(src_path_filename));

      /* build destination filename including path */
      strmcpy(dest_path_filename, dest_pathname, sizeof(dest_path_filename));
      build_filename(tmp_filename, fileblock.ff_name, dest_filename);
      strmcat(dest_path_filename, tmp_filename, sizeof(dest_path_filename));

      if (!xcopy_file(src_path_filename, dest_path_filename))
	  return 0;

      done = findnext(&fileblock);
  }
  
  return 1;
}

/*-------------------------------------------------------------------------*/
/* Checks all dependencies of the source and destination file and calls    */
/* function "copy_file".                                                   */
/*-------------------------------------------------------------------------*/
int xcopy_file(const char *src_filename,
                const char *dest_filename) 
{
  struct stat src_statbuf;
  struct dfree disktable;
  unsigned long free_diskspace;
  unsigned char dest_drive;

  /* Get the destination drive */
  if (dest_filename[1] == ':')
      dest_drive = dest_filename[0];
  else
      dest_drive = getdisk() + 'A' - 1;
  
  
  /* get info of source and destination file */
  stat((char *)src_filename, &src_statbuf);

  /* get amount of free disk space in destination drive */
  getdfree(dest_drive-'A'+1, &disktable);
  free_diskspace = (unsigned long) disktable.df_avail *
                   disktable.df_sclus * disktable.df_bsec;

  /* check free space on destination disk */
  if (src_statbuf.st_size > free_diskspace) 
  {
      error("Insufficient disk space in destination path");
      return(0);
  }

  /* Copy file data */
  return copy_file(src_filename, dest_filename);
}



/*-------------------------------------------------------------------------*/
/* Moves a directory from one place to an other                            */
/*-------------------------------------------------------------------------*/

int MoveDirectory(const char* src_filename, const char* dest_filename)
{
    char src_path[MAXPATH], src_file[MAXPATH];
    char dest_path[MAXPATH], dest_file[MAXPATH];
    
    char drive[MAXDRIVE], dir[MAXDIR], fname[MAXFILE], ext[MAXEXT];
    
    SplitPath(src_filename, drive, dir, fname, ext);
    
    strcpy(src_path, drive);
    strcat(src_path, dir);
    
    strcpy(src_file, fname);
    strcat(src_file, ext);
    
    SplitPath(dest_filename, drive, dir, fname, ext);
    
    strcpy(dest_path, drive);
    strcat(dest_path, dir);    
    
    strcpy(dest_file, fname);
    strcat(dest_file, ext);    
    
    if (dir_exists(dest_filename))
	if (!DelTree(dest_filename))
	    return 0;
    
    if (!CopyTree(src_path, src_file, dest_path, dest_file))
    {
	DelTree(dest_filename);	
	return 0;
    }
    
    return DelTree(src_filename);    
}
