// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _Bookmarks_h_
#define _Bookmarks_h_

extern "C"
{
#include <stdio.h>
}
#include <qlist.h>
#include <qobject.h>
#include <qstack.h>
#include "SgmlParser.h"
#include "Url.h"

class Bookmarks;
class Browser;
class QPopupMenu;

//: The Link class stores a single entry in the heirarchical list of bookmarks.
//. Every link has a title associated with it.  However each link may have
//. either a URL or a list of sublinks, but not both.  If the link has a list
//. of sublinks then it represents a submenu in the bookmarks menu structure.
//. Otherwise it represents a menu item that when selected will open the URL
//. in the active browser window.
class Link : public QObject {
    Q_OBJECT
    Bookmarks*  _bookmarks;
    QString     _name;
    QString     _url;
    QList<Link> _links;
public:
    //. Create a new link associated with the given bookmarks object.
    Link( Bookmarks* bookmarks );

    //. Destroy the link.
    ~Link();

    //. Return a newly constructed popup menu structure base on this link and
    //. all of its sublinks.  The popup menu is owned by the caller.
    QPopupMenu*  menu();

    //. Return my title.
    QString      name() { return _name; }

    //. Return the list of sublinks.
    QList<Link>& links() { return _links; }

    //. Set the title for this link.
    void         setName( QString name ) { _name = name.copy(); }

    //. Set the URL for this link.
    void         setUrl( QString url ) { _url = url.copy(); }

    //. Add a new sublink to this link.
    void         appendLink( Link* link ) { _links.append( link ); }
    
    //. Write out the information for this link and its sublinks to the given
    //. open file pointer.
    void         save( FILE* fptr, int indent );
public slots:
    //. Process the menu selection call back for this link.  Ask the bookmarks
    //. object to open the link in the active browser.
    void selected();
};

//: The Bookmarks class maintains the user's bookmarks, and the menu structure for them.
//. There is exactly one instance of the Bookmarks object, whcih is shared among all of
//. the open browser windows.  The bookmarks object creates and maintains a QPopupMenu
//. containing all of the bookmarks.  This menu is used by all of the browser windows.
//. <P>
//. Whenever a browser becomes "active" (i.e. gets focus) it must register with this
//. object, so that it can be notified when the user selects a bookmark url.
class Bookmarks : public QObject {
    Q_OBJECT
    Link*        _link;
    QStack<Link> _stack;
    Browser*     _active;
    SgmlParser*  _parser;
    QString      _text;
public:
    //. Initialize the global Bookmarks object.  Open the bookmarks file that
    //. is given in the user preferences, and begin parsing it.
    Bookmarks();

    //. Free the menu structure that we created (TBD).
    ~Bookmarks();

    void read();

    //. Return a pointer to the Bookmarks menu structure that we created.
    //. <B>NOTE:</B> the Bookmarks object retains ownership of the menu!
    QPopupMenu* menu();

    //. This function registers a Browser window for future bookmark menu
    //. selection events.  When a menu item is selected, it will be opened
    //. in this browser window.
    void activeBrowser( Browser* browser );

    Browser* active() { return _active; }
public slots:
    void dtdSelected( QString dtdName );

    //. Process a start tag signal from our SgmlParser.  We only recognize the
    //. allowed elements in the bookmarks DTD (BOOKMARKS, LINK, LASTMOD).
    void startTag10();
    void startTag11();

    //. Process an end tag signal from our SgmlParser.
    void endTag10();
    void endTag11();

    //. Process a content signal from our SgmlParser.  The only content that we
    //. look for is the menu item text for a bookmark entry.
    void content10( QString text );
    void content11( QString text );

    void addBookmark();

    void save();

    void updateBookmarksUrl();
signals:
    void bookmarksChanged();
};

extern Bookmarks* bookmarks;

#endif
