/*
 * Back-end main header
 * This is the most important file for you to understand the back-end.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifndef __GDIFF_GDIFF_H__
#define __GDIFF_GDIFF_H__

#include <time.h> /* for time_t */

#ifdef __cplusplus 
extern "C" {
#endif /* __cplusplus */

/*
 * Description of data structures
 *
 * DiffDir is allocated for one session of comparing(diff) two files
 * or two directories.
 * DiffDir keeps the linked list of DiffFiles.
 * Each node(DiffFiles) is allocated for two different files under the specified directories.
 * The DiffFiles has two files information, and the linked list of DiffLines.
 * Each node(DiffLines) is allocated for different part(between lines),
 * and has the type(DiffType) and the line numbers of each different part.
 */

/* Forward declarations of types. */
typedef struct _DiffDir DiffDir;
typedef struct _DiffFiles DiffFiles;
typedef struct _FileInfo FileInfo;
typedef struct _DiffLines DiffLines;
typedef struct _DiffBeginEnd DiffBeginEnd;

/*
 * Contant number
 */
/* IMPORTANT. These numbers are used in DiffLines structure.
 * Basically, they represent type for each differences.
 * In addition, the return value from parse_diff_line() also use these numbers. */
typedef enum { 
	F1ONLY,	/* Lines taken from just the first file. (known as DELETE) */
	F2ONLY,	/* Lines taken from just the second file. (known as ADD) */
	CHANGE,
	/* The followings are used for the return value from parse_diff_line() */
	FINDFILE,
	FINDBINFILE,
	IGNORE,
	ERROR,
} DiffType; 


#define NUM_COMPARE_FILES		2
typedef enum {
	FIRST_FILE = 0,
	SECOND_FILE = 1,
} WhichFile;

/* Status of two files.
   I'm afraid this might be confused with DiffType above.
   This is related to the compared two files(directories).
   On the other hand, DiffType exists for each different parts(lines).
   The reason why they are shifted bit is to use it as row_hide_stat_mask
   of DViewPref(see properties.h). */
typedef enum {
	BINARY_FILES		= (1<<0),
	ONLY_FILE1_EXISTS	= (1<<1),
	ONLY_FILE2_EXISTS	= (1<<2),
	DIFFERENT_FILES		= (1<<3),
	IDENTICAL_FILES		= (1<<4),
	DIRECTORIES			= (1<<5),
} FilesSpecialStatus;

typedef enum {
	MOD_MODIFIED,
	MOD_NOMODIFIED,
	MOD_DONTCARE,
	MOD_ERROR,
} FileModStatue;

	
/*
 * Data structure definitions
 */

/* Directory information data
 * Note: Comparing two files is dealt with as one special case of comparing two directories.
 * - When comparing directories, the first node has the directories info,
 *   and the following nodes has each files info.
 * - When comparing two files, the first node has the files info. */
struct _DiffDir {
	GSList *dfiles_list;	/* Singly linked list of DiffFiles */
};


/* Compared files information data
 * Some special cases as follows,
 * - dlines_list==NULL implies two files are identical.
 * - One of fileinfo[]'s fname==NULL implies that the file exists only in one directory.
 *
 * struct _FileInfo
 * Two possible cases,
 * 1)directory info. 2)file info. */
struct _DiffFiles {
	struct _FileInfo {
		char *fname;	/* file name */
		int nline;		/* Number of lines */
		char *text;		/* (mmap'd)buffer */
		int lenb;		/* Length(Byte) */
		gboolean f_dir;	/* TRUE if directory */
		time_t mtime;/* time of last modification */
		time_t ctime;/* time of last change */
	} fileinfo[NUM_COMPARE_FILES];
	GList *dlines_list;/* Doubly linked list of DiffLines */
	gboolean binary;
};


/* Diff lines information data
 * Keeps difftype and the line numbers. 
 * Different between these lines. */
struct _DiffLines {
	DiffType difftype;
	struct _DiffBeginEnd {
		/* the followings are line number. */
		int begin;
		int end;
	} between[NUM_COMPARE_FILES];
};


/* Global variable declarations */
/* None */

/* Global function declarations */
/* diffdata.c */
extern DiffDir* diff_dir_new(const char *fname1, const char *fname2, const char *args);
extern void diff_dir_delete(DiffDir *diffdir);
extern DiffFiles* diff_dir_add_files(DiffDir *diffdir, const char *fname1, const char *fname2);
extern void diff_dir_remove_files(DiffDir *diffdir, DiffFiles *dfiles);
extern const FileInfo* dfiles_get_fileinfo(DiffFiles *dfiles, WhichFile n, gboolean f_mmap);
extern void dfiles_add_dlines(DiffFiles* dfiles, DiffType dtype, int f1n1, int f1n2, int f2n1, int f2n2);

/* rundiff.c */
extern void run_diff(DiffDir *diffdirr, const char *filename1, const char *filename2, const char *args, DiffFiles *cur_files);

/* diffmisc.c */
extern FilesSpecialStatus dfiles_get_special_status(DiffFiles *dfiles);
extern FileModStatue dfiles_has_file_modified(DiffFiles *dfiles);
extern int dfiles_get_merged_nlines(DiffFiles *files);
extern int dfiles_calc_total_nlines(const DiffFiles *dfiles, WhichFile n);
extern int dfiles_get_max_nlines(DiffFiles *dfiles);
extern const GList* dfiles_get_firstl(const DiffFiles *dfiles, const GList *cur_dlines_list);
extern const GList* dfiles_get_lastl(const DiffFiles *dfiles, const GList *cur_dlines_list);
extern const GList* dfiles_find_nextl(const DiffFiles *dfiles, const GList *cur_dlines_list);
extern const GList* dfiles_find_prevl(const DiffFiles *dfiles, const GList *cur_dlines_list);
extern const DiffLines* dfiles_find_rel_nextl(const DiffFiles *dfiles, WhichFile n, int cur_line);
extern const DiffLines* dfiles_find_rel_prevl(const DiffFiles *dfiles, WhichFile n, int cur_line);
extern const GList* dfiles_get_firstl_in_merge(const DiffFiles *dfiles, const GList *cur_dlines_list, int *sum_f2);
extern const GList* dfiles_get_lastl_in_merge(const DiffFiles *dfiles, const GList *cur_dlines_list, int *sum_f2);
extern const GList* dfiles_find_nextl_in_merge(const DiffFiles *dfiles, const GList *cur_dlines_list, int *sum_f2);
extern const GList* dfiles_find_prevl_in_merge(const DiffFiles *dfiles, const GList *cur_dlines_list, int *sum_f2);
extern const DiffLines* dfiles_find_rel_nextl_in_merge(const DiffFiles *dfiles, int cur_line, int *sum_f2);
extern const DiffLines* dfiles_find_rel_prevl_in_merge(const DiffFiles *dfiles, int cur_line, int *sum_f2);

#ifdef __cplusplus
} 
#endif /* __cplusplus */
 
#endif /* __GDIFF_GDIFF_H__ */
