// header.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "application.h"
#include "comment.h"
#include "datafile.h"
#include "sound.h"
#include "header.h"
#include "typeconvert.h"

Header::Header(int magic, DataType type, int chans)
	: magic_number(magic), data_offset(0), data_size(0),
		data_type(type), nchans(chans), comment(nil),
		forceSwapped(false), raw(false) {}

Header::~Header() { Resource::unref(comment); }

boolean
Header::configure(Controller *c) {
	boolean status = false;
	Requester* requester = configRequester();
	status = (requester != nil && requester->queryForValues(c));
	delete requester;
	return status;
}

void
Header::initialize() {
	magic_number = 0;
	data_offset = 0;
	data_type = NoData;
	data_size = 0;
	nchans = 0;
}

void
Header::setComment(const char *com) {
	setComment(new Comment(com));
}

void
Header::setComment(Comment *c) {
	if(c != comment) {
		Resource::unref(comment);
		comment = c;
		if(comment != nil) {
			comment->ref();
		}
	}
}

int
Header::commentLength() {
	return (comment != nil) ? comment->len() : 0;
}

// this routine reads the magic number from the file and then checks it for
// validity.  If raw file reading is enabled, it fails silently so that the
// user may be queried for raw file reading options

int
Header::checkMagic(DataFile* file) {
	boolean status = true;
	if ((status = readMagic(file)) == true) {
		if (isMagic()) {
			file->setBytesSwapped(magicIsSwapped());
		}
		else {
			if (!Application::globalResourceIsTrue("ReadRawFiles"))
				Application::alert(magicError());
			status = false;
		}
	}
	return status;

}

// static public method

int
Header::readMagicNumber(DataFile *file) {
	int magic = -1;
	file->read((void *) &magic, sizeof(magic));
	file->clear();		// no file failure here because this is used to check
	file->seek(0);		// files that may not yet exist
#ifdef sgi
    // special trick to get constant value for mpeg header
	if (int(magic & MPEG1_MASK) == MPEG1_MAGIC)
	    magic &= MPEG1_MASK; 
#endif
	return magic;
}

int
Header::readMagic(DataFile *file) {
	if((magic_number = Header::readMagicNumber(file)) == -1) {
		Application::alert("Header::readMagic:  file read error.");
		return false;
	}
	return true;
}

// for non-raw i/o, compares the endian attribute of the header class
// with the Application (machine architecture) endian attribute.
// If file is raw, use value of forceSwapped only.

boolean
Header::needsByteSwapping() {
	return forceSwapped ||
	    (!isRaw() && (isLittleEndian() != Application::isLittleEndian()));
}

// default operations for header

int
Header::read(DataFile *file) {
	int retcode = false;
	file->setBytesSwapped(needsByteSwapping());
	if(isRaw()) {
		setDataSize(file->dataSize() - dataOffset());
		retcode = checkHeader();
	}
	else {
		initialize();
		if(file->readable()) {
			if(checkMagic(file) == true)
				retcode = (readInfo(file) && readComment(file));
		}
	}
	if(retcode) {
		// adjust data size and offset to account for inskip and duration
		file->setHeaderSize(dataOffset());	// header size in bytes
		int skipOffset = secondsToBytes(file->skipTime());
		file->setOffset(skipOffset);
		file->setReadSize(secondsToBytes(file->duration()));
		setDataSize(min(dataSize(), file->readSize()));
		// seek to beginning of desired data
		// force base class operation if raw
		if (isRaw())
		    retcode = Header::seekInFile(file, skipOffset);
		else
		    retcode = seekInFile(file, skipOffset);
	}
	else
		file->failif(true);		// to notify calling routines
	return retcode;
}

int
Header::seekInFile(DataFile* file, int offset) {
	if(!file->seek(dataOffset() + offset).good()) {
		Application::error("seekInFile: seek error:");
		return false;
	}
	return true;
}

int
Header::readComment(DataFile *file) {
	int textSize = dataOffset() - diskHeaderInfoSize();
	if(textSize > 0) {
		// seek to beginning of text information
		file->seek(diskHeaderInfoSize());
		char *com = new char[textSize + 1];
		file->read((void *) com, textSize);
		com[textSize] = '\0';		// null terminate
		setComment(com);
		delete [] com;
	}
	return file->good();
}

int
Header::write(DataFile *file) {
	int status = false;
	file->setBytesSwapped(needsByteSwapping());
	if(checkHeader() == true) {
		file->reOpen("w+");				// to truncate file
		file->seek(0);
		if(isRaw())						// dont check any further if raw
			status = true;
		else if(file->writable()) {
			file->setHeaderSize(dataOffset());
			if(writeInfo(file)) {
				if(writeComment(file))
					status = true;
			}
			if(status == false)
				file->error();
		}
		else {
			char msg[256];
			sprintf(msg, "Header::write: %s is not writable.", file->name());
			Application::alert(msg);
		}
	}
	return status;
}

int
Header::writeComment(DataFile *file) {
	static char text[4];
	Comment* com = getComment();
	if(com != nil && com->len() > 0)
		file->write((void *) com->text(), com->len());
	else if(diskHeaderCommentSize() > 0)
		file->write((void *) text, diskHeaderCommentSize());
	int status = file->good();
	if(!status) Application::alert("writeComment:  file write error.");
	return status;
}

int
Header::sampleSize() { return type_to_sampsize(dataType()); }

// this is the bytes/sec conversion needed to determine offsets in time

int
FrameDataHeader::secondsToBytes(double seconds) {
	return int(frameRate() * seconds) * (nChans() * sampleSize());
}

// LPC Header Methods

LPCHeader::Type LPCHeader::default_HeaderType = LPCHeader::With;

LPCHeader::LPCHeader(int poles, double framert, int srate, double duration)
	: FrameDataHeader(LP_MAGIC, FloatData, poles+4, framert, srate, duration),
		npoles(poles) {
	if(defaultHeaderType() == None)
		setRaw();
}

LPCHeader::LPCHeader()
	: FrameDataHeader(LP_MAGIC, FloatData, defaultPoles+4,
	                  Sound::defaultSampleRate()/200.0,
	                  Sound::defaultSampleRate(), 0.0),
		npoles(defaultPoles) {
	if(defaultHeaderType() == None)
		setRaw();
}

// static public method

int
LPCHeader::readMagicNumber(DataFile *file) {
	int magic = -1;
	int temp[2];	// LPC magic is second int in header
	if(file->read((void *) temp, sizeof(temp)).good())
		magic = temp[1];
	file->clear();		// no file failure here because this is used to check
	file->seek(0);		// files that may not yet exist
	return magic;		// -1 indicated bad read
}

int
LPCHeader::readMagic(DataFile *file) {
	if((magic_number = LPCHeader::readMagicNumber(file)) == -1) {
		Application::alert("LPCHeader::readMagic:  file read error.");
		return false;
	}
	return true;
}

const char*
LPCHeader::magicError() {
	static char msg[80];
	sprintf(msg, "Invalid LPC file magic number (%d != %d)", magic(), LP_MAGIC);
	return msg;
}

int
LPCHeader::readInfo(DataFile *file) {
	DiskHeader head;
	int status = false;
	if(file->read((void *) &head, diskHeaderSize()).good()) {
		samprate = int(head.srate);
		npoles = head.npoles;
		dur = head.duration;
		framerate = head.framrate;
		setDataOffset(head.headersize);
		setDataSize(file->dataSize() - head.headersize);
		nchans = head.nvals;
		data_type = FloatData; // since other progs create it this way
		status = true;
	}
	else
		Application::alert("LPCHeader::readInfo:  file read error.");
	return status && checkHeader();
}

int
LPCHeader::checkHeader() {
	char msg[64];
	int retcode = 0;
	if(npoles > maxPoles || npoles != (nchans - 4))
		sprintf(msg, "LPC header: invalid npoles (%d)", npoles);
	else if(samprate < 4000 || samprate > 64000)
		sprintf(msg, "LPC header: invalid samp rate (%d)", samprate);
	else if(data_type != FloatData)
		sprintf(msg, "Unknown LPC data type tag: %d", data_type);
	else retcode = 1;
	if(retcode != 1)
		Application::alert(msg);
	return retcode;
}

int
LPCHeader::writeInfo(DataFile *file) {
	// set offset based on comment length
	int offset = diskHeaderInfoSize()
		+ max(diskHeaderCommentSize(), commentLength());
	setDataOffset(offset);
	DiskHeader head(dataOffset(), magic(), nPoles(), nChans(),
		frameRate(), sampleRate(), duration());
	return file->write((void *) &head, diskHeaderInfoSize()).good();
}

// Pitch Track Header Methods

PCHHeader::PCHHeader()
	: FrameDataHeader(PCH_MAGIC, FloatData, 2,
			  Sound::defaultSampleRate()/200.0,
			  Sound::defaultSampleRate(), 0),
		  framesize(350) {
	setRaw();	// since no header written, always raw
}

PCHHeader::PCHHeader(double framert, int fsize, int srate, double dur)
		: FrameDataHeader(PCH_MAGIC, FloatData, 2, framert, srate, dur),
		  framesize(fsize) {
	setRaw();	// since no header written, always raw
}

int
PCHHeader::readMagic(DataFile *file) {
	return file->seek(0).good();
}

int
PCHHeader::readInfo(DataFile *file) {
	return Super::readInfo(file);
}

// FFT Header Methods

FFTHeader::FFTHeader(int npoints, double frmrate, int srate, double dur)
	: FrameDataHeader(FFT_MAGIC, DoubleData, npoints, frmrate, srate,  dur) {}

FFTHeader::FFTHeader() 
	: FrameDataHeader(FFT_MAGIC, DoubleData, 1024,
			  Sound::defaultSampleRate()/200.0,
			  Sound::defaultSampleRate(),
			  0) {}
const char*
FFTHeader::magicError() {
	static char msg[80];
	sprintf(msg, "Invalid FFT file magic number (%d != %d)",
		magic(), FFT_MAGIC);
	return msg;
}

int
FFTHeader::readInfo(DataFile *file) {
	DiskHeader head;
	int status = false;
	if(file->read((void *) &head, diskHeaderSize()).good()) {
		samprate = int(head.srate);
		nchans = head.npoints;
		setDataOffset(head.headersize);
		setDataSize(file->dataSize() - head.headersize);
		data_type = DoubleData;		// no way to check -- we assume this
		status = true;
	}
	else
		Application::alert("FFTHeader::readInfo:  file read error.");
	return status && checkHeader();
}

int
FFTHeader::checkHeader() {
	int status = false;
	char msg[128];
	if(nChans() % 2 != 0)
		sprintf(msg, "Invalid number of FFT points: %d [must be power of 2]",
			nChans());
	else status = true;
	if(status != true)
		Application::alert(msg);
	return status;
}

int
FFTHeader::writeInfo(DataFile *file) {
	// set offset based on comment length
	int offset = diskHeaderInfoSize()
		+ max(diskHeaderCommentSize(), commentLength());
	setDataOffset(offset);
	DiskHeader head(magic(), dataOffset(), nPoints(), sampleRate());
	return file->write((void *) &head, diskHeaderInfoSize()).good();
}

// Envelope Header Methods

EnvelopeHeader::Type EnvelopeHeader::default_HeaderType = EnvelopeHeader::With;

EnvelopeHeader::EnvelopeHeader(int npts, int srate)
	: Header(EVP_MAGIC, FloatData, EvpChannels),
	  samprate(srate), npoints(npts) {}

EnvelopeHeader::EnvelopeHeader()
	: Header(EVP_MAGIC, FloatData, EvpChannels),
	  samprate(Sound::defaultSampleRate()), npoints(EvpMinPoints) {}

const char*
EnvelopeHeader::magicError() {
	static char msg[80];
	sprintf(msg, "Invalid Envelope file magic number (%d != %d)",
		magic(), EVP_MAGIC);
	return msg;
}

int
EnvelopeHeader::readInfo(DataFile *file) {
	DiskHeader head;
	int status = false;
	if(file->read((void *) &head, sizeof(head)).good()) {
		samprate = int(head.srate);
		npoints = head.npoints;
		setDataOffset(head.headersize);
		setDataSize(file->dataSize() - head.headersize);
		nchans = 1;
		data_type = FloatData;
		status = true;
	}
	else
		Application::alert("EnvelopeHeader::readInfo:  file read error.");
	return status && checkHeader();
}

int
EnvelopeHeader::checkHeader() {
	int status = false;
	if(data_type != FloatData)
		Application::alert("Invalid envelope data type [must be floating point].");
	else if(nchans != 1)
		Application::alert("Envelope header:  nchans must be 1.");
	else status = true;
	return status;
}

int
EnvelopeHeader::writeInfo(DataFile *file) {
	// set offset based on comment length
	int offset = diskHeaderInfoSize()
		+ max(diskHeaderCommentSize(), commentLength());
	setDataOffset(offset);
	DiskHeader head(magic(), dataOffset(), nPoints(), sampleRate());
	return file->write((void *) &head, diskHeaderInfoSize()).good();
}
