/* sync.c - Synchronize files.

   Copyright (C) 1998 Tom Tromey

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

#include <config.h>

#include <assert.h>
#include <string.h>
#include <unistd.h>
#include <sys/mman.h>
#include <errno.h>
#include <fcntl.h>

#include "sync.h"

static void schedule_next_file (struct database *database);



/* Number of bytes to read before we consider the file too big.  We
   enforce a useful minimum bound.  */
#define MAX_SIZE  ((max_file_size < 1) ? 1024 : (max_file_size * 1024))

/* Number of bytes to try to read at once.  */
#define READ_SIZE 1024



/* Return 1 if file is `done', 0 otherwise.  */
static int
file_done (const struct file *file)
{
  return file->state == STATE_DONE || file->state == STATE_MISSING;
}

/* This is called when both files have been completely read.  It
   determines what to do -- copy a file, nothing, ask the user.  Then
   it schedules the next file to be handled.  */
static void
synchronize_one_file (struct reader *reader)
{
  struct database *database = reader->database;

  /* One or the other must exist.  */
  assert (reader->local->state != STATE_MISSING
	  || reader->remote->state != STATE_MISSING);

  if (reader->local->state != STATE_MISSING
      && reader->remote->state != STATE_MISSING
      && ! strcmp (reader->local->checksum, reader->remote->checksum))
    {
      /* Both files exist and are in sync.  */
      display_message (reader->database, reader->local->name,
		       reader->remote->name, "");
      free_reader (reader);
    }
  else if (reader->local->state == STATE_MISSING
	   || ! strcmp (reader->local->checksum, reader->entry->checksum))
    {
      /* Local file is missing, or remote file has changed.  */
      copy (reader, COPY_FROM_REMOTE);
    }
  else if (reader->remote->state == STATE_MISSING
	   || ! strcmp (reader->remote->checksum, reader->entry->checksum))
    {
      /* Remote file is missing, or local file has changed.  */
      copy (reader, COPY_FROM_LOCAL);
    }
  else
    {
      /* Both differ.  Add to list to ask the user.  */
      display_error (reader->database, reader->local->name,
		     reader->remote->name,
		     _("Both files modified since last synchronization"));
      free_reader (reader);
    }

  schedule_next_file (database);
}

/* Wrapper for synchronize_one_file that is called as an idle
   function.  */
static gint
synchronize_when_idle (gpointer client_data)
{
  synchronize_one_file ((struct reader *) client_data);
  return 0;
}

/* Cancel a file read in progress.  */
static void
cancel_io (struct file *file)
{
  if ((file->state == STATE_READING || file->state == STATE_WRITING)
      && file->fd != -1)
    {
      gdk_input_remove (file->tag);
      close (file->fd);
      file->tag = 0;
      file->fd = -1;
    }
  else if ((file->flags & FLAG_MAPPED) && file->buffer != NULL)
    {
      munmap (file->buffer, file->size);
      file->flags &= ~FLAG_MAPPED;
      file->buffer = NULL;
    }
  file->state = STATE_NEW;
}

/* Clean up a file structure.  */
static void
free_file (struct file *file)
{
  /* Cancel I/O on file.  */
  cancel_io (file);
  free (file->name);
  if (file->buffer)
    free (file->buffer);
  free (file);
}

/* Delete and clean up a reader structure.  */
void
free_reader (struct reader *reader)
{
  free_file (reader->local);
  free_file (reader->remote);
  free_entry (reader->entry);

  reader->database->ios = g_list_remove (reader->database->ios,
					 (gpointer) reader);

  free (reader);
}

/* Allocate and return a new file.  */
static struct file *
new_file (const char *name)
{
  struct file *file;

  file = (struct file *) malloc (sizeof (struct file));
  file->name = strdup (name);
  file->checksum[16] = '\0';
  file->fd = -1;
  file->tag = 0;		/* More or less arbitrary.  */
  file->state = STATE_NEW;
  file->mode = 0;
  file->flags = 0;
  md5_init_ctx (&file->context);
  file->slot = 0;
  file->buffer = NULL;

  return file;
}

/* Read a chunk and process it.  We compute the checksum on the fly.
   If we read "too much" data, we don't cache it but instead just
   throw it away after processing.  */
static void
read_chunk (gpointer client_data, gint fd, GdkInputCondition cond)
{
  struct file *file;
  struct reader *reader;
  size_t prev64, ret;

  assert (cond == GDK_INPUT_READ);

  reader = (struct reader *) client_data;
  assert (fd == reader->local->fd || fd == reader->remote->fd);
  file = (reader->local->fd == fd) ? reader->local : reader->remote;

  /* Compute last 64-byte boundary.  */
  prev64 = file->slot & ~63;

  /* If buffer is full, consider allocating some space.  */
  if (file->slot == file->size)
    {
      int too_full = 0;

      if (file->size * 2 <= MAX_SIZE)
	{
	  /* Try to reallocate.  If this fails, we can still continue
	     by using the old buffer.  So do not change this to an
	     "xmalloc"-like call.  */
	  size_t ns = file->size * 2;
	  char *nb = malloc (ns);
	  if (nb != NULL)
	    {
	      memcpy (nb, file->buffer, file->slot);
	      free (file->buffer);
	      file->buffer = nb;
	      file->size = ns;
	    }
	  else
	    too_full = 1;
	}

      if (too_full)
	{
	  /* Either we reached our size limit or we couldn't allocate
	     more memory.  Note that we must preserve any data after
	     the last 64-byte boundary; the reason is that this data
	     hasn't yet been fed into the checksum.  */
	  memcpy (file->buffer, &file->buffer[prev64], file->slot - prev64);
	  file->slot -= prev64;
	  file->flags &= ~FLAG_FULL;
	}
    }

  /* Try to fill the buffer.  */
  ret = read (fd, &file->buffer[file->slot], file->size - file->slot);
  if (ret == (size_t) -1)
    {
      gchar *u;

      /* Certain errors are ok.  */
      if (errno == EINTR)
	return;

      if (file == reader->local)
	u = _("Error while reading local file");
      else
	u = _("Error while reading remote file");

      display_error (reader->database, reader->local->name,
		     reader->remote->name, u);
      free_reader (reader);
      return;
    }
  else if (ret == 0)
    {
      /* Process last bytes.  */
      md5_final_block (&file->buffer[prev64], file->size - prev64,
		       &file->context);

      md5_read_ctx (&file->context, &file->checksum);

      gtk_input_remove (file->tag);
      close (file->fd);
      file->tag = 0;
      file->fd = -1;
      file->state = STATE_DONE;

      /* If both files done reading, then do synchronization.  `Done'
	 means either finished or missing.  */
      if (file_done (reader->local)
	  && file_done (reader->remote))
	synchronize_one_file (reader);
    }
  else
    {
      /* Ordinary read.  */
      size_t next64;
      file->slot += ret;
      next64 = file->slot & ~63;

      /* If we've advanced past a 64-byte boundary, add the new data
	 to the checksum.  */
      if (next64 != prev64)
	md5_process_block (&file->buffer[prev64], next64 - prev64,
			   &file->context);
    }
}

/* Start reading a file.  If we can mmap the file, we do so.
   Otherwise we read it.  We try to keep the file in memory, but if it
   grows too large, we don't.  Returns 0 on error.  */
static int
start_reading (struct file *file, struct reader *reader)
{
  struct stat buf;

  file->fd = open (file->name, O_RDONLY);
  if (file->fd == -1)
    {
      if (errno == ENOENT)
	{
	  /* This is ok here; it is dealt with at a higher level.  */
	  file->state = STATE_MISSING;
	  return 1;
	}
      file->state = STATE_ERROR;
      return 0;
    }

  file->flags = FLAG_FULL;

  /* FIXME: when using vfs, fstat() is probably the wrong thing to
     do.  The size information might be useful to us in the non-mmap
     case; we might be able to collect it with an ordinary stat.  */
  if (fstat (file->fd, &buf) != -1)
    {
      file->mode = buf.st_mode;
      file->size = buf.st_size;
      file->buffer = mmap (NULL, file->size, PROT_READ, MAP_SHARED,
			   file->fd, 0);
      if (file->buffer == (void *) -1)
	file->buffer = NULL;
      else
	{
	  close (file->fd);
	  file->fd = -1;
	  file->state = STATE_DONE;
	  md5_buffer (file->buffer, file->size, &file->checksum);
	  file->flags |= FLAG_MAPPED;
	}
    }
  else
    {
      /* FIXME: another reason to use stat().  */
      file->mode = 0700;
    }

  /* If mmap failed, schedule the reader.  */
  if (file->buffer == NULL)
    {
      /* FIXME: do a stat and use that to possibly size the buffer.
	 Can only really do this if VFS caches interesting info.  */
      file->size = READ_SIZE;
      file->buffer = malloc (file->size);
      file->state = STATE_READING;

      gdk_input_add (file->fd, GDK_INPUT_READ, read_chunk, (gpointer) reader);
    }

  return 1;
}

/* Look up entry for a file and synchronize it with the remote host.
   This will simply start some reads and schedule the remaining work
   to be done in the background.  Return 1 on error, 0 otherwise.  */
static int
start_file_synchronization (struct database *database, const char *local_name)
{
  struct entry *info;
  struct reader *reader;

  info = lookup (database->database, local_name);
  if (! info)
    return 1;

  reader = (struct reader *) malloc (sizeof (struct reader));
  reader->local = new_file (local_name);
  reader->remote = new_file (info->name);
  reader->entry = info;
  reader->database = database;

  database->ios = g_list_prepend (database->ios, (gpointer) reader);

  if (start_reading (reader->local, reader))
    {
      if (! start_reading (reader->remote, reader))
	{
	  display_posix_error (database, local_name, info->name,
			       _("Couldn't open remote file for reading"));
	  free_reader (reader);
	  schedule_next_file (database);
	  return 1;
	}
    }
  else
    {
      display_posix_error (database, local_name, info->name,
			   _("Couldn't open local file for reading"));
      free_reader (reader);
      schedule_next_file (database);
      return 1;
    }

  /* If both files are missing, then that is an error.  It is ok for
     just one to be missing -- in that case it will get copied over.  */
  if (reader->local->state == STATE_MISSING
      && reader->remote->state == STATE_MISSING)
    {
      display_error (database, local_name, info->name,
		     _("Both local and remote files are missing"));
      free_reader (reader);
      schedule_next_file (database);
      return 1;
    }

  display_message (database, local_name, info->name,
		   _("Synchronizing..."));

  /* If both files are finished reading, we still schedule the work in
     the background.  This avoids recursing through
     schedule_next_file().  */
  if (file_done (reader->local) && file_done (reader->remote))
    gtk_idle_add (synchronize_when_idle, reader);

  return 0;
}

/* Called when synchronization finished.  If this was a "just sync"
   invocation, we might exit.  */
static void
finish (struct database *database)
{
  DB *db = database->database;

  /* This function can be called multiple times for a single
     synchronization if we run several transfers in parallel.  So we
     don't actually do the work until the very last transfer has
     finished.  */
  if (database->ios)
    return;

  if (db->sync (db, 0) == -1)
    db_posix_error (_("Error while synchronizing database file"),
		    database->filename);
  if ((database->flags & FLAG_DB_SYNC) && ! (database->flags & FLAG_DB_ERROR))
    force_quit (database);
}

/* Schedule next file for this sync effort.  */
static void
schedule_next_file (struct database *database)
{
  DBT key, value;
  DB *db = database->database;

  while (1)
    {
      int r = db->seq (db, &key, &value, R_NEXT);

      if (r == 1)
	{
	  /* Nothing left in database.  */
	  finish (database);
	  break;
	}
      else if (r == -1)
	{
	  db_posix_error (_("Error while reading database file"),
			  database->filename);
	  finish (database);
	  break;
	}

      if (! start_file_synchronization (database, key.data))
	break;
    }
}

/* Shut down a reader.  Function suitable for use by g_list_foreach.  */
static void
shut_down (gpointer data, gpointer user_data)
{
  struct reader *reader = (struct reader *) data;
  free_reader (reader);
}

/* Primary interface to synchronization.  This starts a sync for a
   given database.  */
void
synchronize (struct database *database)
{
  DBT key, value;
  DB *db = database->database;
  int r, count;

  /* Shut down every currently active reader.  */
  g_list_foreach (database->ios, shut_down, NULL);
  g_list_free (database->ios);
  database->ios = NULL;

  clear_errors (database);

  r = db->seq (db, &key, &value, R_FIRST);
  if (r == 1)
    finish (database);
  else if (r == -1)
    {
      db_posix_error (_("Error while reading database file"),
		      database->filename);
      finish (database);
    }
  else
    {
      if (concurrent_transfers > 1)
	count = concurrent_transfers - 1;
      else
	count = 0;

      if (start_file_synchronization (database, key.data))
	{
	  /* On failure, try one more.  */
	  ++count;
	}

      /* Schedule COUNT more transfers to work in parallel.  */
      for (; count > 0; --count)
	schedule_next_file (database);
    }
}
