/*								-*- C++ -*-
 * $Id: WIN_listbox.cpp,v 1.4 1998/08/30 19:39:55 wg Exp $
 *
 * Purpose: list box panel item
 *
 * Authors: Markus Holzem and Julian Smart
 *
 * Copyright: (C) 1995, AIAI, University of Edinburgh (Julian)
 * Copyright: (C) 1995, GNU (Markus)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Additionally everyone using this library has to announce it with:
 *
 *   This software uses the wxWindows-Xt GUI library
 *   (C) Markus Holzem, available via
 *       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
 */

#ifdef __GNUG__
#pragma implementation "WIN_listbox.h"
#endif

#define  Uses_XtIntrinsic
#define  Uses_wxListBox
#define  Uses_wxStringList
#include "wx.h"
#define  Uses_EnforcerWidget
#define  Uses_MultiListWidget
#define  Uses_ScrollWinWidget
#include "widgets.h"

// don't allocate or free for every append or delete
#define LIST_CHUNK_SIZE	20
#define MULTILIST	((XfwfMultiListWidget)(HWidget()))
#define BORDER_WIDTH    3

//-----------------------------------------------------------------------------
// create and destroy wxListBox
//-----------------------------------------------------------------------------

IMPLEMENT_DYNAMIC_CLASS(wxListBox, wxItem)

wxListBox::wxListBox(void) : wxItem()
{
    __type = wxTYPE_LIST_BOX;

    AllowDoubleClick(TRUE);

    choices = client_data = NULL;
    num_choices = 0;
    num_free = 0;
    multiple = wxSINGLE;

    quoted = FALSE;
}

wxListBox::wxListBox(wxPanel *panel, wxFunction func, Const char *title,
		     Bool Multiple, int x, int y, int width, int height,
		     int n, char **_choices, long style, Constdata char *name):
wxItem()
{
    __type = wxTYPE_LIST_BOX;

    AllowDoubleClick(TRUE);

    choices = client_data = NULL;
    num_choices = 0;
    num_free = 0;

    quoted = FALSE;

    Create(panel, func, title, Multiple, x, y, width, height,
	   n, _choices, style, name);
}

Bool wxListBox::Create(wxPanel *panel, wxFunction func, Const char *title,
		       Bool Multiple, int x, int y, int width, int height,
		       int n, char **choices, long style, Constdata char *name)
{
    multiple  = Multiple & (wxSINGLE|wxMULTIPLE|wxEXTENDED);
    style    |= long(multiple);

    ChainToPanel(panel, style, name);

    // create frame
    FWidget() = XtVaCreateManagedWidget
	(name, xfwfEnforcerWidgetClass, GetParentWidget(parent),
	 XtNlabel,       title,
	 XtNalignment,   (label_position == wxVERTICAL) ? XfwfTop : XfwfTopLeft,
	 XtNbackground,  bg.GetPixel(&cmap),
	 XtNforeground,  label_fg.GetPixel(&cmap),
	 XtNfont,        label_font.GetInternalFont(),
	 XtNhighlightThickness, 0,
	 XtNlabelOffset,        2,
	 XtNtraversalKeys, XfwfTraverseKeyAll
			   & (~(  XfwfTraverseKeyUp
				| XfwfTraverseKeyDown
				| XfwfTraverseKeyHome)),
	 NULL);
    // create viewport
    PWidget() = XtVaCreateManagedWidget
	(name, xfwfScrolledWindowWidgetClass, FWidget(),
	 XtNbackground,      bg.GetPixel(&cmap),
	 XtNhideHScrollbar,  !(style & wxHSCROLL),
	 XtNhInit,	     0 /*-BORDER_WIDTH*/,
	 XtNscrollingPolicy, XfwfAutoScrolling /*XfwfUserScrolling*/,
	 XtNtraversalKeys,   XfwfTraverseKeyAll
	 		     & ~(XfwfTraverseKeyUp
				 | XfwfTraverseKeyDown
				 | XfwfTraverseKeyHome),
	 NULL);
    // create multi list
    HWidget() = XtVaCreateManagedWidget
	(name, xfwfMultiListWidgetClass, PWidget(),
	 XtNbackground,       bg.GetPixel(&cmap),
	 XtNforeground,       fg.GetPixel(&cmap),
	 XtNborderColor,      bg.GetPixel(&cmap),
	 XtNborderWidth,      BORDER_WIDTH,
	 XtNfont,             font.GetInternalFont(),
	 XtNrowSpacing,       2,
	 XtNshadeSurplus,     FALSE,
	 XtNdefaultColumns,   1,
	 XtNforceColumns,     TRUE,
	 XtNcolumnSpacing,    1,
	 XtNmaxSelectable,    style & (wxMULTIPLE|wxEXTENDED) ? 10000 : 1,
	 XtNextendedMultiple, (Boolean)style & wxEXTENDED,
	 NULL);
    Set(n, choices);
    // configure scrollbar
    Dimension row_height;
    XtVaGetValues(HWidget(), XtNrowHeight, &row_height, NULL);
    XtVaSetValues(PWidget(), XtNvIncrement, row_height, NULL);
    // callback
    callback = func;
    XtAddCallback(HWidget(), XtNcallback,
		  wxListBox::EventCallback,  (XtPointer)this);

    panel->PositionItem(this, x, y,
			(width  > -1 ? width  : wxLIST_BOX_WIDTH),
			(height > -1 ? height : wxLIST_BOX_HEIGHT));
    AddEventHandlers();
    return TRUE;
}

wxListBox::~wxListBox(void)
{
    Clear();
}

//-----------------------------------------------------------------------------
// OnChar handles cursor keys
//-----------------------------------------------------------------------------

void wxListBox::OnChar(wxKeyEvent& event)
{
    // do quoting of keycodes
    if (event.ControlDown() && (event.KeyCode()=='q' || event.KeyCode()=='Q')) {
	quoted = TRUE;
	return;
    } else if (quoted) {
	quoted = FALSE;
	wxItem::OnChar(event);
	return;
    }

    // do scrolling and selecting
    Position  d;
    Dimension window_width, window_height;
    int       client_y,     client_height;
    Dimension row;
    int       first, num, old, sel=-1, visi;
    // get all necessary geometries
    XfwfCallComputeInside(PWidget(), &d, &d, &window_width, &window_height);
    XtVaGetValues(HWidget(), XtNrowHeight, &row, NULL);
    client_y = GetScrollPos(wxVERTICAL);
    GetClientSize(NULL, &client_height);
    window_height -= window_height % row; // multiple of row
    // compute positions
    first = client_y / row + (client_y % row ? 1 : 0);
    visi  = window_height / row;
    num   = Number();
    old   = GetSelection();

    switch (event.KeyCode()) {
    // do traversing
    case WXK_TAB:
	if (event.ShiftDown()) TravPrev(&event); else TravNext(&event);
	return;
    case WXK_LEFT:
	TravLeft(&event);
	return;
    case WXK_RIGHT:
	TravRight(&event);
	return;
    // do action
    case WXK_RETURN:
	if (old != -1) GetParent()->GetEventHandler()->OnDefaultAction(this);
	return;
    // do rout up to item event handler
    default:
	wxItem::OnChar(event);
	return;
    // do scrolling and change of selection
    case WXK_HOME:
	sel = (num ? 0 : -1); first = 0;
	break;
    case WXK_END:
	sel = num - 1; first = wxMax(0, num - visi);
	break;
    case WXK_NEXT: {
	if (first == num - visi) return;
	int step = (num-first > visi-1) ? visi-1 : num-first;
	if (first <= old && old < first+visi) // selection visible?
	    sel = wxMin(old + step, num-1);
	first += step;
	} break;
    case WXK_DOWN:
	if (old == num-1) return; // nothing to do
	sel = wxMin(old + 1, num - 1);
	if ( !(first <= sel && sel < first+visi) ) // selection invisible?
	    first = sel - visi + 1;
	break;
    case WXK_PRIOR: {
	if (first == 0) return;
	int step = (first > visi-1) ? visi-1 : first;
	if (first <= old && old < first+visi) // selection visible?
	    sel = old - step;
	first -= step;
	} break;
    case WXK_UP:
	if (old == 0) return; // nothing to do
	sel = wxMax(0, old - 1);
	if ( !(first <= sel && sel < first+visi) ) // selection invisible?
	    first = sel;
	break;
    }
    // !!! everything without a change in selection returned before this point
    // change highlighting and send event

    // debug output
    // cerr << old << " -> " << sel << " (" << num << "), first="
    // 	    << first << " visible=" << visi << endl;

    Scroll(0, first*row);	// scroll nonetheless
    if (old == sel) return;	// nothing to do
    if (old != -1)		// unhighlight old selection
	SetSelection(old, FALSE);
    if (sel != -1) {		// highlight new selection
	SetSelection(sel, TRUE);
	wxCommandEvent event(wxEVENT_TYPE_LISTBOX_COMMAND);
	event.commandInt    = sel;
	event.commandString = GetString(sel);
	event.clientData    = GetClientData(sel);
	ProcessCommand(event);
    }
}

//-----------------------------------------------------------------------------
// override parent methods
//-----------------------------------------------------------------------------

void wxListBox::SetSize(int x, int y, int width, int height, int flags)
{
    Position  dummy;
    Dimension swd, sht, listWidth, listHeight, scrollbarWidth;

    // resize scrolledWindowWidget
    wxItem::SetSize(x, y, width, height, flags);
    // determine necessity of scrollbars and resize multiListWidget
    XfwfCallComputeInside(FWidget(), &dummy, &dummy, &swd, &sht);
    XtVaGetValues(HWidget(),
		  XtNwidth, &listWidth,
		  XtNheight, &listHeight,
		  NULL);
    XtVaGetValues(PWidget(), XtNscrollbarWidth, &scrollbarWidth, NULL);
    scrollbarWidth += 4; // gap between visible frame and scrollbar
    Boolean hide_v = True, hide_h = True;
    for(int i=0; i<2; i++) { // two passes are necessary
	if(hide_v && listHeight+2*BORDER_WIDTH>sht && swd>scrollbarWidth) {
	    hide_v = False;
	    swd -= scrollbarWidth;
	}
	if(hide_h && listWidth+2*BORDER_WIDTH>swd && sht>scrollbarWidth) {
	    hide_h = False;
	    sht -= scrollbarWidth;
	}
    }
    // don't let list bars become narrower than the visible area
    if(swd>=2*BORDER_WIDTH && listWidth<swd-2*BORDER_WIDTH)
	XtVaSetValues(HWidget(), XtNwidth, swd-2*BORDER_WIDTH, NULL);
    XtVaSetValues(PWidget(),
		  XtNhideHScrollbar, hide_h,
		  XtNhideVScrollbar, hide_v,
		  NULL);
}

void wxListBox::Scroll(int WXUNUSED(x_pos), int y_pos)
{
    if (!PWidget() || !HWidget() || y_pos < 0) return;

    Position x, y; Dimension vwd, vht, ht;
    // size of viewport
    XfwfCallComputeInside(PWidget(), &x, &y, &vwd, &vht);
    // get geometry of scrollable window
    XtVaGetValues(HWidget(), XtNy, &y, XtNheight, &ht, NULL);
    // adjust values
    y = wxMin(y_pos, ht - vht);
    y = wxMax(0, y);
    // set geometry of scrollable window
    XtVaSetValues(HWidget(), XtNy, -y, NULL);
}

// (ben@c-lab.de) ADD
// You must also set the colors of the board widget, because if the child
// is small (little entries in listbox), it is visible
void wxListBox::ChangeColours()
{
    wxItem::ChangeColours();

    if(PWidget()) {
	XtVaSetValues(PWidget(),
		      XtNbackground,  bg.GetPixel(&cmap),
		      XtNforeground,  fg.GetPixel(&cmap),
		      XtNborderColor, bg.GetPixel(&cmap),
		      NULL);
    }
}

//-----------------------------------------------------------------------------
// change contents of wxListBox
//-----------------------------------------------------------------------------

void wxListBox::Append(char *item)
{
    if (num_free == 0) {
	num_free = LIST_CHUNK_SIZE;
	char    **new_choices     = wxNEW char *[num_choices+LIST_CHUNK_SIZE];
	char    **new_client_data = wxNEW char *[num_choices+LIST_CHUNK_SIZE];
	// copy current choices
	for (int i=0; i<num_choices; ++i) {
	    new_choices[i] = choices[i];
	    new_client_data[i] = client_data[i];
	}
	// delete old arrays
	delete[] choices;      choices = new_choices;
	delete[] client_data;  client_data = new_client_data;
    }
    // set new item
    choices[num_choices]     = copystring(item);
    client_data[num_choices] = NULL;
    // one choice more, one free space less
    ++num_choices; --num_free;
    // append item to widget
    XfwfMultiListAppendData(MULTILIST, choices, num_choices,
			    0, TRUE, (Boolean*)NULL);
    SetSize(-1, -1);
    placeList();
}

void wxListBox::Append(char *item, char *_client_data)
{
    Append(item);
    client_data[num_choices-1] = _client_data;
}

void wxListBox::Clear(void)
{
    if (choices) {
	// free strings
	for (int i=0; i<num_choices; ++i)
	    delete[] choices[i];
	// free array
	delete[] choices;
	choices = NULL;
    }
    if (client_data) {
	delete[] client_data;
	client_data = NULL;
    }
    num_choices = num_free = 0;
    SetInternalData();
    Scroll(0, 0);
}

void wxListBox::Fit()
{
    float w, h, wmax = -1.0;

    for (int i=0; i<num_choices; ++i) {
	GetTextExtent(choices[i], &w, &h);
	if (w > wmax)
	    wmax = w;
    }
    if (wmax != -1.0) {
	Dimension scrollbarWidth;
	XtVaGetValues(PWidget(), XtNscrollbarWidth, &scrollbarWidth, NULL);
	XtVaSetValues(HWidget(),
		      XtNwidth, (int)wmax+2*BORDER_WIDTH,
		      NULL);
	SetSize((int)wmax+2*BORDER_WIDTH+(scrollbarWidth+8), -1);
    }
}

//----------------------------------------------------------------------------
//// MN    Method name: placeList
//// MC    of class:    wxListBox
//// MSD   Description: places the listbox widget inside the scolling widgets
//// MED   If listbox is smaller it places the left/top, if listbox is bigger it
////       will be placed right/down. This is typically called after a delete
//// 
//// MI    Input :
//// MO    Ouput :
//

void
wxListBox::placeList()
{
    Position x,y;
    Dimension listWidth, listHeight;
    Dimension innerWidth, innerHeight;

    XfwfCallComputeInside(PWidget(), &x, &y, &innerWidth, &innerHeight);
    XtVaGetValues(HWidget(), XtNx, &x, XtNy, &y, XtNwidth, &listWidth,
		  XtNheight, &listHeight, NULL);

    if(x + listWidth <= innerWidth) {
	x = listWidth <= innerWidth ? 0 : innerWidth - listWidth;
	XtVaSetValues( HWidget(), XtNx, x, NULL );
    }
    if(y + listHeight <= innerHeight) {
	y = listHeight <= innerHeight ? 0 : innerHeight - listHeight;
	XtVaSetValues( HWidget(), XtNy, y, NULL );
    }
}

//-------------------- end: wxListBox::placeList ()  -----------------------

void wxListBox::Delete(int n)
{
    if (0 <= n && n < num_choices) {
	delete[] choices[n]; // free string;
	for (int i=n+1; i<num_choices; ++i) { // shrink arrays
	    choices[i-1] = choices[i];
	    client_data[i-1] = client_data[i];
	}
	--num_choices; ++num_free;
	SetInternalData();
	placeList();
    }
}

void wxListBox::InsertItems(int n_items, char **items, int pos)
{
    pos = pos < num_choices ? pos : num_choices;
    int     i, j;
    char    **new_choices     = wxNEW char *[num_choices+n_items];
    char    **new_client_data = wxNEW char *[num_choices+n_items];

    for (i = 0; i < pos; ++i) {			// copy choices previous to pos
	new_choices[i] = choices[i];
	new_client_data[i] = client_data[i];
    }
    for (j = 0; j < n_items; ++i, ++j) {		    // copy new choices
	new_choices[i] = items[j];
	new_client_data[i] = NULL;
    }
    for (j = pos; j < num_choices; ++i, ++j) { 	       // copy left old choices
	new_choices[i] = choices[j];
	new_client_data[i] = client_data[j];
    }
    num_choices+=n_items;
    // delete old arrays
    delete[] choices;      choices = new_choices;
    delete[] client_data;  client_data = new_client_data;

    SetInternalData();
}

void wxListBox::Set(int n, char *_choices[])
{
    // clear ListBox
    Clear();
    // copy choices and initialize client_data
    num_choices = n;
    num_free = LIST_CHUNK_SIZE;
    choices = wxNEW char*[n+num_free];
    client_data = wxNEW char*[n+num_free];
    for (int i=0; i<n; ++i) {
	choices[i] = copystring(_choices[i]);
	client_data[i] = NULL;
    }
    SetInternalData();
    Scroll(0, 0);
}

void wxListBox::Set(wxStringList *slist)
{
    int	n = slist->Number();
    int i = 0;

    // clear ListBox
    Clear();
    // copy choices and initialize client_data
    num_choices = n;
    num_free = LIST_CHUNK_SIZE;
    choices = wxNEW char*[n+num_free];
    client_data = wxNEW char*[n+num_free];
    for (wxNode *node = slist->First(); node; node = node->Next()) {
	choices[i] = copystring((char*)node->Data());
	client_data[i] = NULL;
	++i;
    }
    SetInternalData();
    Scroll(0, 0);
}

void wxListBox::SetInternalData(void)
{
    XfwfMultiListSetNewData(
	MULTILIST, num_choices ? choices : (String*)NULL, num_choices,
	0, TRUE, (Boolean*)NULL);
    // adjust size
    SetSize(-1, -1);
}

void wxListBox::SetFirstItem(int n)
{
    if (0 <= n && n < num_choices) {
	char *_choice, *_client_data;
	_choice        = choices[0];
	choices[0]     = choices[n];
	choices[n]     = _choice;
	_client_data   = client_data[0];
	client_data[0] = client_data[n];
	client_data[n] = _client_data;
	// TOMSMOD replaced with lines below
	// WARNING: I have not tested this but it should work, since
	// I have already tested XfwfMultiListSetStringData in 
	// wxListBox::SetString
	int ww, hh;
	GetSize(&ww, &hh);
	XfwfMultiListSetStringData(MULTILIST, 0,choices, num_choices,
				   ww, TRUE, (Boolean*)NULL);
	XfwfMultiListSetStringData(MULTILIST, n,choices, num_choices,
				   ww, TRUE, (Boolean*)NULL);
    }
}

void wxListBox::SetFirstItem(char *s)
{
    int n;
    if ((n = FindString(s)) > -1) {
	SetFirstItem(n);
    }
}

//-----------------------------------------------------------------------------
// change state of wxListBox
//-----------------------------------------------------------------------------

void wxListBox::Deselect(int n)
{
    XfwfMultiListUnhighlightItem(MULTILIST, n);
}

int wxListBox::FindString(char *s)
{
    for (int i=0; i<num_choices; ++i)
	if (!strcmp(s, choices[i]))
	    return i;
    return -1;
}

char *wxListBox::GetClientData(int n)
{
    if (0 <= n && n < num_choices)
	return client_data[n];
    return NULL;
}

int wxListBox::GetSelection(void)
{
    XfwfMultiListReturnStruct *rs
	= XfwfMultiListGetHighlighted(MULTILIST);
    if (rs->num_selected==1)
	return rs->selected_items[0];
    return -1;
}

int wxListBox::GetSelections(int **list_selections)
{
    XfwfMultiListReturnStruct *rs
	= XfwfMultiListGetHighlighted(MULTILIST);
    *list_selections = rs->selected_items;
    return (rs->num_selected);
}

char *wxListBox::GetString(int n)
{
    if (0 <= n && n < num_choices)
	return choices[n];
    return NULL;
}

char *wxListBox::GetStringSelection(void)
{
    int n;
    if ((n = GetSelection()) > -1)
	return choices[n];
    return NULL;
}

int wxListBox::Number(void)
{
    return num_choices;
}

Bool wxListBox::Selected(int n)
{
    if (0 <= n && n < num_choices)
	return XfwfMultiListIsHighlighted(MULTILIST, n);
    return FALSE;
}

void wxListBox::SetClientData(int n, char *_client_data)
{
    if (0 <= n && n < num_choices)
	client_data[n] = _client_data;
}

void wxListBox::SetSelection(int n, Bool select)
{
    if ( !(0 <= n && n < num_choices) )
	return;

    if (!select) {
	XfwfMultiListUnhighlightItem(MULTILIST, n);
	return;
    }
	
    // selected items should show up (Thanks Tom!)
    XfwfMultiListHighlightItem(MULTILIST, n);
    // do scrolling and selecting
    Position  d;
    Dimension window_width, window_height;
    int       client_y,     client_height;
    Dimension row;
    int       first, num, sel, visi;
    // get all necessary geometries
    XfwfCallComputeInside(PWidget(), &d, &d, &window_width, &window_height);
    XtVaGetValues(HWidget(), XtNrowHeight, &row, NULL);
    client_y = GetScrollPos(wxVERTICAL);
    GetClientSize(NULL, &client_height);
    window_height -= window_height % row; // multiple of row
    // compute positions
    first = client_y / row + (client_y % row ? 1 : 0);
    visi  = window_height / row;
    num   = Number();
    sel   = GetSelection();
    //if (old == num-1) return; // nothing to do
    //sel = wxMin(old + 1, num - 1);
    //if ( !(first <= sel && sel < first+visi) ) // selection invisible?
    //    first = sel - visi + 1;
    if ( first > sel ) // selection invisible?
	first = sel;
    else if ( sel >= first+visi ) // selection invisible?
	first = sel - visi + 1;
    Scroll(0, first*row);	// scroll nonetheless
}

void wxListBox::SetString(int n, char *s)
{
    if(n >= num_choices) {
	Append(s);
	return;
    }
    if (n < 0)
	return;

    delete[] choices[n];
    choices[n]     = copystring(s); 
    client_data[n] = NULL; // do not delete client data
    int ww, hh;
    GetSize(&ww, &hh);
    XfwfMultiListSetStringData(MULTILIST, n,choices, num_choices,
			       ww, TRUE, (Boolean*)NULL);
}

Bool wxListBox::SetStringSelection(char *s)
{
    int n;
    if ((n = FindString(s)) > -1) {
	SetSelection(n);
	return TRUE;
    }
    return FALSE;
}

//-----------------------------------------------------------------------------
// callback for xfwfMultiListWidgetClass
//-----------------------------------------------------------------------------

void wxListBox::EventCallback(Widget WXUNUSED(w),
			      XtPointer dclient, XtPointer dcall)
{
    wxListBox                 *lbox   = (wxListBox*)dclient;
    XfwfMultiListReturnStruct *rs     = (XfwfMultiListReturnStruct*)dcall;
    wxCommandEvent            event(wxEVENT_TYPE_LISTBOX_COMMAND);

    if (rs->action == XfwfMultiListActionDClick
    &&  lbox->allow_dclicks) {
	// event.eventType = wxEVENT_TYPE_LISTBOX_ACTION_COMMAND;
	// double click invokes OnDefaultAction
	lbox->GetParent()->GetEventHandler()->OnDefaultAction(lbox);
	return;
    }

    event.commandInt    = lbox->GetSelection();
    if (event.commandInt > -1) {
	event.commandString = lbox->GetString(event.commandInt);
	event.clientData    = lbox->GetClientData(event.commandInt);
    } else {
	event.commandString = NULL;
	event.clientData    = NULL;
    }
    event.extraLong     = (rs->action==XfwfMultiListActionHighlight ||
			   rs->action==XfwfMultiListActionDClick);
    event.eventObject   = lbox;

    int max_select;
    XtVaGetValues(lbox->HWidget(), XtNmaxSelectable, &max_select, NULL);
    if (max_select==1 
    && (!event.extraLong || event.commandInt <= -1))
	return;

    lbox->ProcessCommand(event);
}
