/* -*- C++ -*-
 * $Id: APP_thread.cpp,v 1.4 1997/05/01 21:33:46 wg Exp $
 *
 * Purpose:   Basic thread support for wxWindows.
 * Author:    Wolfram Gloger
 * Copyright: (C) 1996, 1997 Wolfram Gloger
 * License:   GNU Library General Public License
 */

#ifdef __GNUG__
#ifdef wx_xt
#pragma implementation "APP_thread.h"
#else
#pragma implementation "wx_thrd.h"
#endif
#endif

#include <stdio.h>
#include <unistd.h>

#if 1
// for select()
#include <sys/time.h>
#include <sys/types.h>
#ifdef __sgi
#include <bstring.h>
#endif
#endif

#if defined(wx_xt)

#define  Uses_XtIntrinsic
#define  Uses_wxThread
#define  Uses_wxApp
#include "wx.h"

#elif defined(wx_motif)

#include "wx.h"
#include "wx_thrd.h"
#define wxAPP_CONTEXT (wxTheApp->appContext)

#else
#error "Sorry, only the `Xt port' and Motif is supported."
#endif

#if USE_THREADS_SGI
#include <signal.h>
#include <sys/wait.h>
#include <sys/prctl.h>
#endif

enum thread_state {
	STATE_IDLE = 0,
	STATE_RUNNING,
	STATE_CANCELED,
	STATE_EXITED
};

// Static variables
static int          thread_pipe[2] = { -1, -1 };
static XtWorkProcId thread_wpid;
static XtInputId    thread_inid;

#if USE_THREADS_POSIX

/******************************** Posix threads ****************************/

wxMutex::wxMutex()
{
	pthread_mutex_init(&mutex, NULL);
}

wxMutex::~wxMutex()
{
	pthread_mutex_destroy(&mutex);
}

class
wxThreadPrivate {
public:
	wxThreadPrivate() { state = STATE_IDLE; }
	~wxThreadPrivate() {}
	static void *PthreadStart(void *ptr);
	pthread_t thread_id;
	int state;
};

void*
wxThreadPrivate::PthreadStart(void *ptr)
{
	wxThread *thread = (wxThread *)ptr;

	void* status = thread->Entry();
	thread->Exit(status);
	return 0;
}

int
wxThread::Create()
{
	if(p->state != STATE_IDLE)
		return -1;

#if 0
	pthread_attr_t a;
	struct sched_param sp;
	pthread_attr_init(&a);
	pthread_attr_getschedparam(&a, &sp);
	sp.prio = PTHREAD_MIN_PRIORITY +
		(prio*(PTHREAD_MAX_PRIORITY-PTHREAD_MIN_PRIORITY))/100;
	pthread_attr_setschedparam(&a, &sp);
	pthread_create(&p->thread_id, &a,
				   wxThreadPrivate::PthreadStart, (void *)this);
	pthread_attr_destroy(&a);
#endif
	// this is the point of no return
	p->state = STATE_RUNNING;
	if(pthread_create(&p->thread_id, NULL,
					  wxThreadPrivate::PthreadStart, (void *)this) != 0) {
		p->state = STATE_IDLE;
		return -2;
	}
	//fprintf(stderr, "thread %lx created\n", (unsigned long)p->thread_id);
	return 0;
}

int
wxThread::Cancel()
{
	int res = 0;

	if(p->state == STATE_RUNNING) {
		//fprintf(stderr, "About to cancel thread...\n");
		res = pthread_cancel(p->thread_id);
		if(res == 0) p->state = STATE_CANCELED;
	}
	return res;
}

void*
wxThread::Join()
{
	void* status = 0;

	if(p->state != STATE_IDLE) {
		//fprintf(stderr, "Joining thread %ld...\n", (long)p->thread_id);
		Bool do_unlock = wxThreadIsMain();

		if(do_unlock)
			wxMainMutex.Unlock();
		pthread_join(p->thread_id, &status);
		if(do_unlock)
			wxMainMutex.Lock();
		p->state = STATE_IDLE;
	}
	return status;
}

unsigned long
wxThread::GetID()
{
	return (unsigned long)p->thread_id;
}

void
wxThread::Exit(void* status)
{
	wxThread* ptr = this;
	write(thread_pipe[1], &ptr, sizeof(ptr));
	p->state = STATE_EXITED;
	pthread_exit(status);
}

void
wxThread::TestCancel()
{
	pthread_testcancel();
}

static pthread_t main_id;

Bool
wxThreadIsMain()
{
	return (Bool)pthread_equal(pthread_self(), main_id);
}

#elif USE_THREADS_SGI

/**************************** SGI `sproc' threads *************************/

wxMutex::wxMutex()
{
	init_lock(&mutex);
}

wxMutex::~wxMutex()
{
}

class
wxThreadPrivate {
public:
	wxThreadPrivate() { thread_id = 0; state = STATE_IDLE; }
	~wxThreadPrivate() {}
	static void SprocStart(void *ptr);
	static void SignalHandler(int sig);
	int state, thread_id;
	void* exit_status;
};

#if 0
#define prda_this_ptr (*(wxThread **)(&PRDA->usr_prda))

void
wxThreadPrivate::SignalHandler(int)
{
	int count;

	//signal(SIGTERM, SignalHandler);
	count = prda_this_ptr->p->lock_count;
	prda_this_ptr->p->state = count>0 ? STATE_STOPPED : STATE_EXITED;
	if(count == 0) {
		fprintf(stderr, "exiting %d\n", prda_this_ptr->p->thread_id);
		_exit(0);
	}
	fprintf(stderr, "deferring %d\n", prda_this_ptr->p->thread_id);
}
#endif

void
wxThreadPrivate::SprocStart(void *ptr)
{
	void* status;

	wxThread *thr = (wxThread *)ptr;

#if 0
	prda_this_ptr = thr;
	//signal(SIGTERM, SignalHandler);
	struct sigaction sa;
	sa.sa_flags = SA_RESTART;
	sa.sa_handler = (void (*)(...))SignalHandler;
	sigfillset(&sa.sa_mask);
	sigaction(SIGTERM, &sa, 0);
#endif

	thr->p->thread_id = getpid();
	thr->p->exit_status = 0;
	status = thr->Entry();
	thr->Exit(status);
}

void
wxThread::Exit(void* status)
{
	wxThread* ptr = this;
	write(thread_pipe[1], &ptr, sizeof(ptr));
	p->state = STATE_EXITED;
	p->exit_status = status;
	_exit(0);
}

int
wxThread::Create()
{
	if(p->state != STATE_IDLE)
		return -1;
	p->state = STATE_RUNNING;
	if(sproc(p->SprocStart, PR_SALL, this) < 0) {
		p->state = STATE_IDLE;
		return -2;
	}
	return 0;
}

int
wxThread::Cancel()
{
	if(p->state == STATE_RUNNING) {
		//fprintf(stderr, "About to kill thread...\n");
		p->state = STATE_CANCELED;
		return 0;
	}
	return -1;
}

void*
wxThread::Join()
{
	if(p->state != STATE_IDLE) {
		//fprintf(stderr, "Joining thread %d...\n", (int)p->thread_id);
		Bool do_unlock = wxThreadIsMain();
		int stat;

		if(do_unlock)
			wxMainMutex.Unlock();
		waitpid(p->thread_id, &stat, 0);
		if(do_unlock)
			wxMainMutex.Lock();
		if(!WIFEXITED(stat) && !WIFSIGNALED(stat))
			return 0;
		p->state = STATE_IDLE;
		return p->exit_status;
	}
	return 0;
}

unsigned long
wxThread::GetID()
{
	return (unsigned long)p->thread_id;
}

void
wxThread::TestCancel()
{
	if(p->state == STATE_CANCELED) {
		p->exit_status = 0;
		_exit(0);
	}
}

static int main_id;

Bool
wxThreadIsMain()
{
	return (int)getpid() == main_id;
}

#else /* !USE_THREADS_* */

// No thread support available.

wxMutex::wxMutex() {}
wxMutex::~wxMutex() {}

struct wxThreadPrivate {
	int thread_id;
	void* exit_status;
};

int
wxThread::Create()
{
	p->exit_status = Entry();
	return 0;
}

int
wxThread::Cancel()
{
	return -1;
}

void
wxThread::TestCancel()
{
}

void*
wxThread::Join()
{
	return p->exit_status;
}

unsigned long
wxThread::GetID()
{
	return 0;
}

Bool
wxThreadIsMain()
{
	return TRUE;
}

#endif /* USE_THREADS_* */

/**************************** Common code *********************************/

wxMutex wxMainMutex; // controls access to all GUI functions

wxThread::wxThread()
{
	p = new wxThreadPrivate();
	//fprintf(stderr, "new wxThread %p\n", this);
}

wxThread::~wxThread()
{
	Cancel();
	Join();
	delete p;
}

// The default callback just joins the thread and throws away the result.
void
wxThread::OnExit()
{
	Join();
}

/* We use a workproc to unlock the main lock and sleep when nothing
   else is running.  This workproc is registered first (in
   wxThreadInit(), which is called from wxCommonInit()), so that all
   other user-defined workprocs get a chance to run _before_ this
   one. */

#if 1

/* This alternative is ugly and more complicated than just using
   _XtWaitForSomething() (see below).  But strictly, it is necessary
   because there are races when dropping the main lock before
   _XtWaitForSomething() is entered. */

struct InternalXtApp { // from xc/lib/Xt/InitialI.h
	void*           next;
	void*           process;
	void*           destroy_callbacks;
	Display**       list;
	struct timeval* timerQueue;
	void*           workQueue;
	void**          input_list;
	void*           outstandingQueue;
	// etc. (further elements omitted)
};

static Boolean
ThreadWorkProc(XtPointer)
{
	int allow_exit = 1;
	Display* disp = wxGetDisplay();
	int fd_conn, fd_pipe;
	struct timeval cur_time, dif_time, * sel_time = 0;

	fd_conn = XConnectionNumber(disp);
	fd_pipe = thread_pipe[0];

	// Process timeout
	InternalXtApp* app = (InternalXtApp*)wxAPP_CONTEXT;
	if(app->timerQueue != 0) {
		gettimeofday(&cur_time, 0);
		dif_time.tv_usec = app->timerQueue->tv_usec - cur_time.tv_usec;
		dif_time.tv_sec = app->timerQueue->tv_sec - cur_time.tv_sec;
		if(dif_time.tv_usec < 0) {
			dif_time.tv_usec += 1000000;
			dif_time.tv_sec -= 1;
		}
		if(dif_time.tv_sec < 0)
			return False;
		//fprintf(stderr, "t%ld %ld\n", dif_time.tv_sec, dif_time.tv_usec);
		sel_time = &dif_time;
	}

	if(XPending(disp) == 0) {
	    int fd_max, sel_res;
	    fd_set read_set;

	    FD_ZERO(&read_set);
	    FD_SET(fd_conn, &read_set);
		fd_max = fd_conn+1;
		if(allow_exit) {
			FD_SET(fd_pipe, &read_set);
			if(fd_pipe+1 > fd_max) fd_max = fd_pipe+1;
		}
#if 0 // Does anyone know in what case this could be necessary ?
	    int *fd_list, fd_count;
	    if(!XInternalConnectionNumbers(disp, &fd_list, &fd_count))
		fprintf(stderr, "Can't obtain connection numbers\n");
	    for(int i=0; i<fd_count; i++) {
		FD_SET(fd_list[i], &read_set);
		if(fd_list[i] > fd_max-1)
		    fd_max = fd_list[i]+1;
	    }
	    XFree(fd_list);
#endif
	    wxMainMutex.Unlock();
	    sel_res = select(fd_max, &read_set, NULL, NULL, sel_time);
	    wxMainMutex.Lock();
#if 0
	    if(allow_exit && FD_ISSET(fd_pipe, &read_set)) {
			wxThread* ptr;
			if(read(fd_pipe, &ptr, sizeof(ptr)) == sizeof(ptr)) {
				//fprintf(stderr, "calling OnExit %p\n", ptr);
				ptr->OnExit();
			} else {
				//fprintf(stderr, "this should never happen\n");
			}
			if(sel_res > 1) break;
	    } else break;
#endif
	}
	return False; // never delete this procedure
}

#else

// Internal X toolkit function; very useful here.
extern "C" int _XtWaitForSomething(XtAppContext, int, int, int, int, int,
#ifdef XTHREADS
								   int,
#endif
								   unsigned long*);

static Boolean
ThreadWorkProc(XtPointer)
{
	wxMainMutex.Unlock();
	_XtWaitForSomething(wxAPP_CONTEXT, 0, 0, 0, 0, 1,
#ifdef XTHREADS
						1,
#endif
						0);
	wxMainMutex.Lock();
	return False; // never delete this procedure
}

#endif /* 1 */

/* ThreadExitProc is called as an X11 input handler whenever a thread
   exits, having written a pointer to its wxThread instance to
   thread_pipe. */
static void
ThreadExitProc(XtPointer, int* fd, XtInputId*)
{
	wxThread* ptr;

	if(*fd != thread_pipe[0])
		return;
	if(read(*fd, &ptr, sizeof(ptr)) == sizeof(ptr)) {
		//fprintf(stderr, "calling OnExit %p\n", ptr);
		ptr->OnExit();
	} else {
		//fprintf(stderr, "this should never happen\n");
	}
}

// Global initialization
static void
wxThreadInit(void*)
{
#if USE_THREADS_POSIX || USE_THREADS_SGI
	pipe(thread_pipe);
	thread_wpid = XtAppAddWorkProc(wxAPP_CONTEXT, ThreadWorkProc, 0);
	thread_inid = XtAppAddInput(wxAPP_CONTEXT, thread_pipe[0],
								(XtPointer)XtInputReadMask,
								ThreadExitProc, 0);
#endif
#if USE_THREADS_POSIX
	main_id = pthread_self();
#elif USE_THREADS_SGI
	main_id = (int)getpid();
#endif
	wxMainMutex.Lock();
	//fprintf(stderr, "ThreadInit\n");
}

// Global cleanup
static void
wxThreadExit(void*)
{
	wxMainMutex.Unlock();
#if USE_THREADS_POSIX || USE_THREADS_SGI
	XtRemoveInput(thread_inid);
	XtRemoveWorkProc(thread_wpid);
	close(thread_pipe[0]);
	close(thread_pipe[1]);
#endif
	//fprintf(stderr, "ThreadExit\n");
}

// Let automatic initialization be performed from wxCommonInit().
static struct
wxThreadGlobal {
	wxThreadGlobal() { wxRegisterModule(wxThreadInit, wxThreadExit); }
} dummy;

/*
 * Local variables:
 * tab-width: 4
 * End:
 */
