#
# TCL Library for tkCVS
#

#
# $Id: cvs.tcl,v 1.41.2.31 1999/10/28 03:57:07 dorothyr Exp $
# 
# Contains procedures used in interaction with CVS.
#

proc cvs_notincvs {} {
  cvsfail "This directory is not in CVS.\nPlease import it first."
}

proc cvs_incvs {} {
  cvsfail "You can\'t do that here because this directory is already in CVS."
}

proc cvs_remote_bad {} {
  cvsfail "You can\'t do that with a remote CVS repository."
}

proc cvs_remove args {
#
# This deletes a file from the directory and the repository,
# asking for confirmation first.
#
  global cvs
  global incvs
  global cvscfg

  if {$args == "."} {
    cvsfail "Please select some files to delete first!"
    return
  }

  set mess ""
  if {$incvs} {
    set mess "WARNING!!! This will modify"
    append mess " the contents of your local,"
    append mess " working directory AND"
    append mess " the CVS repository!"
  }
  append mess "\n\nYou are about to remove these"
  append mess " files:"
  if { [ are_you_sure $mess $args ] == 1 } {
    set results ""
    set exec_idx [exec_command_init "CVS Delete"]
    foreach file $args {
      exec_command_body $exec_idx "$cvscfg(rm_cmd) $cvscfg(rm_flags) $file"
      if {$incvs} {
        exec_command_body $exec_idx "$cvs -d $cvscfg(cvsroot) remove $file"
      }
    }
    exec_command_end $exec_idx
    if {$cvscfg(auto_status) == "true"} {
      setup_dir
    }
  }
}

proc cvs_add args {
#
# This adds a file to the repository.
#
  global cvs
  global cvscfg
  global incvs

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  if {$args == "."} {
    set mess "This will add all new files!"
  } else {
    set mess "This will add these files:\n\n"
    foreach file $args {
      append mess "   $file\n"
    }  
  }
  append mess "\nAre you sure?"
  if {[cvsconfirm $mess] == 0} {
    set results ""
    set exec_idx [exec_command_init "CVS Add"]
    if {$args == "."} {
      foreach file [glob -nocomplain $cvscfg(aster)] {
        exec_command_body $exec_idx "$cvs -d $cvscfg(cvsroot) add $file"
      }
    } else {
      foreach file $args {
        exec_command_body $exec_idx "$cvs -d $cvscfg(cvsroot) add $file"
      }
    }
    exec_command_end $exec_idx
    if {$cvscfg(auto_status) == "true"} {
      setup_dir
    }
  }
}

proc cvs_add_binary args {
#
# This adds a binary file to the repository, using the -kb option to rcs.
# Probably a pop-up box would be better, or even for the underneath code
# to have an additional argument to cvs_add instead of a separate subroutine.
# But this seemed easiest at the time.
# This does not check to make sure the user is using RCS 5.7 or greater.
#
  global cvs
  global cvscfg
  global incvs

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  if {$args == "."} {
    set mess "This will add all new files!"
  } else {
    set mess "This will add these files:\n\n"
    foreach file $args {
      append mess "   $file\n"
    }  
  }
  append mess "\nAre you sure?"
  if {[cvsconfirm $mess] == 0} {
    set exec_idx [exec_command_init "CVS Add Binary"]
    if {$args == "."} {
      foreach file [glob -nocomplain $cvscfg(aster)] {
        exec_command_body $exec_idx "$cvs -d $cvscfg(cvsroot) add -kb $file"
      }
    } else {
      foreach file $args {
        exec_command_body $exec_idx "$cvs -d $cvscfg(cvsroot) add -kb $file"
      }
    }
    exec_command_end $exec_idx
    if {$cvscfg(auto_status) == "true"} {
      setup_dir
    }
  }
}

proc cvs_diff args {
#
# This diffs a file with the repository.
#
  global cvs
  global cvscfg
  global incvs

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  if {$args == "."} {
    cvsfail "Please select one or more files to compare!"
  } else {
    foreach file $args {
      catch {eval "exec $cvscfg(tkdiff) $file &"} view_this
    }
  }
}

proc cvs_diff_r {rev1 rev2 args} {
#
# This diffs a file with the repository, using two revisions or tags.
#
  global cvs
  global cvscfg
  global incvs
 
  #if {! $incvs} {
  #  cvs_notincvs
  #  return 1
  #}
  if {$rev1 == {} || $rev2 == {}} {
    cvsfail "Must have two revision numbers for this function!"
    return 1
  }
 
  foreach file $args {
    catch {eval "exec $cvscfg(tkdiff) -r$rev1 -r$rev2 $file &"} view_this
  }
}

proc cvs_view_r {rev args} {
#
# This views a specific revision of a file in the repository.
#
  global cvs
  global incvs
  global cvscfg
 
  #if {! $incvs} {
    #cvs_notincvs
    #return 1
  #}
 
  if {$args == "."} {
    foreach file [glob -nocomplain $cvscfg(aster)] {
      catch {eval "exec $cvs -d $cvscfg(cvsroot) update -p \
      -r $rev $file 2>$cvscfg(null)"} view_this
      ### exec_command: problems to handle stdout/stderr seperately
      view_output "CVS View" $view_this
    }
  } else {
    foreach file $args {
      catch {eval "exec $cvs -d $cvscfg(cvsroot) update -p \
      -r $rev $file 2>$cvscfg(null)"} view_this
      ### exec_command: problems to handle stdout/stderr seperately
      view_output "CVS View" $view_this
    }
  }
}

proc cvs_logcanvas args {
#
# This looks at a log from the repository.
#
  global cvs
  global incvs
  global cvscfg

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  #puts "cvs_logcanvas {$args}"
  if {$args == "."} {
    cvsfail "Please select one or more files!"
    return
  }

  foreach file $args {
    set ret [catch {eval \
        "exec $cvs -d $cvscfg(cvsroot) -l -n log -l $file" \
      } view_this]
    if {$ret} {
      cvsfail $view_this
      return
    }
    # New style file log viewer
      set lcv [new_logcanvas $file $view_this]
  }
}

proc cvs_log args {
#
# This looks at a log from the repository.
# This is an old style file log with just the output from cvs log.
#
  global cvs
  global incvs
  global cvscfg

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  if {$args == ""} {
    set filelist "."
  } else {
    set filelist $args
  }
  # modified to allow for varying level of detail
  # -sj
  if {$cvscfg(cvsver) < 1.3} {
    set cvscfg(ldetail) "verbose"
  }
  set awkCmdFileName ""
  set awkCmd "BEGIN {LM=-1000}\n \$0 ~ /----------------------------/"
  append awkCmd " {LM=0; printf(\"\\n--------------------\\n\");}"
  append awkCmd " \$0 ~ /\\=\\=\\=\\=\\=\\=\\=\\=\\=\\=\\=\\=\\=\\="
  append awkCmd "\\=\\=\\=\\=\\=\\=\\=\\=\\=\\=\\=\\=\\=\\=/ "
  append awkCmd " {LM=0; printf(\"\\n\");}"
  append awkCmd " {if (LM \=\= 1) printf(\"%s \", \$0);"
  append awkCmd " if (LM \>\= 3) printf(\"%s \", \$0); LM\+\+;}"
  if { $cvscfg(awkCmdOverFile) } {
    set pid [pid]
    set awkCmdFileName "$cvscfg(tmpdir)/awkCmd-$pid"
    set awkFile [open $awkCmdFileName w]
    puts $awkFile $awkCmd
    close $awkFile
    set awkparms "-f $awkCmdFileName"
  } else {
    set awkparms "{$awkCmd}"
  }

  if { $cvscfg(ldetail) == "verbose" } {
    set commandline "$cvs -d $cvscfg(cvsroot) log $filelist"
  } elseif { $cvscfg(ldetail) == "last" || $cvscfg(ldetail) == "summary"} {
    set commandline "$cvs -d $cvscfg(cvsroot) -l -n log -l $filelist | $cvscfg(awk) $awkparms | tail \+2"
    if {$cvscfg(ldetail) == "last" } {
      append commandline " | head \-2 "
    }
  }
  #puts $commandline
  exec_command "CVS Log" $commandline

  if { $awkCmdFileName != "" } {
    eval exec $cvscfg(rm_cmd) $cvscfg(rm_flags) $awkCmdFileName
  }
}

proc cvs_commit {revision comment args} {
#
# This commits changes to the repository.
#
# The parameters work differently here -- args is a list.  The first
# element of args is a list of file names.  This is because I can't
# use eval on the parameters, because comment contains spaces.
#
  global cvs
  global cvscfg
  global incvs
  global filelist

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  if {$comment == ""} {
    cvsfail "You must enter a comment!"
    return 1
  }
  regsub -all "\"" $comment "\\\"" comment

  if {$args == ""} {
    set filelist "."
    if {$cvscfg(cvsver) < 1.3} {
      # cosmetic cleanup
      # -sj
      set errstr "You must select files to be committed with CVS version"
      append errstr " $cvscfg(cvsver)."
      append errstr "\n\nEither select a list of files or upgrade"
      append errstr " your CVS to version 1.3"
      cvsfail "$errstr"
      return
    }
  } else {
    set filelist [lindex $args 0]
  }

  # changed the message to be a little more explicit.
  # -sj
  set commit_output ""
  if { $filelist == "." } {
    set mess "This will commit your changes to ** ALL ** files in"
    append mess " and under this directory."
  } else {
    foreach file $filelist {
      set commit_output "$commit_output\n$file"
    }
    set mess "This will commit your changes to:$commit_output"
  }
  append mess "\n\nAre you sure?"
  set commit_output ""
  if {[cvsconfirm $mess] == 0} {
    set exec_idx [exec_command_init "CVS Commit"]
    if {$revision != ""} {
      if {$cvscfg(cvsver) < 1.3} {
        foreach file $filelist {
          exec_command_body $exec_idx \
            "$cvs -d $cvscfg(cvsroot) commit -f -r $revision -m \"$comment\" $file"
        }
      } else {
        foreach file $filelist {
          exec_command_body $exec_idx \
            "$cvs -d $cvscfg(cvsroot) commit -r $revision -m \"$comment\" $file"
        }
      }
    } else {
      if {$cvscfg(cvsver) < 1.3} {
        foreach file $filelist {
          exec_command_body $exec_idx \
            "$cvs -d $cvscfg(cvsroot) commit -f -m \"$comment\" $file"
        }
      } else {
        foreach file $filelist {
          exec_command_body $exec_idx \
            "$cvs -d $cvscfg(cvsroot) commit -m \"$comment\" $file"
        }
      }
    }
    exec_command_end $exec_idx
    if {$cvscfg(auto_status) == "true"} {
      setup_dir
    }
  }
}

proc cvs_tag {tagname forceflag branch args} {
#
# This tags a file in a directory.
#
  global cvs
  global cvscfg
  global incvs
  global filelist

  #puts "cvs_tag {$tagname $forceflag $branch $args}"
  if {$cvscfg(cvsver) < 1.3} {
    set mess "This function is not supported in CVS version $cvscfg(cvsver)."
    append mess "\n\nPlease upgrade your CVS version to 1.3 or greater"
    cvsfail $mess
    return
  }

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  if {$tagname == ""} {
    cvsfail "You must enter a tag name!"
    return 1
  }

  if {$cvscfg(cvsver) == 1.3} {
    set forceflag ""
  }

  if {$args == ""} {
    set filelist "."
  } else {
    set filelist $args
  }

  set exec_idx [exec_command_init "CVS Tag"]
  if {$branch == "yes"} {
    # Make the branch then update so we're on the branch
    #puts "$cvs -d $cvscfg(cvsroot) tag $forceflag -b $tagname $filelist"
    exec_command_body $exec_idx  \
      "$cvs -d $cvscfg(cvsroot) tag $forceflag -b $tagname $filelist"
    exec_command_body $exec_idx \
      "$cvs -d $cvscfg(cvsroot) update -r $tagname $filelist"
    #puts "$cvs -d $cvscfg(cvsroot) update -r $tagname $filelist"
  } else {
    exec_command_body $exec_idx \
      "$cvs -d $cvscfg(cvsroot) tag $forceflag $tagname $filelist"
    #puts "$cvs -d $cvscfg(cvsroot) tag $forceflag $tagname $filelist"
  }
  exec_command_end $exec_idx
  if {$cvscfg(auto_status) == "true"} {
    setup_dir
  }
}

proc cvs_update {tagname normal_binary action_if_no_tag get_all_dirs dir args} {
#
# This updates the files in the current directory.
#
  global cvs
  global cvscfg
  global incvs

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  if {$args == ""} {
    set filelist "."
  } else {
    set filelist $args
  }

  if { $normal_binary == "Normal" } {
      set mess "Using normal (text) mode.\n"
  } elseif { $normal_binary == "Binary" } {
      set mess "Using binary mode.\n"
  } else {
      set mess "Unknown mode:  $normal_binary\n"
  }

  if { $tagname != "BASE"  && $tagname != "HEAD" } {
      append mess "\nIf a file does not have tag $tagname"
      if { $action_if_no_tag == "Remove" } {
          append mess " it will be removed from your local directory.\n"
      } elseif { $action_if_no_tag == "Get_head" } {
          append mess " the head revision will be retrieved.\n"
      } elseif { $action_if_no_tag == "Skip" } {
          append mess " it will be skipped.\n"
      }
  }

  if { $tagname == "HEAD" } {
    append mess "\nYour local files will be updated to the"
    append mess " latest main trunk (head) revision."
    append mess " CVS will try to preserve any local, un-committed changes.\n"
  }

  append mess "\nIf there is a directory in the repository"
  append mess " that is not in your local, working directory,"
  if { $get_all_dirs == "Yes" } {
    append mess " it will be checked out at this time.\n"
  } else {
    append mess " it will not be checked out.\n"
  }

  if { $filelist == "." } {
    append mess "\nYou are about to download from"
    append mess " the repository to your local"
    append mess " filespace ** ALL ** files which"
    append mess " have changed in it."
  } else {
    append mess "\nYou are about to download from"
    append mess " the repository to your local"
    append mess " filespace these files which"
    append mess " have changed:\n"
  
    foreach file $filelist {
      append mess "\n\t$file"
    }
  }
  append mess "\n\nAre you sure?"
  if {[cvsconfirm $mess] == 0} {
    set str [join $filelist " " ]
    # modified by jo to build the commandline incrementally
    set commandline "$cvs update -P"
    if { $normal_binary == "Binary" } {
      append commandline " -kb"
    }
    if { $get_all_dirs == "Yes" } {
      append commandline " -d $dir"
    }
    if { $tagname != "BASE" && $tagname != "HEAD" } {
      if { $action_if_no_tag == "Remove" } {
          append commandline " -r $tagname"
      } elseif { $action_if_no_tag == "Get_head" } {
          append commandline " -f -r $tagname"
      } elseif { $action_if_no_tag == "Skip" } {
          append commandline " -s -r $tagname"
      }
    }
    if { $tagname == "HEAD" } {
      append commandline " -A"
    }
    append commandline " $str"

    exec_command "CVS Update" $commandline
    if {$cvscfg(auto_status) == "true"} {
      setup_dir
    }
  }
}

proc cvs_join {localfile branchver} {
#
# This does a join (merge) of the branchver revision of localfile to the
# head revision.
#
  global cvs
  global cvscfg
  global incvs

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  set mess "This will merge revision $branchver to"
  append mess " the head revision of $localfile"
  append mess "\n\nAre you sure?"
  if {[cvsconfirm $mess] == 0} {
    exec_command "CVS Merge" \
      "$cvs -d $cvscfg(cvsroot) update -j$branchver $localfile"
    if {$cvscfg(auto_status) == "true"} {
      setup_dir
    }
  }
}

proc cvs_delta {localfile ver1 ver2} {
#
# This merges the changes between ver1 and ver2 into the head revision.
#
  global cvs
  global cvscfg
  global incvs

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  if {$ver1 == {} || $ver2 == {}} {
    cvsfail "Must have two revision numbers for this function!"
    return 1
  }
  set mess "This will merge the changes between revision $ver1 and $ver2"
  append mess " (if $ver1 > $ver2 the changes are removed)"
  append mess " to the head revision of $localfile"
  append mess "\n\nAre you sure?"
  if {[cvsconfirm $mess] == 0} {
    exec_command "CVS Merge" \
      "$cvs -d $cvscfg(cvsroot) update -j$ver1 -j$ver2 $localfile"
    if {$cvscfg(auto_status) == "true"} {
      setup_dir
    }
  }
}

proc cvs_status args {
#
# This does a status report on the files in the current directory.
#
  global cvs
  global incvs
  global cvscfg

  set global_options ""
  set cmd_options ""

  # if there are selected files, I want verbose output for those files
  # so I'm going to save the current setting here
  # - added by Jo
  set verbosity_setting ""

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  # if recurse option is false or there are selected files, don't recurse
  if { ( $cvscfg(recurse) == "false" ) ||
       ( ( $args != {} ) && ( $args != "." ) ) } { 
    set cmd_options "$cmd_options -l"
  }

  # if there are selected files, use verbose output
  # but save the current setting so it can be reset
  # - added by Jo
  if { ( $args != {} ) && ( $args != "." ) } {
    set verbosity_setting $cvscfg(rdetail)
    set cvscfg(rdetail) "verbose"
  }

  if {$args == ""} {
    if { $cvscfg(no_dot) == 1 } {
      set filelist ""
    } else {
      set filelist "."
    }
  } else {
    set filelist $args
  }

  # Additional support added for 1.4
  # -sj
  if {$cvscfg(cvsver) < 1.3} {
    set cmd_options ""
    set cvscfg(rdetail) "verbose"
  } elseif { $cvscfg(cvsver) < 1.4} {
    set global_options "$global_options -q"
  } else {
    set global_options "$global_options -Q"
  }
  # added to support verious levels of verboseness. Ideas derived and some
  # of the awk expressions derived from GIC.
  # -sj
  set awkCmdFileName ""
  set commandline "$cvs -d $cvscfg(cvsroot) $global_options status $cmd_options $filelist"
  if { $cvscfg(rdetail) == "summary" || $cvscfg(rdetail) == "terse" } {
    if { $cvscfg(rdetail) == "summary" } {
      set awkCmd "\$3 ~ /Status:/ "
      append awkCmd "{printf(\"%s %s %s %s \\t%s\\n\", "
      append awkCmd "\$4, \$5, \$6, \$7, \$2)}"
    } else {
      # terse
      set awkCmd "\$3 ~ /Status:/ {if (\$4 != \"Up-to-date\") "
      append awkCmd "{printf(\"%s %s %s %s \\t%s\\n\", "
      append awkCmd "\$4, \$5, \$6, \$7, \$2)}}"
    }
    if { $cvscfg(awkCmdOverFile) } {
      set pid [pid]
      set awkCmdFileName "$cvscfg(tmpdir)/awkCmd-$pid"
      set awkFile [open $awkCmdFileName w]
      puts $awkFile $awkCmd
      close $awkFile
      append commandline " | $cvscfg(awk) -f $awkCmdFileName"
    } else {
      append commandline " | $cvscfg(awk) {$awkCmd}"
    }
  }

  #puts $commandline
  exec_command "CVS Status" "$commandline" 0 "Everything is Up-to-date."

  if { $awkCmdFileName != "" } {
    eval exec $cvscfg(rm_cmd) $cvscfg(rm_flags) $awkCmdFileName
  }

  # reset the verbosity setting if necessary -jo
  if { $verbosity_setting != "" } {
    set cvscfg(rdetail) $verbosity_setting
  }
}


proc cvs_tag_status args {
#
# This grep through the output of 'cvs status' to provide a simplistic
# report of the current tags on files
#
  global cvs
  global incvs
  global cvscfg

  set global_options ""
  set cmd_options ""

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  set cmd_options "$cmd_options -l"

  if {$args == ""} {
    set filelist "."
  } else {
    set filelist $args
  }

  if {$cvscfg(cvsver) < 1.3} {
    set cmd_options ""
    set cvscfg(rdetail) "verbose"
  } elseif { $cvscfg(cvsver) < 1.4} {
    set global_options "$global_options -q"
  } else {
    set global_options "$global_options -Q"
  }

  set awkCmdFileName ""
  set commandline "$cvs -d $cvscfg(cvsroot) $global_options status $cmd_options $filelist "
  set awkCmd "\$0 ~ /File:/ || \$0 ~ /Sticky / {print}"
  if { $cvscfg(awkCmdOverFile) } {
    set pid [pid]
    set awkCmdFileName "$cvscfg(tmpdir)/awkCmd-$pid"
    set awkFile [open $awkCmdFileName w]
    puts $awkFile $awkCmd
    close $awkFile
    append commandline " | $cvscfg(awk) -f $awkCmdFileName"
  } else {
    append commandline " | $cvscfg(awk) {$awkCmd}"
  }
  #puts $commandline
  exec_command "CVS Sticky Status" "$commandline" 0 "Everything is Up-to-date."

  if { $awkCmdFileName != "" } {
    eval exec $cvscfg(rm_cmd) $cvscfg(rm_flags) $awkCmdFileName
  }
}

proc format_check_msg {file msg} {
  return [format "%-40s: %s" $file $msg]
}

proc cvs_check_filter_proc {line} {
#
# This filter annotates each line of cvs_check output
#
  global cvscfg

  regexp {^([UARMC?]) (.*)} $line junk mode file
  if {[info exists mode]} {
    switch -exact -- $mode {
      U {
    	set new_line [format_check_msg $file \
                  "file changed in repository, needs updating"]
      }
      A {
    	set new_line [format_check_msg $file \
                  "file added, not committed"]
      }
      R {
    	set new_line [format_check_msg $file \
                  "file removed, not committed"]
      }
      M {
    	set new_line [format_check_msg $file \
                  "file modified, not committed"]
      }
      C {
    	set new_line [format_check_msg $file \
                  "file modified and in conflict, not committed"]
      }
      ? {
    	# some samba drawbacks-workaround code
                # (always downcase letters: /cvs )
    	set start [string last "/$cvscfg(cvsdir)" $file]
    	if { [string range $file $start end] != "/$cvscfg(cvsdir)" } {
    	  if { $file != $cvscfg(cvsdir) } {
    		set new_line [format_check_msg $file \
                           "file unknown, not in CVS"]
    	  }
    	}
      }
      default {
    	set new_line $line
      }
    }
  } else {
    set new_line $line
  }
  return $new_line
}

proc cvs_check_eof_proc {} {
#
# This proc is called after cvs check is done, code is from cvscheck.tcl
#
# now find directories not added.  This is accomplished by finding all of
# the directories in the current directory seeing if there is a CVS
# control file in each one.
#
  global cvscfg

  set dir_lines ""
  set files [glob -nocomplain -- .??* *]
  set dirs {}
  foreach file $files {
    if {[file isdirectory $file] && "$file" != "$cvscfg(cvsdir)"} {
      lappend dirs $file
    }
  }
  # see if there are any directories not added.
  if {[llength $dirs]} {
    foreach dir $dirs {
      if {! [file exists "$dir/$cvscfg(cvsdir)"] \
      || ! [file isdirectory "$dir/$cvscfg(cvsdir)"]} {
        append dir_lines \
          [format_check_msg $dir "directory unknown, not in CVS\n"]
      }
    }
  }
  return $dir_lines
}

proc cvs_check {} {
#
# This does a cvscheck on the files in the current directory.
#
  global cvs
  global incvs
  global cvscfg

  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  set cvscfg(exec_line_filter) "cvs_check_filter_proc"
  set cvscfg(exec_eof) "cvs_check_eof_proc"
  exec_command "CVS Check" \
      "$cvs -n -q update $cvscfg(checkrecursive)" \
      0 "Nothing to report."
  set cvscfg(exec_line_filter) ""
  set cvscfg(exec_eof) ""
}

proc cvs_checkout {mcode revision} {
  #
  # This checks out a new module into the current directory.
  #
  global cvs
  global cvscfg
  global incvs
  global feedback

  if {$incvs} {
    set mess "You are already in a CVS controlled directory.  Are you"
    append mess " sure that you want to check out another module in"
    append mess " to this directory?"
    if {[cvsconfirm $mess] == 1} {
      return 1
    }
  }

  set mess "This will check out $mcode from CVS.\nAre you sure?"
  if {[cvsconfirm $mess] == 0} {
    feedback_cvs $feedback(cvs) "Checking out module $mcode, please wait"
    if {$revision == {}} {
      exec_command "CVS Checkout" \
          "$cvs -d $cvscfg(cvsroot) checkout -P $mcode"
    } else {
      exec_command "CVS Checkout" \
          "cvs -d $cvscfg(cvsroot) checkout -P -r $revision $mcode"
    }
    feedback_cvs $feedback(cvs) ""
    if {$cvscfg(auto_status) == "true"} {
      setup_dir
    }
  }
}

proc cvs_filelog {mcode filename} {
#
# This looks at a revision log of a file from the repository.
#
  global cvs
  global location
  global cvscfg
  global cwd
  
  set pid [pid]
  #puts "cvs_filelog {$mcode $filename}"
  set file $location($mcode)/$filename
  set filetail [file tail $filename]
  
  # Big note: the temp directory fed to a remote servers's command line
  # needs to be seen by the server.  It can't cd to an absolute path.
  # In addition it's fussy about where you are when you do a checkout -d.
  # Best avoid that altogether.
  cd $cvscfg(tmpdir)
  if {! [file isdirectory  cvstmpdir.$pid]} {
    set ret [catch {eval exec mkdir cvstmpdir.$pid} view_this]
    if {$ret} {
      cvsfail $view_this
      cd $cwd
      return
    }
  }

  cd cvstmpdir.$pid

  #puts "$cvs -d $cvscfg(cvsroot) -l checkout $mcode/$filename"
  set ret [catch {eval "exec \
      $cvs -d $cvscfg(cvsroot) -l checkout $mcode/$filename" \
    } view_this]
  if {$ret} {
    cvsfail $view_this
    cd $cwd
    return
  }

  cd $mcode
  #puts "$cvs -d $cvscfg(cvsroot) -l -n log -l $filetail"
  set ret [catch {eval "exec \
      $cvs -d $cvscfg(cvsroot) -l -n log -l $filetail" \
    } view_this]
  if {$ret} {
    cvsfail $view_this
    cd $cwd
    return
  }
  cd $cwd

  # Log canvas viewer
  new_logcanvas $filetail $view_this
}

proc rcs_fileview {filename revision} {
#
# This views an RCS file in the repository.
# Called from Log Browser -> View
#
  global cvscfg

  if {$cvscfg(remote)} {
    cvs_remote_bad
    return 1
  }

  if {$revision == {}} {
    catch {exec co -p $filename 2>$cvscfg(null)} view_this
  } else {
    catch {exec co -p$revision $filename 2>$cvscfg(null)} view_this
  }
  view_output "CVS File View" $view_this
}

proc cvs_fileview {mcode filename revision} {
#
# This looks at a revision of a file from the repository.
# Called from Module Browser -> File Browse -> View
#
  global cvs
  global cvscfg

  #puts "cvs_fileview: {$mcode $filename $revision}"
  if {$revision == {}} {
    catch {eval "exec \
      $cvs -d $cvscfg(cvsroot) checkout -p $mcode/$filename \
      2>$cvscfg(null)"} view_this
  } else {
    catch {eval "exec \
      $cvs -d $cvscfg(cvsroot) checkout -p -r $revision $mcode/$filename \
      2>$cvscfg(null)"} view_this
  }
  view_output "CVS File View" $view_this
}

proc rcs_filediff {filename ver1 ver2} {
#
# This does a diff of an RCS file within the repository.
#
  global cvscfg

  if {$cvscfg(remote)} {
    cvs_remote_bad
    return 1
  }

  if {$ver1 == {} || $ver2 == {}} {
    cvsfail "Must have two revision numbers for this function!"
    return 1
  }
  # catch {exec rcsdiff -r$ver1 -r$ver2 $filename} view_this
  # view_output "CVS File Diff" $view_this
  catch {eval "exec $cvscfg(tkdiff) -r$ver1 -r$ver2 $filename &"} view_this
}


proc cvs_filediff {mcode filename ver1 ver2} {
#
# This looks at a diff of a file from the repository without
# checking it out.
#
  global cvscfg
  global location

  rcs_filediff $cvscfg(cvsroot)/$location($mcode)/$filename $ver1 $ver2
}

proc cvs_export {mcode revision} {
#
# This exports a new module (see man cvs and read about export) into
# the current directory.
#
  global cvs
  global incvs
  global cvscfg

  if {$incvs} {
    set mess "You are already in a CVS controlled directory.  Are you"
    append mess " sure that you want to export a module in"
    append mess " to this directory?"
    if {[cvsconfirm $mess] == 1} {
      return 1
    }
  }

  if {$cvscfg(cvsver) < 1.3} {
    cvsfail "This function is not supported in CVS version $cvscfg(cvsver).

Please upgrade your CVS to version 1.3"
    return
  }

  if {$revision == {}} {
    cvsfail "You must enter a tag name for this function."
    return
  }

  set mess "This will export $mcode from CVS.\nAre you sure?"
  if {[cvsconfirm $mess] == 0} {
    catch {exec $cvs -d $cvscfg(cvsroot) export -r $revision $mcode} view_this
    view_output "CVS Export" $view_this
    if {$cvscfg(auto_status) == "true"} {
      setup_dir
    }
  }
}

proc cvs_patch {mcode revision1 revision2} {
#
# This creates a patch file between two revisions of a module.  If the
# second revision is null, it creates a patch to the head revision.
#
  global cvs
  global cvscfg
 
  if {$revision1 == {}} {
    cvsfail "You must enter a tag name for this function."
    return
  }
 
  set mess "This will make a patch file for $mcode from CVS.\nAre you sure?"
  if {[cvsconfirm $mess] == 0} {
    if {$revision2 == {}} {
      catch {exec $cvs -d $cvscfg(cvsroot) patch -r $revision1 $mcode > $mcode.pat} view_this
      ###	exec_command: problems to handle stdout/stderr seperately
    } else {
      catch {exec $cvs -d $cvscfg(cvsroot) patch -r $revision1 -r $revision2 $mcode > $mcode.pat} \
        view_this
      ###	exec_command: problems to handle stdout/stderr seperately
    }
    set view_this "$view_this\n\nPatch file $mcode.pat created."
    view_output "CVS Patch" $view_this
    if {$cvscfg(auto_status) == "true"} {
      setup_dir
    }
  }
}

proc cvs_patch_summary {mcode revision1 revision2} {
#
# This creates a patch summary of a module between 2 revisions.
#
  global cvs
  global cvscfg
 
  if {$revision1 == {}} {
    cvsfail "You must enter a tag name for this function."
    return
  }
 
  if {$revision2 == {}} {
    exec_command "CVS Patch Summary" \
       "$cvs -d $cvscfg(cvsroot) patch -s -r $revision1 $mcode"
  } else {
    exec_command "CVS Patch Summary" \
       "$cvs -d $cvscfg(cvsroot) patch -s -r $revision1 -r $revision2 $mcode"
  }
}

proc cvs_version {} {
#
# This shows the current CVS version number.
#
  global cvs
  global cvscfg

  exec_command "CVS version" "$cvs -d $cvscfg(cvsroot) -v"
}

proc cvs_rtag {mcode branch tagnameA tagnameB} {
#
# This tags a module in the repository.
#
  global cvs
  global cvscfg

  if {$cvscfg(cvsver) < 1.3} {
    set command "tag"
  } else {
    set command "rtag"
  }

  set cmd_options ""

  if {$branch == "yes"} {
    if {$cvscfg(cvsver) < 1.3} {
      set mess "This function is not supported in CVS version $cvscfg(cvsver)."
      append mess "\n\nPlease upgrade your CVS version to 1.3 or greater"
      cvsfail $mess
      return
    }
    set cmd_options "-b"
  }

  # puts stderr "Using options for CVS version $cvscfg(cvsver)"
  if {$cvscfg(cvsver) > 1.3} {
    set cmd_options "$cmd_options -F"
  }

  if {$tagnameA == ""} {
    cvsfail "You must enter a tag name!"
    return 1
  }

  set mess "This will tag module \"$mcode\" in CVS with tag \"$tagnameA\"."
  if {$tagnameB == ""} {
    append mess "\n\nThe head revision of all files will be tagged."
  } else {
    append mess "\n\nThe revisions tagged with \"$tagnameB\" will be tagged."
    set cmd_options "$cmd_options -r $tagnameB"
  }
  append mess "\n\nAre you sure?"
  if {[cvsconfirm $mess] == 0} {
    exec_command "CVS Rtag" \
        "$cvs -d $cvscfg(cvsroot) $command $cmd_options $tagnameA $mcode"
  }
}

proc cvs_usercmd args {
#
# Allows the user to run a user-specified cvs command.
#
  global cvs

  exec_command "CVS $args" "$cvs $args"

  #catch {eval "exec $cvs $args"} view_this
  #view_output "CVS [lindex $args 0]" $view_this
}

proc cvs_anycmd {args} {
#
# Allows the user to run any user-specified command.
#
  # This won't really work because the command may expect
  # to open its own window -dr
  #exec_command "$args" "$args"

  catch {eval "exec $args"} view_this
  view_output [lindex $args 0] $view_this
}

#
# exec_command stuff is intended as a replacement for "view_output".
# 
# Enables to see the output while the command is running
# There are some places where view_output still refused to go: problems
# with stdout/err
#
proc bgerror {msg} {
  global errorInfo

  puts $errorInfo
}

#
# Set up a dialog containing a text box to view
# the report of the command during execution.
#
proc exec_command_init {title} {
  global exec_win

  static {exec_viewer 0}
  set my_idx $exec_viewer
  incr exec_viewer

  set exec_win($my_idx,lines) 0
  set exec_win($my_idx,win) ".exec$my_idx"
  set exec_win($my_idx,text) ".exec$my_idx.text"
  set exec_win($my_idx,scroll) ".exec$my_idx.scroll"
  set exec_win($my_idx,ok) ".exec$my_idx.ok"
  set exec_win($my_idx,destroyed) 0

  toplevel $exec_win($my_idx,win)
  text $exec_win($my_idx,text) -setgrid yes -relief sunken -border 2 \
        -yscroll "$exec_win($my_idx,scroll) set"
  scrollbar $exec_win($my_idx,scroll) -relief sunken \
        -command "$exec_win($my_idx,text) yview"
  button $exec_win($my_idx,ok) -text "Stop" \
      -command "stop_command 0 $my_idx" -state disabled
  pack $exec_win($my_idx,ok) -side bottom -fill x
  pack $exec_win($my_idx,scroll) -side right -fill y -padx 2 -pady 2
  pack $exec_win($my_idx,text) -fill both -expand 1
  wm title $exec_win($my_idx,win) "$title"
  return $my_idx
}

proc exec_command_body { my_idx command } {
  global exec_win

  if {$command == ""} {
    cvsfail "Nothing to execute."
  } else {
    if [catch {open "| $command |& cat"} exec_win($my_idx,log)] {
      $exec_win($my_idx,text) insert end $exec_win($my_idx,log)\n
      exec_command_end $my_idx
    } else {			
      set exec_win($my_idx,run) 1
      fileevent $exec_win($my_idx,log) readable "ins_exec_log_line $my_idx"
      $exec_win($my_idx,ok) configure -state normal
      # puts stderr "start wait ($my_idx)..."
      # puts stderr "exec_win($my_idx,run): $exec_win($my_idx,run)"
      tkwait variable exec_win($my_idx,run)
      # puts stderr "...end wait ($my_idx)"
      # puts stderr "exec_win($my_idx,run): $exec_win($my_idx,run)"
    }
  }
  update idletasks
}

proc exec_command_end { my_idx } {
  global exec_win

  if { $exec_win($my_idx,destroyed) == 0 } {
  	$exec_win($my_idx,ok) configure -text "Please wait..." -command ""
    update idletasks
    $exec_win($my_idx,ok) configure -text "Ok" -command "stop_command 1 $my_idx"
  }
}

proc exec_command {title command args} {
  global exec_win

  set my_idx [exec_command_init $title]
  set exec_win($my_idx,lines) 0
  exec_command_body $my_idx $command
  if {[llength $args] == 2} {
     set arg1 [lindex $args 0]
     set arg2 [lindex $args 1]
     if {$arg1 == 0} {
       # append if text empty
       if {$exec_win($my_idx,lines) == 0} {
         $exec_win($my_idx,text) insert end "$arg2\n"
       }
     } else {
       if {$arg1 == 1} {
        # always append
        $exec_win($my_idx,text) insert end "$arg2\n"
      }
    }
  }
  exec_command_end $my_idx
}

proc ins_exec_log_line { my_idx } {
  global exec_win cvscfg
    
  #puts stderr "ins_exec_log_line ($my_idx)"
  if [eof $exec_win($my_idx,log)] {
    catch {close $exec_win($my_idx,log)}
    # do some thing at end of program
    if { [info exists cvscfg(exec_eof)] } {
      if { $cvscfg(exec_eof) != "" } {
    	set line [$cvscfg(exec_eof)]
    	if { $line!="" } {
    	  $exec_win($my_idx,text) insert end $line
    	  $exec_win($my_idx,text) see end
    	  incr exec_win($my_idx,lines)
    	}
      }
    }
    set exec_win($my_idx,run) 0
    #puts stderr "end of pipe ($my_idx)!"
  } else {
    gets $exec_win($my_idx,log) line
    $exec_win($my_idx,text) insert end $line\n
    $exec_win($my_idx,text) see end
    if {$line!=""} {
      incr exec_win($my_idx,lines)
    }
  }
  update idletasks
}

proc stop_command {force my_idx } {
  global exec_win

  set mess "Really quit command?"
  if { $force == 1 || [tk_dialog .message {Confirm!} $mess warning 0 NO YES]} {
    catch {close $exec_win($my_idx,log)}
    if { $exec_win($my_idx,destroyed) == 0 } {
      set exec_win($my_idx,destroyed) 1
      destroy $exec_win($my_idx,win)
    }
  }
  update idletasks
}

proc view_output {title output_string} {
#
# Set up a dialog containing a text box that can be used to view
# the report on the screen.
#
  static {viewer 0}

  # If nothing to report, then say so.
  if {$output_string == ""} {
    set mess "Nothing to report."
    cvsok $mess
  } else {
    incr viewer
    set cvsview ".cvsview$viewer"
    toplevel $cvsview
    text $cvsview.text -setgrid yes -relief sunken -border 2 \
      -yscroll "$cvsview.scroll set"
    scrollbar $cvsview.scroll -relief sunken \
      -command "$cvsview.text yview"
    button $cvsview.ok -text "OK" \
      -command "destroy $cvsview"

    pack $cvsview.ok -side bottom -fill x
    pack $cvsview.scroll -side right -fill y -padx 2 -pady 2
    pack $cvsview.text -fill both -expand 1

    wm title $cvsview "$title Output"
    $cvsview.text insert end $output_string
    $cvsview.text configure -state disabled
  }
}

proc dbg_show {filelist} {
#
# to debug the selected filelist (of lists?)
#
  puts "checked filelist: \"$filelist\""
}

proc dbg_and_fix {filelist_by_ref} {
#
# to debug the selected filelist (of lists?)
#
  upvar $filelist_by_ref filelist

  if {[llength $filelist] == 1} {
    # strip the outer {}
    set filelist [lindex $filelist 0]
    # puts "stripped filelist: \"$filelist\""
  }
}


