package biss.jde;

import biss.FileLib;
import biss.ObserverSocket;
import biss.awt.Awt;
import biss.awt.DefaultIndexModel;
import biss.awt.GifProducer;
import biss.awt.GraphPane;
import biss.awt.Mouse;
import biss.awt.Prompter;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.ColorModel;
import java.awt.image.ImageConsumer;
import java.awt.image.ImageProducer;
import java.awt.image.IndexColorModel;
import java.awt.image.MemoryImageSource;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.PrintStream;
import java.util.Hashtable;
import java.util.Observable;
import java.util.Observer;
import java.util.StringTokenizer;

/**
 * logical view of a tool to graphically specify ClassImageSource derived
 * classes. This is some kind of a bitmap/icon editor which produces
 * Java classes rather than platform specific icon files.
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class ImageEditor
  implements Observer, CUObserver, ImageConsumer
{
	ImageViewer View;
	Dimension ImageDim = new Dimension( 16, 16);
	Color[][] Pels;
	CompileUnit CU;
	TypeDecl TD;
	Point PStart = new Point(0,0);
	Point PEnd = new Point(0,0);
	Color CurrentColor;
	Color[] GifIdxTable;

public ImageEditor(){
	View = new ImageViewer();
	dimChange( 16, 16);
	registerObserver();
	setCurrentColor( Color.black);
}

public ImageEditor( CompileUnit cu, TypeDecl t){
	CU = cu;
	TD = t;

	if ( isDesignable( t)) {
		View = new ImageViewer();
		View.setTitle( "ImageEditor: " + CU.MainTypeName);
		dimChange( 16, 16);
		registerObserver();
		setCurrentColor( Color.black);
	}
}

public ImageEditor( String pathName){
	try {
		byte[] buf = FileLib.getFileData( new File( pathName) );
		Parser p   = new Parser( pathName, buf);
		CU = p.parseCompileUnit();

		if ( isDesignable( CU.mainType()) ){
			View = new ImageViewer();
			View.setTitle( "ImageEditor: " + pathName);
			openOn( CU, CU.mainType() );
			registerObserver();
			setCurrentColor( Color.black);
		}
	}
	catch ( Throwable x ) { x.printStackTrace(); }

}

void circlePath () {

	int dx = PEnd.x - PStart.x;
	int dy = PEnd.y - PStart.y;
	double r = Math.sqrt( dx*dx + dy*dy);

	double teta = Math.PI / 180 * 5;
	double sinTet = Math.sin( teta);
	double cosTet = Math.cos( teta);

	double x = 0;
	double y = -r;

	Graphics g = View.PelPane.getGraphics();
	for ( double i=0; i<=2*Math.PI; i+=teta){
		int rx = (int)x + PStart.x;
		int ry = (int)y + PStart.y;
		try {
			if ( Pels[rx][ry] != CurrentColor) {
				Pels[rx][ry] = CurrentColor;
				redrawPel( rx, ry, g);
			}
		}
		catch( Throwable t) {}
		double xl = x;
		x = cosTet * x + sinTet * y;
		y = cosTet * y - sinTet * xl;
	}
	g.dispose();

}

void dimChange ( int x, int y) {
	Color[][] pca = Pels;
	Dimension pd = ImageDim;

	setDimensions( x, y);

	if ( pca != null) {
		for ( int cx=0; cx<pd.width; cx++){
			for ( int cy=0; cy<pd.height; cy++) {
				if ( cx<x && cy<y)
					Pels[cx][cy] = pca[cx][cy];
			}
		}
	}
	View.CXDim.setRange( 3, 300, x);
	View.CYDim.setRange( 3, 300, y);

	View.PelPane.redraw();
}

public boolean expandCU ( CompileUnit cu ) {
	if ( CU != cu ) return false;

	if ( cu.SrcFile != null )
		return expandSourceCU();
	//  return expandBytecodeCU();

	return false;
}

boolean expandSourceCU () {
	try {
		byte[] buf = FileLib.getFileData( CU.SrcFile);
		Parser p = new Parser( CU.FileName, buf);
		p.storeOutput();
		CompileUnit fullCu = p.parseCompileUnit();
		CU.expandFrom( fullCu);
		return true;
	}
	catch( ParseException e) { return false; }
}

public void imageComplete(int status) {
	dimChange( ImageDim.width, ImageDim.height);
	View.CPrevView.redraw();
}

boolean initCompileUnit () {
	try {
		ByteArrayOutputStream o = new ByteArrayOutputStream( 300);
		PrintStream s = new PrintStream( o);
		String sup = null;

		s.println( "import biss.awt.ClassImageSource;");
		s.println( "public class " + CU.MainTypeName);
		s.println( "  extends ClassImageSource");
		s.println( "{");

		s.println( "static byte[] PelValues;");
		s.println( "static int Width;");
		s.println( "static int Height;");

		s.println( "public " + CU.MainTypeName + "() {" );
		s.println( "  super( Width, Height, PelValues);" );
		s.println( "}" );

		s.println( "}" );

		Parser p = new Parser( o.toString() );
		CompileUnit cu = p.parseCompileUnit();
		CU.mergeTypes( cu.TypeDecls, true);
		CU.mergeImports( cu.Imports);
		TD = p.TDecl;

		s.close();
		o.close();

		return true;
	}
	catch( Throwable t) {
		t.printStackTrace();
		return false;
	}
}

void initFromString ( String s, int w, int h) {
	int x=-1, y=0;
	StringTokenizer st = new StringTokenizer( s, "{,\n}");

	Pels = new Color[w][h];
	ImageDim = new Dimension( w, h);

	while( st.hasMoreTokens() ){
		try {
			int idxVal = Integer.parseInt( st.nextToken() );
			if ( ++x >= w) {
				x = 0;
				y++;
			}
			Pels[x][y] = DefaultIndexModel.colorFor( idxVal);
		}
		catch( Throwable t) {
		}
	}

	View.PelPane.redraw();
	View.CXDim.setRange( 3, 300, w);
	View.CYDim.setRange( 3, 300, h);
}

boolean isDesignable( TypeDecl t ){
	if ( t == null)
		return true;
	if ( ! CU.IsExpanded) {
		if ( ! expandSourceCU() )
			return false;
	}
	if ( t instanceof ClassDecl ) {
		if ( t.hasData( "PelValues"))
			return true;
	}
	return false;
}

void linePath () {
	int dx = PEnd.x - PStart.x;
	int dy = PEnd.y - PStart.y;
	int xa = Math.abs( dx);
	int ya = Math.abs( dy);
	int x, y;

	if ( dx > 0){
		if ( dy > 0) {
			if ( ya <= xa ) { x = PStart.x; y = PStart.y; }
			else            { x = PStart.x; y = PStart.y; }
		}
		else {
			if ( ya <= xa ) { x = PStart.x; y = PStart.y; }
			else            { x = PEnd.x; y = PEnd.y; }
		}
	}
	else {
		if ( dy > 0) {
			if ( ya <= xa ) { x = PEnd.x; y = PEnd.y; }
			else            { x = PStart.x; y = PStart.y; }
		}
		else {
			if ( ya <= xa ) { x = PEnd.x; y = PEnd.y; }
			else            { x = PEnd.x; y = PEnd.y; }
		}
	}

	Graphics g = View.PelPane.getGraphics();

	if ( ya > xa ) {
		float ang = (float)dx / dy;
		for ( int yc=0; yc<=ya; yc++) {
			int nx = Math.round( ang * yc);
			Pels[x+nx][y+yc] = CurrentColor;
			redrawPel( x+nx, y+yc, g);
		}
	}
	else {
		float ang = (float)dy / dx;
		for ( int xc=0; xc<=xa; xc++) {
			int ny = Math.round( ang * xc);
			Pels[x+xc][y+ny] = CurrentColor;
			redrawPel( x+xc, y+ny, g);
		}
	}
	g.dispose();
}

public boolean loadGif( String pathName) {
	File f = new File( pathName);
	if ( f.exists() ){
		new GifProducer( f).startProduction( this);
		return true;
	}
	return false;
}

public static void main( String[] args) {
	if ( args.length > 0)
		new ImageEditor( args[0]);
	else
		new ImageEditor();
}

public boolean needsExpandedCU ( CompileUnit cu ) {
	return cu == CU;
}

void newImage () {
	for ( int y=0; y<ImageDim.height; y++) {
		for ( int x=0; x<ImageDim.width; x++)
			Pels[x][y] = null;
	}
	View.PelPane.redraw();
	View.CPrevView.redraw();
}

void openOn( CompileUnit cu, TypeDecl td ){
	CU = cu;
	TD = td;

	if ( CU.TypeDecls.size() == 0)
		initCompileUnit();

	try {
		initFromString( TD.dataAt( "PelValues").InitExpr,
		                Integer.parseInt( TD.dataAt( "Width").InitExpr),
		                Integer.parseInt( TD.dataAt( "Height").InitExpr) );
	}
	catch( Throwable t){
	}
}

void printImageOn (PrintStream ps, boolean transparent) {

	ps.print( "static byte[] PelValues = {" );
	for ( int y=0; y<ImageDim.height; y++) {
		for ( int x=0; x<ImageDim.width; x++) {
			ps.print( DefaultIndexModel.indexFor( Pels[x][y]) );
			if ( x<ImageDim.width-1)
				ps.print( ",");
		}
		if ( y<ImageDim.height-1)
			ps.println( ",");
	}
	ps.println( "};");

	ps.println( "static int Width  = " + ImageDim.width + ";" );
	ps.println( "static int Height = " + ImageDim.height + ";" );
}

void rectPath () {

	int x, y;
	int xd = PEnd.x > PStart.x ? 1 : -1;
	int yd = PEnd.y > PStart.y ? 1 : -1;

	Graphics g = View.PelPane.getGraphics();

	for ( x = PStart.x; x != PEnd.x; x += xd){
		Pels[x][PStart.y] = CurrentColor;
		Pels[x][PEnd.y] = CurrentColor;
		redrawPel( x, PStart.y, g);
		redrawPel( x, PEnd.y, g);
	}
	for ( y = PStart.y; y != PEnd.y; y += yd){
		Pels[PStart.x][y] = CurrentColor;
		Pels[PEnd.x][y] = CurrentColor;
		redrawPel( PStart.x, y, g);
		redrawPel( PEnd.x, y, g);
	}
	Pels[x][y] = CurrentColor;
	redrawPel( x, y, g);

	g.dispose();
}

void redrawPel ( int x, int y, Graphics g) {
	boolean tmpGraph = ( g == null);

	if ( tmpGraph)
		g = View.PelPane.getGraphics();

	Rectangle r = View.PelPane.getDrawRect();
	int   dx = r.width / ImageDim.width;
	int   dy = r.height / ImageDim.height;

	Color c = Pels[x][y] != null ? Pels[x][y] : View.PelPane.getBackground();
	g.setColor( c);
	g.fillRect( x*dx+r.x+1, y*dy+r.y+1, dx-1, dy-1);

	if ( tmpGraph)
		g.dispose();
}

void redrawPelPane ( Graphics g) {
	int x, y;
	Rectangle r = View.PelPane.getDrawRect();
	int       dx = r.width / ImageDim.width;
	int       dy = r.height / ImageDim.height;

	View.PelPane.blank( g);

	for ( x=0; x<ImageDim.width; x++){
		for ( y=0; y<ImageDim.height; y++){
			Color c = Pels[x][y];
			if ( c != null){
				g.setColor( c);
				g.fillRect( x*dx+r.x+1, y*dy+r.y+1, dx-1, dy-1);
			}
		}
	}

	g.setColor( Color.black);
	for ( x=0; x<=ImageDim.width; x++)
		g.drawLine( x*dx+r.x, r.y, x*dx+r.x, r.y+ImageDim.height*dy);
	for( y=0; y<=ImageDim.height; y++)
		g.drawLine( r.x, y*dy+r.y, r.x+ImageDim.width*dx, y*dy+r.y);
}

public void registerObserver() {
	if ( CU != null)
		CU.addObserver( this);

	View.OsClose.addObserver( this);
	View.BMenu.OsCommand.addObserver( this);
	View.Color1.OsMouseBtn.addObserver( this);
	View.Color2.OsMouseBtn.addObserver( this);
	View.Color3.OsMouseBtn.addObserver( this);
	View.Color4.OsMouseBtn.addObserver( this);
	View.Color5.OsMouseBtn.addObserver( this);
	View.Color6.OsMouseBtn.addObserver( this);
	View.Color7.OsMouseBtn.addObserver( this);
	View.Color8.OsMouseBtn.addObserver( this);
	View.Color9.OsMouseBtn.addObserver( this);
	View.Color10.OsMouseBtn.addObserver( this);
	View.Color11.OsMouseBtn.addObserver( this);
	View.Color12.OsMouseBtn.addObserver( this);
	View.Color13.OsMouseBtn.addObserver( this);
	View.ColorTrans.OsMouseBtn.addObserver( this);

	View.PelPane.OsDraw.addObserver( this);
	View.PelPane.OsMouseBtn.addObserver( this);
	View.CPrev.OsAction.addObserver( this);
	View.CNew.OsAction.addObserver( this);
	View.CClose.OsAction.addObserver( this);
	View.CSave.OsAction.addObserver( this);
	View.CPrevView.OsDraw.addObserver( this);

	View.CXDim.OsSpinChanged.addObserver( this);
	View.CYDim.OsSpinChanged.addObserver( this);
}

boolean save ( boolean transparent) {
	AcceptParser ap;
	int rgbB = transparent ? 0 : View.PelPane.getBackground().getRGB();

	if ( CU == null)
		return false;

	if ( TD == null)
		initCompileUnit();
	else CU.backup();

	try {
		ByteArrayOutputStream o = new ByteArrayOutputStream( 1000);
		PrintStream s = new PrintStream( o);

		printImageOn( s, transparent);
		ap = new AcceptParser( o.toString() );
		TypeDecl td = ap.parseDataOrMethod();
		TD.mergeDatas( td.Datas, null);
		TD.mergeMethods( td.Methods, null); 

		o.close();
		s.close();

		CU.save();
		CU.notifyObservers( this);
		return true;
	}
	catch( Throwable t) {
		t.printStackTrace();
		return false;
	}
}

public void setColorModel( ColorModel model) {
	IndexColorModel im = (IndexColorModel)model;
	int s = im.getMapSize();
	int t = im.getTransparentPixel();
	GifIdxTable = new Color[s];

	for ( int i=0; i<s; i++){
		if ( (t > 0) && (i == t) )
			GifIdxTable[i] = DefaultIndexModel.getTransColor();
		else
			GifIdxTable[i] = DefaultIndexModel.nearestColorFor( im.getRGB( i));
	}

}

void setCurrentColor ( Color c) {
	View.CurCol.setBackground( c);
	if ( c == DefaultIndexModel.getTransColor() )
		CurrentColor = null;
	else
		CurrentColor = c;
}

public void setDimensions( int width, int height) {
	ImageDim = new Dimension( width, height);
	Pels = new Color[width][height];
}

public void setHints( int hintflags) {
}

boolean setPel ( Event evt) {

	if ( evt.id == Event.MOUSE_UP)
		return false;

	try {
		Rectangle r = View.PelPane.getDrawRect();
		int dx = r.width / ImageDim.width;
		int dy = r.height / ImageDim.height;
		int x = evt.x / dx;
		int y = evt.y / dy;
		Color cc = CurrentColor;
		int   path = View.CPath.getContents();

		if ( Mouse.is2Active() ){
			if ( path == 0){
				PStart.x = x;
				PStart.y = y;
				cc = null;
			}
			else {
				PEnd.x = x;
				PEnd.y = y;
				if ( path == 1)       linePath();
				else if ( path == 2)  rectPath();
				else if ( path == 3)  circlePath();
				PStart.x = x;
				PStart.y = y;
				return false;
			}
		}
		else {
			PStart.x = x;
			PStart.y = y;
		}
		if ( Pels[x][y] != cc) {
			Pels[x][y] = cc;
			redrawPel( x, y, null);
			return true;
		}
		return false;
	}
	catch( Throwable t) {
		return false;
	}
}

public void setPixels(int x1, int y1, int w, int h, 
               ColorModel model, byte pixels[], int off, int scansize) {
	int x, y, sx;
	int sy = off;
	int x2 = x1+w;
	int y2 = y1+h;

	for(y=y1; y<y2; y++) {
		sx = sy;
		for(x=x1; x<x2; x++) 
			Pels[x][y] = GifIdxTable[pixels[sx++]];
		sy += scansize;
	}
}

public void setPixels(int x1, int y1, int w, int h, 
               ColorModel model, int pixels[], int off, int scansize) {
}

public void setProperties( Hashtable props) {
}

void unregisterObserver(){
	if ( CU != null)
		CU.deleteObserver( this);
}

public void update ( Observable obs, Object arg ) {
	if ( obs == CU){
		if ( arg != this)
			openOn( CU, TD);
		return;
	}
	ObserverSocket os = (ObserverSocket)obs;
	Object oso = os.getOwner();

	if ( oso instanceof GraphPane){
		GraphPane gp = (GraphPane)oso;
		if ( obs == gp.OsMouseBtn)
			if ( gp == View.PelPane)
			setPel( (Event)arg);
		else
			setCurrentColor( gp.getBackground() );
		else if ( obs == View.PelPane.OsDraw)
			redrawPelPane( (Graphics)arg);
		else if ( obs == View.CPrevView.OsDraw)
			updatePreview( (Graphics) arg);
	}
	else if ( obs == View.OsClose)
		unregisterObserver();
	else if ( obs == View.CXDim.OsSpinChanged)
		dimChange( View.CXDim.getContents(), View.CYDim.getContents() );
	else if ( obs == View.CYDim.OsSpinChanged)
		dimChange( View.CXDim.getContents(), View.CYDim.getContents() );
	else if ( obs == View.CPrev.OsAction )
		updatePreview( null);
	else if ( obs == View.CClose.OsAction )
		View.dispose();
	else if ( obs == View.CNew.OsAction)
		newImage();
	else if ( obs == View.CSave.OsAction)
		save( false);
	else if ( obs == View.BMenu.OsCommand){
		if ( "Save".equals( arg))
			save( false);
		else if ( "Load GIF".equals( arg))
			new Prompter( "GIF PathName", null, this, View) ;
	}
	else if ( obs instanceof Prompter)
		loadGif( ((Prompter)obs).getContents() );

}

void updatePreview( Graphics g) {
	boolean tg = (g == null);
	int x0 = (View.CPrevView.getWidth() - ImageDim.width) / 2;
	int y0 = (View.CPrevView.getHeight() - ImageDim.height) / 2;
	if ( tg)
		g = View.CPrevView.getGraphics();
	View.CPrevView.blank( g);
	for ( int y=0; y<ImageDim.height; y++) {
		for ( int x=0; x<ImageDim.width; x++) {
			Color c = Pels[x][y];
			if ( c != null) {
				g.setColor( c);
				g.drawLine( x+x0, y+y0, x+x0, y+y0);
			}
		}
	}
	if ( tg)
		g.dispose();
}
}
