package biss.awt;

import biss.ObserverSocket;
import java.awt.Color;
import java.awt.Event;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.LayoutManager;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * auxiliary class for objects representing single notebook tabs
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
class BookTab
{
	String TabLabel;
	int TabWidth;
	static FontMetrics Fm;
	static int TabHeight;
	PageSpec Master;
	static FontMetrics FmSmall;

static {
	Fm = Awt.getFontMetrics( Awt.BookTabFont);
	FmSmall = Awt.getFontMetrics( Awt.BookSmallFont);
	TabHeight = Fm.getHeight() + 5;
}

public BookTab ( String tabText, PageSpec bp) {
	Master = bp;
	TabLabel = tabText;
	TabWidth = Fm.stringWidth( tabText) + 10;
}

void drawAt (Graphics g, Color c, int x) {
	int dx=0, dy=0, i;
	int db=1;
	int rad = TabHeight/5;

	g.setColor( c );
	g.fillRect( x, 0, TabWidth, TabHeight);

	if ( Master.IsOnTop ){
		g.setColor( Color.white);
		db = 2;
	}
	else {
		g.setColor( Color.black);
		dx = 1; dy = 1;
	}

	// topleft
	for ( i=0; i<db; i++){
		g.drawLine( x+rad, i, x+TabWidth-rad-1, i);             //top
		g.drawLine( x, rad+i, x+rad, i);                        //topleft
		g.drawLine( x+TabWidth-rad-1, i, x+TabWidth-1, rad+i);  //topright
		g.drawLine( x+i, rad, x+i, TabHeight-1);                //left
	}

	// topshadow
	g.setColor( Color.white);
	g.drawLine( x+rad, 1, x+TabWidth-rad-1, 1);

	// right
	g.setColor( Color.black);
	for ( i=0; i<db; i++)
		g.drawLine( x+TabWidth-1-i, rad, x+TabWidth-1-i, TabHeight-1);

	if ( ! Master.IsOnTop)
		g.setFont( Awt.BookSmallFont);
	// page-connector
	else {
		g.setFont( Awt.BookTabFont);
		g.setColor( c);
		for ( i=0; i<3; i++)
			g.drawLine( x+2, TabHeight+i, x+TabWidth-3, TabHeight+i);
		g.setColor( Color.black);
	}

	// label
	g.drawString( TabLabel, x + 5 + dx, TabHeight - 4 + dy );

}

void drawBreakTab (Graphics g, Color c, int tw, int x) {
	int cw = 6;
	int pikes = 6;

	g.setColor( c);
	g.fillRect( tw-cw, 0, cw, TabHeight);

	g.setColor( Color.black);
	int xn, xo=tw-cw;
	for ( int i=0; i<pikes; i++) {
		xn = tw - (int)( Math.random() * cw);
		g.drawLine( xo, i*TabHeight/pikes, xn, (i+1)*TabHeight/pikes);
		xo = xn;
	}  
}
}

/**
 * class to implement a notebook widget (basically a container with
 * variable contents/pages). Rather than implementing a notebook class
 * as a another container widget (the usual way), we have choosen to
 * implement it just as a control widget that can be used to switch
 * between different layouts (you can think of it as a visual front end
 * for a LayoutManager like the CardLayout). This design implies that
 * neither notebook- "pages" nor widgets within these "pages" are childs
 * of their notebook. And this implies that you can have widgets
 * appearing on more than a single "page". This, in turn, reliefs you
 * of the burden to sync several widgets displaying the same data.
 * And - best of all - this can be implemented MUCH cleaner than the
 * usual (container-) notebook. Period.
 *
 * 
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class Notebook
  extends Control
{
	Vector Pages = new Vector();
	Layouter FrameLayouter;
	PageSpec TopPage;
	int TabOffs = 0;
	int TabEnd;
	static String Shift;
	static String BackShift;
	static int ShiftLen;
/**
 * triggered by TopPage change
 * the observer may call the lock() method to disable page change
 * parameters: int[] ( previous top index, next top index)
 */
	public ObserverSocket OsTopPage = new ObserverSocket( this);
	boolean Locked = false;

static {
	Shift = ">>";
	BackShift = "<<";
	ShiftLen = BookTab.Fm.stringWidth( Shift);
}

public Notebook () {
	setFont( Awt.BookTabFont);
}

public Notebook ( String labels) {
	this();
	StringTokenizer tok = new StringTokenizer( labels, ",:;");
	while( tok.hasMoreTokens() )
		addPage( tok.nextToken(), null );
	setTopPage( 0);
}

public void addNotify () {
	super.addNotify();
	LayoutManager lom = getParent().getLayout();
	if ( lom instanceof Layouter)
		FrameLayouter = (Layouter) lom;
	if ( TopPage == null)
		setTopPage( 0);
}

/**
 * use tabText of <null, "*" or " "> 
 * to build a tabless page ( minor)
 */
public int addPage ( String tabText, String status) {
	addPage( tabText, status, Pages.size() );
	return Pages.size();
}

public boolean addPage ( String tabText, String status, int layoutIdx) {
	return insertPage( layoutIdx, tabText, status);
}

void drawBookFrame ( Graphics g) {
	int fh = BookTab.Fm.getHeight();

	if ( Awt.PlainBook) {
		g.setColor( Color.white);
		g.drawLine( 0, BookTab.TabHeight, Width, BookTab.TabHeight);
		g.drawLine( 0, BookTab.TabHeight+1, Width, BookTab.TabHeight+1);
		g.setColor( Color.black);
		g.drawLine( 0, Height-1, Width, Height-1);
	}
	else {
		GraphicsLib.paint3dBox( g, 0, BookTab.TabHeight,
		                        Width, Height-BookTab.TabHeight,
		                        false, getBackground() );
	}

	drawSpins( g, fh);
	drawStatus( g, fh);
}

void drawSpins ( Graphics g, int fh) {
	int y = BookTab.TabHeight + fh;
	int w = ShiftLen;
	int h = fh;

	g.setColor( Color.black);
	g.drawString( BackShift, 4, y);
	g.drawString( Shift, Width-w-4, y);

	h -= 3;
	GraphicsLib.fillTri( g, w+8, y, w+8+h, y, w+8+h, y-h, Color.white);
	g.setColor( Color.black);
	g.drawRect( w+8, y-h, h, h);
	g.drawLine( w+8, y, w+8+h, y-h);  

	w = Width-w-8-h;
	GraphicsLib.fillTri( g, w, y-h, w, y, w+h, y, Color.white);
	g.setColor( Color.black);
	g.drawRect( w, y-h, h, h);
	g.drawLine( w, y-h, w+h, y);  
}

boolean drawStatus ( Graphics g, int fh) {
	int y = BookTab.TabHeight + fh;
	int w = ShiftLen;

	int x1 = 2*(w+4);
	int x2 = Width - x1; 

	if ( (TopPage == null) || (TopPage.Status == null) )
		return false;

	g.setFont( Awt.BookSmallFont);
	g.drawString( TopPage.Status, x1 + ((x2-x1)-
	                                    BookTab.FmSmall.stringWidth( TopPage.Status)) / 2, y);

	return true;
}

void drawTabs ( Graphics g) {
	int   w = 0;
	Color c = getBackground();

	for ( TabEnd=TabOffs; TabEnd<Pages.size(); TabEnd++) {
		PageSpec pg = (PageSpec) Pages.elementAt( TabEnd);
		if ( pg.Tab != null){
			pg.Tab.drawAt( g, c, w);
			w += pg.Tab.TabWidth;
			if ( w > Width) {
				pg.Tab.drawBreakTab( g, c, Width, w);
				break;
			}
		}
	}
	TabEnd--;
	if ( w < Width) {
		g.setColor( c);
		g.fillRect( w, 0, Width - w, BookTab.TabHeight);
	}  
}

public boolean insertPage ( int insIdx, String tabText, String status) {
	return insertPage( insIdx, tabText, status, insIdx);
}

public boolean insertPage ( int insIdx, String tabText, String status, int layoutIdx) {
	try {
		PageSpec pg = new PageSpec( layoutIdx, tabText, status, this);
		Pages.insertElementAt( pg, insIdx);
		if ( isShowing() )
			redraw();
		return true;
	}
	catch( Exception e) {
		return false;
	}
}

public void lock() {
	Locked = true;
}

public void mouse1Up ( Event evt) {
	// Windows sometimes messes up the ect.x/y coordinates in MOUSE_DOWN events, so
	// let us do Button/Menu like page switching

	int w = 0;
	int tw = Height - BookTab.TabHeight;

	if ( evt.y < BookTab.TabHeight) {
		for ( int i=TabOffs; i<=TabEnd; i++) {
			PageSpec pg = (PageSpec)Pages.elementAt( i);
			if ( pg.Tab == null)
				continue;
			if ( w + pg.Tab.TabWidth > evt.x){
				setTopPage( pg);
				break;
			}
			w += pg.Tab.TabWidth;
		}
	}
	else {
		int h = BookTab.Fm.getHeight();
		int dw = 4+ShiftLen;

		if ( evt.y < BookTab.TabHeight + h) {
			if ( evt.x < dw)
				shiftLeft();
			else if ( evt.x < 2*dw )
				prevPage();
			else if ( evt.x > Width-dw)
				shiftRight();
			else if ( evt.x > Width-2*dw)
				nextPage();
		}
	}
}

public void nextPage() {
	if ( TopPage == null)
		setTopPage( 0);
	else
		setTopPage( Pages.indexOf( TopPage) +1);
}

public void prevPage() {
	if ( TopPage == null)
		setTopPage( 0);
	else
		setTopPage( Pages.indexOf( TopPage) -1);
}

public void redraw ( Graphics g) {
	drawBookFrame( g);
	drawTabs( g);
}

public boolean removePage ( int pageIdx ) {
	try {
		PageSpec ps = (PageSpec)Pages.elementAt( pageIdx);
		if ( ps == TopPage)
			TopPage = null;
		Pages.removeElement( ps);
		if ( isShowing() )
			redraw();
		return true;
	}
	catch( Exception e) {
		return false;
	}

}

boolean setTopPage ( PageSpec bp) {
	if ( (bp == TopPage) || (bp == null) )
		return false;

	int ci = (TopPage != null) ? Pages.indexOf( TopPage) : -1;
	int ni = Pages.indexOf( bp);

	Locked = false;
	int[] ia = { ci, ni}; 
	OsTopPage.notifyObservers( ia );

	if ( Locked)
		return false;

	if ( ci > -1)
		TopPage.IsOnTop = false;
	TopPage = bp;
	TopPage.IsOnTop = true;
	redraw();

	if ( FrameLayouter != null){
		if ( FrameLayouter.countLayouts() > TopPage.LayoutIndex)
			FrameLayouter.setLayout( TopPage.LayoutIndex);
	}

	return true;
}

public boolean setTopPage ( int idx) {
	try {
		PageSpec pg = (PageSpec)Pages.elementAt( idx);
		shiftShow( idx);
		return setTopPage( pg);
	}
	catch( Throwable t) {
		return false;
	}
}

void shiftLeft() {
	if ( TabOffs > 0){
		TabOffs--;
		redraw();
	}
}

void shiftRight() {
	if ( TabEnd < Pages.size()-1){
		TabOffs++;
		redraw();
	}
}

/**
 * shift pagetab to visible region ( -> setTopPage() )
 */
void shiftShow ( int idx) {
	if ( idx < TabOffs)
		TabOffs = idx;
	else if ( idx > TabEnd) {
		int tw = 0;
		for ( TabOffs=idx; TabOffs>-1; TabOffs--) {
			PageSpec pg = (PageSpec)Pages.elementAt( TabOffs);
			if ( pg.Tab != null)
				tw += pg.Tab.TabWidth;
			if ( tw > Width){
				TabOffs++;
				break;
			}
		}
	}
}
}

class PageSpec
{
	BookTab Tab;
	String Status;
	boolean IsOnTop = false;
	Notebook Master;
	int LayoutIndex;

public PageSpec ( int lIdx, String tabText, String status, Notebook bf) {
	Master = bf;
	LayoutIndex = lIdx;

	if (tabText != null) {
		if (! tabText.equals( " ") && ! tabText.equals( "*") )
			Tab = new BookTab( tabText, this);
	}
	Status = status;
}

void toTop() {
	Master.setTopPage( this);
}
}
