package biss.awt;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.image.ImageObserver;

/**
 * common draw support for biss.awt classes
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class GraphicsLib
{
	final public static int NORTH = 0;
	final public static int EAST = 1;
	final public static int SOUTH = 2;
	final public static int WEST = 3;
	final public static int NORTHWEST = 4;
	final public static int NORTHEAST = 5;
	final public static int SOUTHEAST = 6;
	final public static int SOUTHWEST = 7;

public static void drawBox ( Graphics g, int x, int y, int w, int h, int thik, Color clr ){

	int i, x0, y0, x1, y1, ww, hh;

	g.setColor( clr);
	x0 = x; y0 = y; ww=w-1; hh=h-1;
	for ( i=0; i<thik; i++ ) {
		g.drawRect( x0, y0, ww, hh);
		x0++; y0++; ww-=2; hh-=2;
	}
}

public static void drawFocusMark ( Graphics g, int x, int y ) {
	drawFocusMark( g, x, y, Awt.ScrollBarWidth - Awt.BorderWidth);
}

public static void drawFocusMark ( Graphics g, int x, int y, int w ) {
	int u[] = { x,  x,  x+w };
	int v[] = { y+w, y, y };

	g.setColor( Awt.FocusClr);
	g.fillPolygon( u, v, 3);

	g.setColor( Color.black);  
	g.drawLine( u[2], v[2], u[0], v[0]);

	g.setColor( Color.gray);
	u[0]++; v[2]++;
	g.drawLine( u[2], v[2], u[0], v[0]);

	v[0]++; u[2]++;
	g.drawLine( u[2], v[2], u[0], v[0]);
}

public static boolean drawImageCentered ( Graphics g, Image img,
                            int width, int height, ImageObserver ios ) { 
	return drawImageCentered( g, img, img.getWidth( ios),
	                          img.getHeight( ios), 0, 0,
	                          width, height, ios);
}

public static boolean drawImageCentered ( Graphics g, Image img, int imgWidth, int imgHeight,
                            int width, int height, ImageObserver ios ) { 
	return drawImageCentered( g, img, imgWidth, imgHeight, 0, 0,
	                          imgWidth, imgHeight, ios);
}

public static boolean drawImageCentered ( Graphics g, Image img, int imgWidth, int imgHeight,
                            int x, int y, int width, int height, ImageObserver ios ) { 
	if ( (imgWidth > 0) && (imgHeight > 0) )
		return g.drawImage( img, x + (width - imgWidth) / 2,
	                    y + (height - imgHeight) / 2,
	                    imgWidth, imgHeight, ios);

	return false;
}

public static boolean drawImageVCentered ( Graphics g, Image img,
                             int x, int yoffs, int height, ImageObserver ios ) { 
	return drawImageVCentered( g, img, img.getWidth( ios),
	                           img.getHeight( ios),
	                           x, yoffs, height, ios);
}

public static boolean drawImageVCentered ( Graphics g, Image img, int imgWidth, int imgHeight,
                             int x, int yoffs, int height, ImageObserver ios ) { 
	if ( (imgWidth > 0) && (imgHeight > 0) )
		return g.drawImage( img, x, yoffs + (height - imgHeight) / 2,
	                    imgWidth, imgHeight, ios);
	return false;
}

public static void drawRightTri ( Graphics g, int x, int y, int w, int h, boolean inset ) {
	int x0, y0, x1, y1;
	Color cl, ct, cb;
	int i, n = Awt.BorderWidth;

	if ( inset ) {
		cl = Awt.ShadowDarkClr;
		ct = Awt.ShadowMedDarkClr;
		cb = Awt.ShadowBrightClr;
	}
	else {
		cl = Awt.ShadowBrightClr;
		ct = Awt.ShadowMedDarkClr;
		cb = Awt.ShadowDarkClr;
	}
	
	g.setColor( cl);
	x0 = x; y0 = y; y1 = y+h;
	for ( i=0; i<n; i++ ) {
		g.drawLine( x0, y0, x0, y1);
		x0++; y0++; y1--;
	}

	g.setColor( cb);
	x0 = x; y0 = y+h; x1 = x+w; y1 = y+h/2;
	for ( i=0; i<n; i++ ) {
		g.drawLine( x0, y0, x1, y1);
		x0++; y0--; x1--;
	}
	
	g.setColor( ct);
	x0 = x; y0 = y; x1 = x+w;
	for ( i=0; i<n; i++ ) {
		g.drawLine( x0, y0, x1, y1);
		x0++; y0++; x1--;
	}
}

public static void drawStepTri ( Graphics g, int x, int y, int w, int h,
                   int direction, boolean inset ) {
	fillStepTri( g, x, y, w, h, direction, inset, null);
}

public static void drawString ( Graphics g, String s, int x, int y, Color c, boolean carved ) {
	if ( carved){
		g.setColor( Awt.ShadowBrightClr);
		g.drawString( s, x + 1, y + 1);
	}
	g.setColor( c );
	g.drawString( s,  x, y);
}

public static void drawStringVCentered ( Graphics g, String s, int x, int y,
                           int height, Font fnt, Color c, boolean carved ) {
	FontMetrics fm = Awt.DefToolkit.getFontMetrics( fnt);
	int yb = y + height - ( height - fm.getHeight()) / 2 - fm.getDescent();

	g.setFont( fnt);
	if ( carved){
		g.setColor( Awt.ShadowBrightClr);
		g.drawString( s, x + 1, yb + 1);
	}
	g.setColor( c );
	g.drawString( s,  x, yb);
}

public static void drawTri ( Graphics g, int x1, int y1, 
               int x2, int y2, int x3, int y3 ) {
	g.drawLine( x1, y1, x2, y2);
	g.drawLine( x2, y2, x3, y3);
	g.drawLine( x3, y3, x1, y1);
}

public static void fillStepTri ( Graphics g, int x, int y, int w, int h,
                   int direction, boolean inset, Color fill ) {

	int u[] = new int[3];
	int v[] = new int[3];
	int b = Awt.BorderWidth;
	Color clrB = inset ? Awt.ShadowDarkClr : Awt.ShadowBrightClr;
	Color clrD = inset ? Awt.ShadowBrightClr : Awt.ShadowDarkClr;

	switch ( direction ) {
	case NORTH:
		u[0] =  x+w/2; v[0] = y;     u[1] = x+w;  v[1] = y+h;    u[2] = x;   v[2] = v[1];
		if ( fill != null ) {
			g.setColor( fill);
			g.fillPolygon( u, v, 3);
		}
		g.setColor( clrD);
		for ( int i=0; i<b; i++ ){
			g.drawPolygon( u, v, 3);
			v[0]++;  u[1]--; v[1]--; u[2]++; v[2]--;
		}
		g.drawLine( u[2], v[2]+2, u[1], v[2]+2);
		g.setColor( clrB);
		for ( int i=0; i<b; i++ ){
			v[0]--; u[2]--; v[2]++;
			g.drawLine( u[2], v[2], u[0], v[0]);
		}
		break;
	case NORTHWEST:
		u[0] = x; v[0] = y+h;  u[1] = x; v[1] = y; u[2] = x+w;  v[2] = y;
		if ( fill != null){
			g.setColor( fill);
			g.fillPolygon( u, v, 3);
		}
		g.setColor( clrB);
		for ( int i=0; i<b; i++ ){
			g.drawPolygon( u, v, 3);
			u[0]++;  v[0]--; u[1]++; v[1]++; u[2]--; v[2]++;
		}

		g.setColor( clrD);
		for ( int i=0; i<b; i++ ){
			u[0]--; v[0]++; u[2]++; v[2]--;
			g.drawLine( u[2], v[2], u[0], v[0]);
		}
		break;
	case EAST:
		u[0] =  x;    v[0] = y+h;   u[1] = x;    v[1] = y;      u[2] = x+w; v[2] = y+h/2;
		if ( fill != null ) {
			g.setColor( fill);
			g.fillPolygon( u, v, 3);
		}
		g.setColor( clrB);
		for ( int i=0; i<b; i++ ){
			g.drawPolygon( u, v, 3);
			u[0]++; v[0]--;  u[1]++; v[1]++; u[2]--;
		}
		g.setColor( clrD);
		for ( int i=0; i<b; i++ ){
			u[0]--; v[0]++; u[2]++;
			g.drawLine( u[2], v[2], u[0], v[0]);
		}
		break;
	case SOUTH:
		u[0] =  x+w/2; v[0] = y+h;   u[1] = x;    v[1] = y;      u[2] = x+w; v[2] = y;
		if ( fill != null ) {
			g.setColor( fill);
			g.fillPolygon( u, v, 3);
		}
		g.setColor( clrB);
		for ( int i=0; i<b; i++ ){
			g.drawPolygon( u, v, 3);
			v[0]--; u[1]++; v[1]++; u[2]--; v[2]++;
		}
		g.setColor( clrD);
		for ( int i=0; i<b; i++ ){
			v[0]++; u[2]++; v[2]--;
			g.drawLine( u[2], v[2], u[0], v[0]);
		}
		break;
	case WEST:
		u[0] =  x+w;  v[0] = y;     u[1] = u[0]; v[1] = y+h;    u[2] = x;   v[2] = y+h/2;
		if ( fill != null ) {
			g.setColor( fill);
			g.fillPolygon( u, v, 3);
		}
		g.setColor( clrD);
		for ( int i=0; i<b; i++ ){
			g.drawPolygon( u, v, 3);
			u[0]--; v[0]++; u[1]--; v[1]--; u[2]++; 
		}
		g.setColor( clrB);
		for ( int i=0; i<b; i++ ){
			u[0]++; v[0]--; u[2]--;
			g.drawLine( u[2], v[2], u[0], v[0]);
		}
		break;
	}
}

public static void fillTri ( Graphics g, int x1, int y1, 
               int x2, int y2, int x3, int y3, Color fillClr ) {
	int x[] = { x1, x2, x3 };
	int y[] = { y1, y2, y3 };
	g.setColor( fillClr);
	g.fillPolygon( x, y, 3);
}

public static void flashText ( Graphics g, String text, int x, int y ) {

	g.setColor( Awt.ButtonAlertForeClr);
	g.drawString( text, x-1, y-1);
	g.drawString( text, x+1, y+1);
	g.drawString( text, x-1, y+1);
	g.drawString( text, x+1, y-1);

	g.setColor( Awt.ButtonAlertBackClr);		
	g.drawString( text, x, y);
}

public static void paint3dBox ( Graphics g, int x, int y, int w, int h, boolean inset, Color fill ){

	int i, x0, y0, x1, y1, ww, hh;
	Color c1, c2;

	if ( inset ) {
		c1 = Awt.ShadowDarkClr;
		c2 = Awt.ShadowBrightClr;
	}
	else {
		c1 = Awt.ShadowBrightClr;
		c2 = Awt.ShadowDarkClr;
	}

	g.setColor( c1);
	x0 = x; y0 = y; x1 = x+w-1; y1 = y+h-1;
	for ( i=0; i<Awt.BorderWidth; i++ ) {
		g.drawLine( x0, y0, x1, y0);
		g.drawLine( x0, y0, x0, y1);
		x0++; y0++; x1--; y1--;
	}

	g.setColor( c2 );
	x0 = x; y0 = y; x1 = x+w-1; y1 = y+h-1;
	for ( i=0; i<Awt.BorderWidth; i++ ) {
		g.drawLine( x0, y1, x1, y1);
		g.drawLine( x1, y1, x1, y0);
		x0++; y0++; x1--; y1--;
	}

	if ( fill != null ) {
		g.setColor( fill);
		x1++; y1++;
		g.fillRect( x0, y0, (x1-x0), (y1-y0));
	}
}

public static void paintButton ( Graphics g, int x, int y, int w, int h, boolean isPushed, Color fill ){

	int i, n = Awt.BorderWidth;
	int xw = x+w-1;
	int yh = y+h-1;
	int x0, x1, y0, y1;
	int w1 = w-1, h1 = h-1;

	if ( isPushed ) {
		w1--; h1--;
		g.setColor( Awt.ShadowMedDarkClr);
		g.drawRect( x, y, w1, h1);
		g.setColor( Awt.ShadowBrightClr);
		g.drawRect( x+1, y+1, w1, h1);

		g.setColor( Awt.ShadowDarkClr);
		g.drawLine( x, y, xw, y);
		g.drawLine( x, y, x,  yh);

		/*
		  g.setColor( Awt.ShadowMedDarkClr);
		  g.drawRect( x, y, w1-1, h1-1);
		  g.setColor( Awt.ShadowBrightClr);
		  x0 = x; y0 = y; x1 = xw; y1 = yh;
		  g.drawLine( x0, y1, x1, y1);
		  g.drawLine( x1, y1, x1, y0);
		  g.setColor( Awt.ShadowDarkClr);
		  x0++; y0++; x1--; y1--;
		  g.drawLine( x0, y0, x1, y0);
		  g.drawLine( x0, y0, x0, y1);
		 */

		if ( fill != null ) {
			g.setColor( fill );
			g.fillRect( x+2, y+2, w1-2, h1-2);
		}

		return;
	}

	// upper contour
	g.setColor( Awt.ShadowMedDarkClr);
	g.drawRect( x, y, w1, h1);
	x0 = x+n; y0 = y+n; x1 = x+w-n; y1 = y+h-n;
	g.drawLine( x0, y1, x1, y1);
	g.drawLine( x1, y0, x1, y1);

	// lower edge & contour
	g.setColor( Awt.ShadowDarkClr);
	x0 = x+1; y0 = y+1; w1--; h1--;
	for ( i=1; i<n; i++ ) {
		g.drawRect( x0, y0, w1, h1);
		x0++; y0++; w1-=2; h1-=2;
	}

	// upper edge
	g.setColor( Awt.ShadowBrightClr);
	x0 = x+1; y0 = y+1; x1 = xw-1; y1 = yh-1;
	for ( i=1; i<n; i++ ) {
		g.drawLine( x0, y0, x1, y0);
		g.drawLine( x0, y0, x0, y1);
		x0++; y0++; x1--; y1--;
	}

	if ( fill != null ) {
		g.setColor( fill);
		g.fillRect( x0, y0, (x1-x0+1), (y1-y0+1));
	}
}

public static void paintCarvedBox ( Graphics g, int x, int y, int w, int h){
	g.setColor( Color.white);
	g.drawRect( x+1, y+1, w-2, h-2);
	g.setColor( Color.black);
	g.drawRect( x, y, w-2, h-2);
}

public static void paintChoiceButton ( Graphics g, int x, int y, int w, int h, boolean pressed){
	paintButton( g, x, y, w, h, pressed, Awt.ButtonBackClr);

	int b = Awt.BorderWidth;
	int dx = (w - 2*b -4) | 1;
	int dx2 = dx/2;
	int dy = dx2 + 2;
	int x0, y0, x1, y1, x2;
	
	x0 = x + b + 1;
	x1 = x0 + dx2;
	x2 = x0 + dx;
	y0 = y + (h - dy)/2;
	y1 = y0 + dy - 1;

	g.setColor( Awt.ShadowDarkClr);
	g.drawLine( x0, y0, x2, y0);
	y0++;
	g.setColor( Awt.ShadowMedDarkClr);
	g.drawLine( x0, y0, x1, y1);
	g.setColor( Awt.ShadowBrightClr);
	x1++;
	g.drawLine( x2, y0, x1, y1);
}

public static void paintInstepBox ( Graphics g, int x, int y, int w, int h, Color fill ){

	int i, x0, y0, x1, y1, ww, hh;

	g.setColor( Awt.ShadowDarkClr);
	x0 = x; y0 = y; ww=w-1; hh=h-1;
	for ( i=1; i<Awt.BorderWidth; i++ ) {
		g.drawRect( x0, y0, ww, hh);
		x0++; y0++; ww-=2; hh-=2;
	}

	g.setColor( Awt.ShadowMedDarkClr);
	x1 = x0 + ww; y1 = y0 + hh;
	g.drawLine( x0, y0, x0, y1);
	g.drawLine( x0, y0, x1, y0);

	g.setColor( Awt.ShadowBrightClr );
	x0 = x; y0 = y; x1 = x+w-1; y1 = y+h-1;
	for ( i=1; i<Awt.BorderWidth; i++ ) {
		g.drawLine( x0, y1, x1, y1);
		g.drawLine( x1, y1, x1, y0);
		x0++; y0++; x1--; y1--;
	}

	g.setColor( Awt.ShadowMedBrightClr);
	g.drawLine( x0, y1, x1, y1);
	g.drawLine( x1, y1, x1, y0);

	if ( fill != null ) {
		g.setColor( fill);
		x0++; y0++;
		g.fillRect( x0, y0, (x1-x0), (y1-y0));
	}
}

public static void paintRadio ( Graphics g, int x, int y, 
                  int w, int h, boolean checked ){

	int i, x0, y0, x1, y1, xw2, yh2, xw, yh;
	int n = Awt.BorderWidth;

	xw2 = x + (w>>1);
	yh2 = y + (h>>1);
	xw = x+w-1;
	yh = y+h-1;

	g.setColor( Awt.ShadowDarkClr);
	x0 = x; y0 = yh2; x1 = xw2-1; y1 = y+1;
	for ( i=0; i<n; i++ ) {
		g.drawLine( x0, y0, x1, y1);
		x0++; y1++;
	}

	g.setColor( Awt.ShadowMedDarkClr);
	x0 = x+1; y0 = yh2+1; x1 = xw2-1; y1 = (yh-1);
	for ( i=0; i<n; i++ ) {
		g.drawLine( x0, y0, x1, y1);
		x0++; y1--;
	}
	x0 = xw2; y0 = y; x1 = xw-1; y1 = yh2-1;
	for ( i=0; i<n; i++ ) {
		g.drawLine( x0, y0, x1, y1);
		y0++; y1++;
	}

	g.setColor( Awt.ShadowBrightClr);
	x0 = xw2; y0 = yh; x1 = xw; y1 = yh2;
	for ( i=1; i<n; i++ ) {
		g.drawLine( x0, y0, x1, y1);
		y0--; x1--;
	}

	g.setColor( Awt.ShadowMedBrightClr);
	g.drawLine( x0, y0, x1, y1);

	// fill the interior (we do it explicitly because we don't want to
	// create an Array object)
	g.setColor( checked ? Awt.CheckboxSetClr : Awt.CheckboxUnsetClr);
	x = x+Awt.BorderWidth; y = yh2; w -= n<<1; h = 1;

	n = w>>1;
	for ( i=0; i<=n; i++ ) {
		g.fillRect( x, y, w, h );
		x++; y--; w-=2; h+=2;
	}
}

public static void paintSpinButton ( Graphics g, int x, int y, int w, int h, int state ){

	int i, n = Awt.BorderWidth;
	int yh = y+h-1;
	int yh2 = y + h/2;
	int x0, x1, y0, y1;
	int w1 = w-1, h1 = h-1;

	g.setColor( Awt.ShadowMedDarkClr );
	g.drawLine( x, y, x, yh);
	x++;

	int xw = x+w-2;

	if ( state == 0 ) {
		g.setColor( Awt.ShadowBrightClr);
		g.drawLine( x, y, x, yh);
		
		x0 = x; y0 = y; x1 = xw;
		for ( i=1; i<n; i++ ) {
			g.drawLine( x0, y0, x1, y0);
			y0++;
		}

		g.setColor( Awt.ShadowDarkClr);
		x0++; y1 = yh;
		for ( i=1; i<n; i++ ) {
			g.drawLine( x0, y1, x1, y1);
			y1--;
		}
		g.drawLine( x1, y0-1, x1, yh2);

		g.setColor( Awt.ShadowMedDarkClr);
		g.drawLine( x0, y1, x1, y1);
		g.drawLine( x1, y1, x1, yh2);
		w1 = x1-x0;
		g.fillRect( x0, y0, w1, (yh2-y0));
		
		g.setColor( Awt.ButtonBackClr);
		y0 = yh2+1;
		g.fillRect( x0, y0, w1, (y1-y0));
	}
	else if ( state == 1 ) {
		x0 = x+1; y0 = y; x1 = xw; y1 = yh;
		
		g.setColor( Awt.ShadowDarkClr);
		for ( i=0; i<n; i++ ){
			g.drawLine( x0, y1, x1, y1);
			y1--;
		}
		
		g.setColor( Awt.ShadowMedDarkClr);		
		g.drawLine( x0, y1, x1-1, y1);
		for ( i=1; i<n; i++ ) {
			g.drawLine( x0, y0, x1, y0);
			y0++;		
		}
		
		// g.setColor( Awt.ShadowMedBrightClr);
		// g.drawLine( x0, y, x1-1, y);
	}
	else if ( state == 2 ) {
		g.setColor( Awt.ShadowBrightClr);
		x0 = x; y0 = y; x1 = xw; y1 = yh;
		for ( i=01; i<n; i++ ) {
			y0++;
			g.drawLine( x0, y0, x1, y0);
		}

		g.setColor( Awt.ShadowMedDarkClr);
		y0++; x0++;
		x0 = x+1; x1 = xw; y1 = yh;
		g.drawLine( x0, y1, x1, y1);
		g.drawLine( x1, y, x1, y1);

		g.setColor( Awt.ShadowDarkClr);
		g.drawLine( x1, y0, x1, yh2-1);
		
		g.setColor( Awt.ButtonBackClr );
		x1--; y1 = yh-1;
		for ( i=0; i<n; i++ ) {
			g.drawLine( x0, y1, x1, y1);
			y1--;
		}
	}
}
}
