package biss.awt;

import java.awt.Color;
import java.awt.Image;
import java.awt.image.ColorModel;
import java.awt.image.ImageConsumer;
import java.awt.image.ImageProducer;
import java.awt.image.IndexColorModel;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Hashtable;

/**
 * Simple GIF file/stream reader which can be used to replace (or
 * implement) Toolkit.getImage(). Doesn't need any help from the native
 * layer and tries to do some speed and memory optimizations. Doesn't
 * support incremental processing of images yet.
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @version 0.86  
 * @author P.C.Mehlitz
 */
public class GifProducer
  extends ImageBuilder
{
	InputStream In;
	int CodeSize;
	int CodeClear;
	int CodeEOF;
	int CodeFree;
	int CodeFirstFree;
	int CodeSizeInit;
	int CodeMax;
	int ReadMask;
	int BitOffset;
	int BitMask;
	byte[] Block;
	int BlockSize;
	boolean AllPixelsRead;
	int BS_1;
	int BS_3;
	int N = 0;
	int Transparent = -1;
	boolean IsInterlaced;
	int X;
	int Y;
	int Pass;
	static int YInc[] = { 8, 8, 4 };
	static int YSet[] = { 4, 2, 1 };

public GifProducer ( File gifFile ) {
	Source = gifFile;
}

public GifProducer ( URL gifURL ) {
	Source = gifURL;
}

ColorModel getColorModel ( int cmapSize, byte rgb[], int transp ) throws IOException {
	ColorModel model;
	String     tk = System.getProperty( "awt.toolkit");

	if ( tk != null && tk.equals( "biss.awt.kernel.Toolkit") ){
		model = new MutableIndexClrModel( 8, cmapSize, rgb, 0, false, transp );
	}
	else {
		model = new IndexColorModel( 8, cmapSize, rgb, 0, false, transp);
	}

	return model;
}

int nextCode ()  throws IOException {
	int i = BitOffset >> 3;  // 8
	int code;

	if ( AllPixelsRead ) {
		if ( i >= BS_1 )
			return CodeEOF;
	}
	else if ( i >= BS_3 ) {
		readNextBlock( i);
		if ( !AllPixelsRead )	i = 0;
	}

	code = (Block[i++] & 0x000000ff) + ( (Block[i++] & 0x000000ff) << 8);

	if ( CodeSize >= 8 )
		code += (Block[i] & 0x000000ff) << 16;

	code >>= (BitOffset & 0x00000007);   // % 8
	BitOffset += CodeSize;

	code &= ReadMask;

	return code;
}

private int read2byte ()  throws IOException {
	int i = In.read();
	i += In.read() << 8;

	return i;
}

void readCodeInits ()  throws IOException {
	CodeSize = In.read();
	CodeClear = 1 << CodeSize;
	CodeEOF = CodeClear + 1;
	CodeFree = CodeClear + 2;
	CodeFirstFree = CodeFree;

	CodeSize++;
	CodeSizeInit = CodeSize;
	CodeMax = 1 << CodeSize;
	ReadMask = CodeMax - 1;
}

boolean readExtension ()  throws IOException {
	int b;

	b = In.read();

	if ( b == 0xf9 ) { // graphics control
		b = In.read();
		if ( (In.read() & 1) > 0 ) {
			In.skip(2);
			Transparent = In.read();
			In.skip( b-3);
		}
		else
			In.skip( b);
	}
	else {
		while ( (b=In.read()) != 0 )
			In.skip( b);
	}

	return true;
}

boolean readHeader ()  throws IOException {
	int      b, width, height;
	int      scrWidth, scrHeight, bitpp, numCols, cmapSize;
	int      background, left, top;
	byte     rgb[];
	ColorModel clrModel;
	boolean  hasGlobalCmap;
	Hashtable props = null;

	if ( !readSignature() ) return false;
	scrWidth = read2byte();
	scrHeight = read2byte();

	hasGlobalCmap = ((b = In.read()) & 0x80) > 0;
	bitpp = (b & 7) + 1;
	numCols = 1 << bitpp;
	cmapSize = numCols;
	BitMask = cmapSize - 1;
	background = In.read();

	if ( In.read() != 0 ) return false;

	rgb = new byte[cmapSize*3];
	if ( hasGlobalCmap )
		In.read( rgb);

	while ( (b=In.read()) == 0x21 )
		readExtension();

	if ( b != 0x2c ) return false;

	left = read2byte();
	top = read2byte();
	width = read2byte();
	height = read2byte();
	IsInterlaced = (In.read() & 0x40) > 0;

	readCodeInits();

	setDimensions( width, height, width, 0);	
	if ( props != null )
		setProperties( props);

	clrModel = getColorModel( cmapSize, rgb, Transparent);
	setColorModel( clrModel);

	setHints( ImageConsumer.TOPDOWNLEFTRIGHT | ImageConsumer.SINGLEFRAME |
	          ImageConsumer.SINGLEPASS | ImageConsumer.COMPLETESCANLINES );

	return true;
}

private void readNextBlock ( int curPos )  throws IOException {
	int n, i = 0, read = 0, r;

	if ( (n = In.read()) > 0 ) {
		while ( curPos < BlockSize ) {
			Block[i++] = Block[curPos++];
		}

		BlockSize = n;
		while ( read < BlockSize ) {
			if ( (r = In.read( Block, i + read, BlockSize - read)) == -1 ) {
				// throw new IOException( "too few bytes in GIF block"); // too harsh
				break;  // more relaxed, better display some garbage than nothing at all
			}
			else
				read  += r;
		}

		BlockSize += i;
		BitOffset = (BitOffset & 0x00000007);   // % 8

		BS_1 = BlockSize - 1;
		BS_3 = BlockSize - 3;
	}
	else {
		AllPixelsRead = true;
	}
}

void readPixelData () throws IOException {
	byte pixels[]  = new byte[Width*Height+10];
	short outCode[] = new short[1024];
	short prefix[]  = new short[4096];
	short suffix[]  = new short[4096];
	int i = 0, maxClr = 0;
	int b;

	int  c, curCode, oldCode = 0, finChar = 0, inCode, out = 0;

	Block = new byte[260];
	AllPixelsRead = false;
	BlockSize = BS_1 = BS_3 = 0;
	BitOffset = 0;

	readNextBlock( 0);

	while ( (c= nextCode()) != CodeEOF ) {
		if ( c == CodeClear ) {
			CodeSize = CodeSizeInit;
			CodeMax = 1 << CodeSize;
			ReadMask = CodeMax - 1;
			CodeFree = CodeFirstFree;

			c = nextCode();
			curCode = oldCode = c;
			finChar = c & BitMask;

			pixels[i++] = (byte) finChar;
			if ( finChar > maxClr ) maxClr = finChar;
		}
		else {
			curCode = inCode = c;

			if ( curCode >= CodeFree ) {
				curCode = oldCode;
				outCode[out++] = (short) finChar;
			}

			while ( curCode > BitMask ) {
				outCode[out++] = suffix[curCode];
				curCode = prefix[curCode];
			}

			finChar = curCode & BitMask;
			outCode[out] = (short) finChar;

			// store pixel values
			if ( IsInterlaced ) { //----------------- interlaced, compute X, Y
				for ( ; out >= 0; out-- ){
					b = outCode[out];
					if ( b > maxClr ) maxClr = b;

					if ( Y < Height )
						pixels[ Y* ScanLine + X] = (byte) b;
					if ( ++X == Width ) {
						X = 0;
						if ( Pass < 3 ) {
							if ( (Y += YInc[Pass]) >= Height ) {
								Y = YSet[Pass];	Pass++;
							}
						}
						else
							Y += 2;
					}
				}
			}
			else {                //------------------- non-interlaced -> consecutive
				for ( ; out >= 0; out-- ){
					b = outCode[out];
					if ( b > maxClr ) maxClr = b;
					pixels[i++] = (byte) b;
				}
			}

			out = 0;

			prefix[CodeFree] = (short) oldCode;
			suffix[CodeFree] = (short) finChar;
			oldCode = inCode;

			if ( ++CodeFree >= CodeMax ) {
				if ( CodeSize < 12 ) {
					CodeSize++;
					CodeMax *= 2;
					ReadMask = (1 << CodeSize) - 1;
				}
			}
		}
	}

	if ( CM instanceof MutableIndexClrModel )
		((MutableIndexClrModel)CM).shrinkRgbs( maxClr);

	setPixels( pixels, 0, i);
	imageComplete( ImageConsumer.STATICIMAGEDONE);
	Block = null;
}

boolean readSignature ()  throws IOException {
	byte b[] = new byte[6];

	In.read( b, 0, 6);

	if ( b[0] != 'G' || b[1] != 'I' || b[2] != 'F' ) {
		Error = "wrong header";
		return false;
	}

	return true;
}

void startProduction () {
	File f;

	try {
		if ( Source instanceof File ) {
			f = (File) Source;
			if ( !f.exists() ){
				errorNotify();
				return;
			}

			In = new FileInputStream( (File) Source);
		}
		else if ( Source instanceof URL ) {
			In = ((URL)Source).openStream();
		}

		readHeader();
		if ( !Consumers.isEmpty() )
			readPixelData();

		In.close();
	}
	catch ( Exception x ) {
		x.printStackTrace();
		errorNotify();
	}
}
}
