/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _WIDGET_H
#define _WIDGET_H

#include <list>
#include <sigc++/object.h>
#if SIGC_MAJOR_VERSION == 1 && SIGC_MINOR_VERSION == 0
#include <sigc++/signal_system.h>
#else
#include <sigc++/signal.h>
#endif

#include <wftk/rect.h>
#include <wftk/screenarea.h>
#include <wftk/color.h>
#include <wftk/surface.h>
#include <wftk/focus.h>

namespace wftk {

/** Widget Baseclass.
 */
class Widget : public ScreenArea, virtual public SigC::Object
{
 public:
  /// Contstructor
  Widget();
  /// Constructor which sets the widget background from a given Surface
  explicit Widget(const Surface& backgrnd, bool tileIt = true);
  /// Constructor which sets the widget background from a given Resource
  explicit Widget(Surface::Resource* backgrnd, bool tileIt = true);
  /// Constructor which sets the widget background from a given named string 
	/// (string is the name of a Resource)
  explicit Widget(const std::string& backgrnd, bool tileIt = true);
  /// Destructor
  virtual ~Widget();

  /// Set background image.
  void setBackground(const Surface& backgrnd, bool tileIt = true);
  /// Set background image.
  void setBackground(Surface::Resource* backgrnd, bool tileIt = true);
  /// set background from named resource
  void setBackground(const std::string&, bool tileIt = true);
  /// clear background image
  void clearBackground() {setBackground(0);}
  /// get the current background
  Surface::Resource* getResourceBackground() const {return backgrnd_;}

  ///setting background color.
  void setColor(const Color &color);
  /// set background color from named resource
  void setColor(const std::string&);
  /// need this one due to needing Color constructor from const char*
  void setColor(const char* name) {setColor(std::string(name));}

  ///to set the background colour when the widget is disabled
  void setDisabledColor(const Color& color);
  /// set disabled color from named resource
  void setDisabledColor(const std::string&);
  /// and the const char* version, just like for setColor()
  void setDisabledColor(const char* name) {setDisabledColor(std::string(name));}

  ///enable this widget tree
  void enable();
  /**disable this widget tree. 
     The widget tree gets greyed and the members ignore any events.
  */
  void disable();

  ///
  virtual bool mouseEvent(const Point&, const Point&, Mouse::Button) {return false;}
  ///
  virtual bool buttonEvent(Mouse::Button, bool pressed, const Point&)
	{if (activateOnClick_) checkGrabFocus(); return false;}
  ///
  virtual void gainedMouse() {if (activateOnMouse_) checkGrabFocus();}
  /// keyboard event handler
  virtual bool keyEvent(const SDL_keysym& sym, bool pressed) {return false;}
  /// focus gain handler
  virtual void gainedFocus() {}
  /// focus loss handler
  virtual void lostFocus() {}

  /// returns whether this widget is enabled
  bool isEnabled() const;

  /// returns the background color
  const Color& color() const {return color_;}

  /// true if the widget has keyboard focus
  bool hasFocus() const {return Focus::instance()->currentFocus() == this;}
  /// release the keyboard focus
  bool releaseFocus() {return hasFocus() && Focus::instance()->grab(0);}
  /// grab the keyboard focus
  bool grabFocus() {return acceptsFocus() && Focus::instance()->grab(this);}
  /// set whether to grab focus on a mouse click
  void setClickToFocus(bool val = true);
  /// set whether to grab focus on mouse entry
  void setPointToFocus(bool val = true);
  /// get the next focus in the focus list
  Widget* nextFocus() {return Focus::instance()->nextFocus();}
  /// get the previous focus in the focus list
  Widget* prevFocus() {return Focus::instance()->prevFocus();}
  /// return whether this widget accepts keyboard focus
  bool acceptsFocus() const {return activateOnClick_ || activateOnMouse_;}
 
  /// set background color and image from named resources
  void getResourceBackground(const std::string&);

 protected: 

  virtual void draw(Surface&, const Point& offset, const Region&);
  virtual void drawAfter(Surface&, const Point& offset, const Region&);

  virtual void handleResize(Uint16 w, Uint16 h);

  /// used in Widget's handleResize() implementation,
  /// many widgets can reimplement this instead of handleResize()
  virtual bool isOpaque() const;

  virtual void setPackingInfo();

 private:
  /**background image. 
     may be NULL
  */
  Surface::Resource *backgrnd_;
  /// scaled/tiled copy of background used for actual blitting
  Surface scaled_;
  /// set the value of scaled_, based on backgrnd_ and tileBackground_
  void setScaled();

  ///background color.
  Color color_;

  //disable bg color
  Color colorDis_;

  ///
  bool disabled_;

  ///
  bool tileBackground_;

  ///
  bool activateOnMouse_;
  ///
  bool activateOnClick_;
  ///
  void checkGrabFocus();
};

} // namespace wftk

#endif // _WIDGET_H
