/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _SLIDER_H
#define _SLIDER_H

#include <wftk/widget.h>

namespace wftk {
    
/** Slider Class.
*/
class Slider : public Widget
{
 public:
  /// Enumeration for visual alignment of this Slider
  enum Orientation { VERTICAL, HORIZONTAL };
  /// Create a new Slider using a given Orientation and pixel thickness
  explicit Slider(Orientation orient, unsigned thickness = 25);
  ///
  virtual ~Slider();
  
  /// Set the color of the index button of this slider
  void setButtonColor(const Color&);
  /// Set the index button Surface of this slider, using direct reference
  void setButtonSurface(const Surface&);
  /// Set the index button Surface, using a Resource pointer
  void setButtonSurface(Surface::Resource*);
  /// Set the index button Surface, using a named Resource 
  void setButtonSurface(const std::string&);

  /// Set the minimal and maximal values reportable by this Slider
  void setRange(int min, int max);
  /// Set the amount moved with calls to increase() and decrease()
  void setDelta(int delta);
  
  /// Move the slider towards the minimum range, by Delta, if possible
  virtual void increase();
  /// Move the slider towards maximum range, by Delta, if possible
  virtual void decrease();
  
  /// Set the slider to an absolute position, if within range
  void setValue(int);
  /// Return the value representing current position in range
  int value() const { return (int)(value_+0.45); };
  
  /// Signal that the value of this slider has changed
  SigC::Signal1<void, int> valueChanged;

  ///
  virtual bool mouseEvent(const Point& pos, const Point& rel, Mouse::Button mask);
  ///
  virtual bool buttonEvent(Mouse::Button, bool pressed, const Point&);
  ///
  //virtual void lostMouse() {moving_ = false;}
 protected: 
  ///
  virtual void draw(Surface&, const Point&, const Region&);
  ///
  virtual void handleResize(Uint16 w, Uint16 h);
  ///
  virtual void setPackingInfo();
 private:
  ///
  Surface::Resource* buttonSurface_;
  ///
  void setScaledButtonSurface();
  ///
  Surface scaledSurface_;
  ///
  Color buttonColor_;
  ///
  Rect button_;
  ///
  Orientation orientation_;
  ///
  unsigned thickness_;
  ///
  float value_;
  ///
  int delta_;
  ///
  int min_;
  ///
  int max_;
  ///
  Point position_;
  ///
  bool moving_;
  ///
  float delta_per_pixel_;
};

} // namespace wftk

#endif // _SLIDER_H
