/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _SAFE_SET_H_
#define _SAFE_SET_H_

#include <set>
#include <map>

namespace wftk {

// This is basically std::set<> without iterators.
// Using the for_each() functions instead makes it possible
// to do locking so insert()/erase() calls that would otherwise
// invalidate the iterators don't.

template<class C>
class SafeSet
{
 public:
  SafeSet() : lock_(0) {}
  SafeSet(const SafeSet& s) : map_(s.map_), deleted_(s.deleted_),
	pending_(s.pending_), lock_(0) {if(s.lock_) cleanup();}

  /// returns true on successful insertion, false otherwise
  bool insert(C*);
  /// returns true on successful erasure, false otherwise
  bool erase(C*);

  /// returns true if the argument is in the set, false otherwise
  bool has(C*) const;

  // These functions are for_each() instead of forEach() to
  // conform to the naming of the STL for_each function

  void for_each(void (*func)(C*))
  {
    ++lock_;
    for(MapIterator I = map_.begin(); I != map_.end(); ++I)
      if(I->second)
        func(I->first);
    if(--lock_ == 0)
      cleanup();
  }
  template<class D>
  void for_each(void (*func)(C*, D), D d)
  {
    ++lock_;
    for(MapIterator I = map_.begin(); I != map_.end(); ++I)
      if(I->second)
        func(I->first, d);
    if(--lock_ == 0)
      cleanup();
  }
  void for_each(void (C::*func)(void))
  {
    ++lock_;
    for(MapIterator I = map_.begin(); I != map_.end(); ++I)
      if(I->second)
        (I->first->*func)();
    if(--lock_ == 0)
      cleanup();
  }
  template<class D>
  void for_each(void (C::*func)(D), D d)
  {
    ++lock_;
    for(MapIterator I = map_.begin(); I != map_.end(); ++I)
      if(I->second)
        (I->first->*func)(d);
    if(--lock_ == 0)
      cleanup();
  }

 private:
  SafeSet& operator=(const SafeSet&);

  void cleanup();

  // A map from instances to 'valid' flags
  typedef typename std::map<C*,bool> Map;
	typedef typename Map::iterator MapIterator;
	typedef typename Map::const_iterator MapConstIterator;
	typedef typename Map::value_type MapValueType;
  Map map_;
  typedef typename std::set<C*> Set;
	typedef typename Set::iterator SetIterator;
  Set deleted_;
  Set pending_;
  unsigned long lock_;
};

template<class C>
bool SafeSet<C>::insert(C* c)
{
  if(!lock_)
    return map_.insert(Map::value_type(c, true)).second;

  MapIterator I;
  if((I = map_.find(c)) != map_.end() && I->second)
    return false; // reinserting a valid iterator in the main map

  return pending_.insert(c).second;
}

template<class C>
bool SafeSet<C>::erase(C* c)
{
  MapIterator I = map_.find(c);

  assert(I == map_.end() || I->second || lock_);

  if(I == map_.end() || !I->second) {
    // We may be erasing an id that was added inside the current lock
    SetIterator J;
    if(lock_ && (J = pending_.find(c)) != pending_.end()) {
      pending_.erase(J);
      return true;
    }
    return false;
  }

  if(lock_) {
    bool success = deleted_.insert(c).second;
    assert(success);
    I->second = false;
  }
  else
    map_.erase(I);

  return true;
}

template<class C>
bool SafeSet<C>::has(C* c) const
{
  MapConstIterator I = map_.find(c);

  assert(I == map_.end() || I->second || lock_);

  if(I != map_.end() && I->second)
    return true;

  if(!lock_)
    return false;

  return pending_.find(c) != pending_.end();
}

template<class C>
void SafeSet<C>::cleanup()
{
  // Must do deletion first, since the memory for deleted
  // instances may have been reallocated to new instances.
  // Thus, while pointers in map_ and pending_ are separately
  // unique, they may overlap

  for(SetIterator I = deleted_.begin(); I != deleted_.end(); ++I) {
    MapIterator J = map_.find(*I);
    assert(J != map_.end());
    assert(!J->second);
    map_.erase(J);
  }
  deleted_.clear();

  for(SetIterator I = pending_.begin(); I != pending_.end(); ++I) {
    MapValueType val(*I, true);
    bool success = map_.insert(val).second;
    assert(success);
  }
  pending_.clear();
}

} // namespace

#endif
