/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _REF_MAP_H_
#define _REF_MAP_H_

#include <map>
#include <cassert>

namespace wftk {

// This is a map which keeps single instances of its value
// type available for refcounted keys. You write the create()
// function which is used when an unknown key is ref'ed
// for the first time.

template<class Key, class T, class Cmp = std::less<Key> >
class RefMap
{
 public:
  RefMap(bool autodelete = true) : autodelete_(autodelete) {}
  virtual ~RefMap()
  {
    // We can't delete elements from the map if other things
    // still have references to them. The best we can do
    // is assert that all references have been dropped, and
    // rely on the users of this class to handle that.
    assert(map_.empty());
  }

  T* ref(const Key& key)
  {
    ValType& val = map_[key]; // find, and create if not already present
    if(!val.value) // must be newly created element, from new key
      val.value = create(key);
    ++val.refcount;
    return val.value;
  }

  void unref(const Key& key)
  {
    typename MapType::iterator I = map_.find(key);
    assert(I != map_.end());
    if(--I->second.refcount == 0) {
      delete I->second.value;
      map_.erase(I);
      if(autodelete_ && map_.empty())
        delete this;
    }
  }

  bool empty() const {return map_.empty();}

 private:
  virtual T* create(const Key&) = 0;

  struct ValType
  {
    ValType() : value(0), refcount(0) {}

    T* value;
    unsigned long refcount;
  };
  typedef std::map<Key, ValType, Cmp> MapType;
  MapType map_;

  bool autodelete_;
};

} // namespace

#endif
