/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _POLL_H_
#define _POLL_H_

#include <set>
#include <sigc++/object.h>
#include <SDL/SDL_types.h>

#if defined( __WIN32__ ) || defined( __CYGWIN32__ )
#include <winsock2.h>
#endif

#include "application.h"

namespace wftk {

class PollData;

/// Base class for watching for socket events
class PollBase : virtual public SigC::Object
{
 public:
  /// raw socket descriptior
#if defined( __WIN32__ ) || defined( __CYGWIN32__ )
  typedef SOCKET Socket;
#else
  typedef int Socket;
#endif

  enum {
    READ = 1 << 0,
    WRITE = 1 << 1,
    EXCEPT = 1 << 2,
    MASK = (1 << 3) - 1
  };

  PollBase();
  virtual ~PollBase();

  /// Poll the sockets, blocking if wait > 0
  static void poll(Uint32 wait);

 protected:
  // wrappers for some PollData functions, so we don't
  // have to export fd_set

  /// add a socket to the set to be polled
  static void addPoll(PollData&, Socket, int mask);
  /// check if a socket has data
  static int checkPoll(PollData&, Socket, int mask);
  /// increase PollData refcount
  static void refPoll(PollData&);
  /// decrease PollData refcount
  static void unrefPoll(PollData&);

  // virtual functions to implement polling for generic sets of sockets

  /// add your sockets to PollData in this callback using addPoll()
  virtual void setup(PollData&) = 0;

  /**
   * In this callback, use checkPoll() to see if your sockets are ready, and
   * push an event onto the application event queue if they are.
   **/
  virtual void pushEvent(PollData&) = 0;

 private:
  static std::set<PollBase*> polls_;
};

/// Watch a socket for data
class Poll : public PollBase
{
  /// create a poll to watch a socket
  Poll(Socket, int mask);
  ///
  ~Poll();

  /// get the polling mask
  int mask() const {return mask_;}

  /// Set the polling mask
  void setMask(int m) {mask_ = m;}

 protected:

  virtual void setup(PollData& data) {addPoll(data, socket_, mask_);}
  virtual void pushEvent(PollData&);

 private:
  // unimplemented
  Poll(const Poll&);
  Poll& operator=(const Poll&);

  /// Emitted when the poll is triggered
  virtual void event(int mask) = 0;

  typedef Application::FloatingEvent<Poll> BaseEvent;
  friend class BaseEvent;
  class Event : public BaseEvent
  {
   public:
    Event(Poll& poll, int mask) : BaseEvent(poll), mask_(mask) {}

    virtual void operator()() {if(data()) data()->event(mask_);}

   private:
    int mask_;
  };

  ///
  Socket socket_;
  ///
  int mask_;

  BaseEvent* currentEvent_;
};

} // namespace

#endif
