/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "poll.h"

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

#include <SDL/SDL_timer.h>
#include <cerrno>
#include <cassert>

#include "application.h"
#include "refobj.h"

#if defined( _WIN32 ) || defined( __CYGWIN32__ )
#include <winsock.h>
#include <iostream>
#else
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#endif

std::set<wftk::PollBase*> wftk::PollBase::polls_;

namespace wftk {

class PollData : public RefCountObj {
 public:
  PollData() : RefCountObj(false) {clear();}

  void clear();

  void add(Poll::Socket, int mask);
  int check(Poll::Socket, int mask); // returns 'found' mask

  bool poll(Uint32 wait);

 private:
  // unimplemented
  PollData(const PollData&);
  PollData& operator=(const PollData&);

  fd_set read, write, except;
  Poll::Socket maxfd;
};

} // namespace

void wftk::PollData::clear()
{
  FD_ZERO(&read);
  FD_ZERO(&write);
  FD_ZERO(&except);
  maxfd = 0;
}

void wftk::PollData::add(Poll::Socket s, int mask)
{
  mask &= Poll::MASK;
  if(!mask || s < 0)
    return;

  if(mask & Poll::READ)
    FD_SET(s, &read);
  if(mask & Poll::WRITE)
    FD_SET(s, &write);
  if(mask & Poll::EXCEPT)
    FD_SET(s, &except);

  if(s >= maxfd)
    maxfd = s + 1;
}

int wftk::PollData::check(Poll::Socket s, int mask)
{
  if(s < 0 || s >= maxfd)
    return 0;

  int out_mask = 0;

  if((mask & Poll::READ) && FD_ISSET(s, &read))
    out_mask |= Poll::READ;
  if((mask & Poll::WRITE) && FD_ISSET(s, &write))
    out_mask |= Poll::WRITE;
  if((mask & Poll::EXCEPT) && FD_ISSET(s, &except))
    out_mask |= Poll::EXCEPT;

  return out_mask;
}

bool wftk::PollData::poll(Uint32 wait)
{
#ifdef __WIN32__
  // broken win32 select()
  if(maxfd == 0) {
    SDL_Delay(wait);
    return false;
  }
#endif

  struct timeval wait_time = {wait / 1000, wait % 1000};
  int result = select(maxfd, &read, &write, &except, &wait_time);

  if(result > 0)
    return true;
  else if(result < 0) {
#ifdef EINTR
    if(errno == EINTR) // got a ctrl-c
      exit(-1);
#endif
    throw Fatal(std::string("PollData::poll() got error from select(): ") + strerror(errno));
  }

  return false;
}

wftk::PollBase::PollBase()
{
  bool success = polls_.insert(this).second;
  assert(success);
}

wftk::PollBase::~PollBase()
{
  bool success = polls_.erase(this);
  assert(success);
}

void
wftk::PollBase::addPoll(PollData& data, Socket s, int mask)
{
  data.add(s, mask);
}

int
wftk::PollBase::checkPoll(PollData& data, Socket s, int mask)
{
  return data.check(s, mask);
}

void
wftk::PollBase::refPoll(PollData& data)
{
  data.ref();
}

void
wftk::PollBase::unrefPoll(PollData& data)
{
  data.unref();
}

wftk::Poll::Poll(Socket s, int m) : socket_(s), mask_(m), currentEvent_(0)
{
}

wftk::Poll::~Poll()
{
  if(currentEvent_)
    currentEvent_->clear();
}

void wftk::PollBase::poll(Uint32 wait)
{
  PollData* data = new PollData();

  std::set<PollBase*>::iterator I;

  for(I = polls_.begin(); I != polls_.end(); ++I)
    (*I)->setup(*data);

  if(data->poll(wait))
    for(I = polls_.begin(); I != polls_.end(); ++I)
      (*I)->pushEvent(*data);

  // Drop the ref we got when we created the PollData,
  // the events hold references if they need it.
  data->unref();
}

void wftk::Poll::pushEvent(PollData& data)
{
  int mask = checkPoll(data, socket_, mask_);
  if(mask)
    Application::instance()->pushEvent(new Event(*this, mask));
}

