/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "movie.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"

#include <sigc++/object_slot.h>

#include <assert.h>

namespace wftk {

Movie::Movie() :
  ScreenArea(),
  index_(0),
  myTimer_(500)
{
  myTimer_.alarm.connect(slot(*this, &Movie::tick));
  myTimer_.halt();

  setPackingInfo();
  handleResize(width(), height());
}

Movie::~Movie()
{
  for(unsigned i = 0; i < frames_.size(); ++i)
    if(frames_[i])
      frames_[i]->free();
}

void
Movie::draw(Surface& surf, const Point& offset, const Region& r)
{
  Surface::Resource* res = frames_[index_];

  if(!res)
    return;

  const Surface* frame = res->res();

  assert(frame);

  //center the image
  Region dest = Rect((width()-frame->width())/2 + offset.x,
		     (height()-frame->height())/2 + offset.y,
		     frame->width(), frame->height());
  dest &= r;
  
  frame->blit(surf, offset, dest);
}

void Movie::start()
{
  myTimer_.run();
  invalidate();
}

void Movie::stop()
{
  myTimer_.halt();
}

void Movie::load(const std::vector<Surface::Resource*>& data)
{
  for(unsigned i = 0; i < frames_.size(); ++i)
    if(frames_[i])
      frames_[i]->free();

  frames_.clear();

  frames_ = data;

  for(unsigned i = 0; i < frames_.size(); ++i)
    if(frames_[i])
      frames_[i]->bind();

  invalidate();
  
  index_ = 0;

  setPackingInfo();
  handleResize(width(), height());
}

void Movie::setDelay(unsigned int data)
{
  myTimer_.setInterval(data);
}

void Movie::tick()
{
  invalidateCurrentFrame();

  index_++;
  if(index_ >= (int)(frames_.size()))
    index_ = 0;

  invalidateCurrentFrame();
}

void Movie::handleResize(Uint16 w, Uint16 h)
{
  Uint16 min_width = (frames_.size() != 0) ? 0xffff : 0;
  Uint16 min_height = min_width;

  for(unsigned i = 0; i < frames_.size(); ++i) {
     if(min_width > frames_[i]->res()->width())
       min_width = frames_[i]->res()->width();
     if(min_height > frames_[i]->res()->height())
       min_height = frames_[i]->res()->height();
  }

  Rect opaque;
  opaque.x = (w > min_width) ? (w - min_width) / 2 : 0;
  opaque.y = (h > min_height) ? (h - min_height) / 2 : 0;
  opaque.w = (w > min_width) ? min_width : w;
  opaque.h = (h > min_height) ? min_height : h;

  setShape(Rect(0, 0, w, h), opaque);
}

void Movie::setPackingInfo()
{
  Uint16 width = 0, height = 0;

  for(unsigned i = 0; i < frames_.size(); ++i) {
     if(width < frames_[i]->res()->width())
       width = frames_[i]->res()->width();
     if(height < frames_[i]->res()->height())
       height = frames_[i]->res()->height();
  }

  packing_info_.x.expand = packing_info_.y.expand = false;
  packing_info_.x.min = packing_info_.x.pref = width;
  packing_info_.y.min = packing_info_.y.pref = height;
}

void Movie::invalidateCurrentFrame()
{
  const Surface::Resource* res = frames_[index_];

  if(!res || !res->res())
    return;

  Rect area = res->res()->rect();

  area.x = (area.w < width()) ? (width() - area.w)/2 : 0;
  area.y = (area.h < height()) ? (height() - area.h)/2 : 0;

  invalidate(area);
}
  
} // namespace

