/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  This file was not based on Libuta.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _MARSHAL_H
#define _MARSHAL_H

#include <sigc++/marshal.h>

namespace wftk {

/**
 * A custom marshaller for bools.
 *
 * This is a custom marshaller for signals which return bool
 * used by wftk. It halts signal emmision if a callback returns
 * 'true'. The signal returns 'true' if any of the callbacks did.
 *
 * It is used in cases where we won't care where the signal comes 
 * from, we just want to know if it fires for any case (e.g. mouse
 * entering our application window)
 * 
 * @author R Steinke
 */
class BoolMarshal
{
  public:
    /// return type expected from the callback
    typedef bool InType;
    /// return type passed to the signal
    typedef bool OutType;

    BoolMarshal() : retval_(false) {}

    /// value returned to the calling signal
    OutType value() {return retval_;}

    /// handle the return value of a callback
    bool marshal(const InType& val)
    {
#if SIGC_MAJOR_VERSION == 1 && SIGC_MINOR_VERSION == 0
      if (SigC::RetCode::check_ignore())
        return false;
#endif
      retval_ = val;
      return val;
    }

    /// the default value to return if no callback is called
    static bool default_value() {return false;}

  private:
   bool retval_;
};

} // namespace

#endif
