//
//  The Worldforge Project
//  Copyright (C) 1998,1999  The Worldforge Project
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as 
//  published by the Free Software Foundation; either version 2.1 of the
//  License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, SA.
//
//  For information about Worldforge and its authors, please contact
//  the Worldforge Web Site at http://www.worldforge.org.
//

// Originally written by Karsten Laux
// C++ and streaming modifications by Ron Steinke

#ifndef _WFTK_DEBUG_H_
#define _WFTK_DEBUG_H_

#ifdef DEBUG
#include <iostream>
#endif

namespace wftk {
	
/** The Debug class handles the debug channels.  It also has
 * a few convenience classes and methods to make debugging
 * cleaner. 
 *
 * To use the debugging output, create an instance of the
 * Debug::out class, and then write to it using the &lt;&lt;
 * operator.  Use the 'Debug::endl' symbol instead of
 * std::endl.
 *
 * The debug() and debugN() macros are now deprecated.
 */
class Debug {
 public:

  /// A list of library-related debugging channels
  /**
   * Flags 0-15 appear reserved for user development,
   * 16-31 for wftk library debugging. This is a list
   * specifying what the wftk library flags are used for.
   **/
  enum {
    INVALIDATE = 1 << 16,
    GENERIC = 1 << 17,
    DRAWING = 1 << 18,
    MAINLOOP = 1 << 19,
    STARTUP = 1 << 20, // and shutdown
    SOUND = 1 << 21,
    EVENTS = 1 << 22,
    WIDGET_CREATE = 1 << 23, // and destroy
    TEXT_WIDGETS = 1 << 24,
    FONTS = 1 << 25,
    PACKING = 1 << 26,
    OPENGL = 1 << 27,
    DRAW_TIMING = 1 << 28,
    APP_MASK = 0xffff,
    LIB_MASK = 0xffff0000
  };

  /// The type of the debugging channel mask
  typedef unsigned long Mask;

  /// Set the debugging channel mask
  static void init(Mask mask)
  {
#ifdef DEBUG
    if(mask)
      std::cerr<<"debugging enabled.("<<mask<<")"<<std::endl;
    else if(mask_)
      std::cerr<<"debugging disabled."<<std::endl;
    mask_ = mask;
    out.valid_ = (mask != 0);
#endif
  }

  /// Add a debugging channel to the mask
  static void addChannels(Mask mask)
  {
#ifdef DEBUG
    if(mask)
      out.valid_ = true;
    mask_ |= mask;
#endif
  }

  /// Add a debugging channel to the mask
  static void removeChannels(Mask mask)
  {
#ifdef DEBUG
    mask_ &= ~mask;
    if(!mask_)
      out.valid_ = false;
#endif
  }

  /// Create a generic Debug instance
  Debug() : valid_(mask_ != 0) {}
  /// Create a Debug instance for a particular channel
  Debug(Mask mask) : valid_((mask_ & mask) != 0) {}

  /// Like std::cerr, if DEBUG is defined and debugging is turned on
  static Debug out;
  /// Like std::cerr for a particular debugging channel.
  static Debug& channel(Mask mask) {return (mask_ & mask) ? out : dummy_;}

  /// Write to std::cerr if debugging is turned on
  template<class C>
  Debug& operator<<(const C& c)
  {
#ifdef DEBUG
    if(valid_)
      std::cerr << c;
#endif // DEBUG
    return *this;
  }

  /// cast to bool: true if debugging is turned on for this channel
  operator bool() const
  {
#ifdef DEBUG
    return valid_;
#else
    // we could return valid_ here too, but it wouldn't optimise as well
    return false;
#endif
  }

  /// A class that emulates std::endl when DEBUG is defined, but doesn't require &lt;iostream&gt; when it's not
  class Endl
  {
    // don't need this if DEBUG isn't defined, since
    // 'out << endl;' doesn't write to std::cerr in that case
#ifdef DEBUG
    inline friend std::ostream& operator<<(std::ostream& os, const Endl& e)
    {
      os << std::endl;
      return os;
    }
#endif
  };

  /// The Debug std::endl emulator
  static Endl endl;

  /// A class that emulates std::flush when DEBUG is defined, but doesn't require &lt;iostream&gt; when it's not
  class Flush
  {
    // don't need this if DEBUG isn't defined, since
    // 'out << flush;' doesn't write to std::cerr in that case
#ifdef DEBUG
    inline friend std::ostream& operator<<(std::ostream& os, const Flush& e)
    {
      os << std::flush;
      return os;
    }
#endif
  };

  /// The Debug std::flush emulator
  static Flush flush;

 private:
  bool valid_;
  static Mask mask_;
  static Debug dummy_;
};

} // namespace

#endif // _DEBUG_H_
