/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _BUTTON_H
#define _BUTTON_H

#include <wftk/singlecontainer.h>
#include <wftk/timer.h>
#include <wftk/font.h>

namespace wftk {
    
/** Button Class.
*/
class Button : public SingleContainer
{
 //OBJECT

 public:
  ///
  Button();
  /**
   * Explicit Constructor
   *
   * This creates a Button type object. It's not really inteded to be used 
   * as standalone object, but it's a parent for other buttons, eg. the 
   * PushButton class.
   *\param text the text the button should have, defaults to an empty string
   *\param font the font the button text should have, defaults to button_font
   */
  explicit Button(const std::string& text, 
		  const Font &font=Font::registry.find("button_font"));

  // a signal with an overloaded connect()
  class ButtonSignal : public SigC::Signal0<void>
  {
   public:
    /**
     * the constructor takes a Button pointer to pass
     * to slots which use the second version of connect(),
     * and an optional sample name to play at event time
     **/
    ButtonSignal(Button* b, const std::string& sample = "") :
	button_(b), sample_(sample) {}

    SigC::Connection connect(const SigC::Slot0<void>& slot);
    SigC::Connection connect(const SigC::Slot1<void,Button*>& slot);

    void operator()();

   private:
    Button* button_;
    std::string sample_;
  };

  /**
   * SigC signal triggered when button is pressed,
   * also plays the named Sample "press"
   */
  ButtonSignal pressed;
  /**
   * SigC signal triggered when button is released,
   * also plays the named Sample "release"
   */
  ButtonSignal released;
  /**
   * SigC signal triggered when button is clicked
   */
  ButtonSignal clicked;
 
  /**
   * press the button by calling a function
   *
   * This is used if the programmer for some reason needs to press a button
   * without the user doing so
   */
  void press();
  /**
   * release the button by calling a function
   *
   * This is used to release a key without user input (e.g. for unsetting
   * buttons in a radio-button group)
   */
  void release();
  /**
   * repeat the click on the button
   *
   * This is called if the the click needs to be repeated.
   */
  void repeat();
  /**
   * toggle the button, i.e. set it to pressed if it isn't, and vice versa
   */
  void toggle();
  
  /**
   * check if the button is pressed right now
   *\return true if the button is pressed, false otherwise
   */
  bool isPressed() const {return isPressed_;}
  /**
   * set the repeat interval
   *\param repeat new interval in ms, 0 turns of repeating
   */
  void setRepeat(unsigned repeat = 100);
  
  /**
   * turn repeating on or off
   *\overload
   *\param repeat true to turn on repeating, false to turn it off
   */
  void setRepeat(bool repeat);

  /**
   * triggered when a button event happens
   *\param button which mouse button is producing the event
   *\param pressed_val is the button pressed or released
   *\param pos position of the cursor (I think)
   *\return true on a handled button even, false if it's unhandled, I think
   */
  virtual bool buttonEvent(Mouse::Button button, bool pressed_val, 
			   const Point& pos);
  /**
   * triggered when a keyboard event happens
   *\param sym SDL_keysym type, i.e. the key that triggered the event
   *\param pressed_val is the key pressed or released
   *\return true on handled key even, false otherwise
   */
  virtual bool keyEvent(const SDL_keysym& sym, bool pressed_val);
  /**
   * called if the user moves the mouse off the button after pressing it
   *
   * it unsets the pressed state of the button and triggers a released signal
   */
  virtual void lostMouse();
  /**
   * called if the window loses focus
   *
   * It unsets the pressed state without triggering a released signal
   */
  virtual void lostFocus() {isPressed_ = false;}

 private:
  ///state
  bool isPressed_;
  ///
  Timer autoRepeat_;
};

} // namespace wftk

#endif // _BUTTON_H
