/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _WFTK_BOX_H
#define _WFTK_BOX_H

#include <wftk/screenarea.h>

namespace wftk {

/** a box which can hold multiple screen areas
 *
 * Boxes have an Orientation, either vertical or horizontal.
 * Elements (ScreenAreas) inside the box are ordered within the 
 * box using the packBack() and packFront() methods.
 *
 * If you want to mix horizontal and vertical alignments, use
 * boxes within boxes with varying alignments for each.
 */
class Box : public ScreenArea
{
 public:
  /** The orientation of the box
   *
   * Possible values: LEFT_TO_RIGHT, RIGHT_TO_LEFT, TOP_TO_BOTTOM, BOTTOM_TO_TOP
   */
  enum Orientation {
    LEFT_TO_RIGHT,
    RIGHT_TO_LEFT,
    TOP_TO_BOTTOM,
    BOTTOM_TO_TOP
  };

  /// create a box with a particular orientation
  explicit Box(Orientation);

  /// get the box's orientation
  Orientation orientation() const {return orientation_;}
  /// set the box's orientation
  void setOrientation(Orientation);

  /// internal type
  typedef std::list<ScreenArea*> ChildList;

  /// an iterator over the box's children
  class iterator {
   public:
    /// increment
    iterator& operator++() {++iter_; return *this;}
    /// increment
    iterator operator++(int) {iterator tmp = *this; ++iter_; return tmp;}
    /// decrement
    iterator& operator--() {--iter_; return *this;}
    /// decrement
    iterator operator--(int) {iterator tmp = *this; --iter_; return tmp;}

    /// dereference
    operator ScreenArea*() {return *iter_;}
    /// dereference
    operator ScreenArea*() const {return *iter_;}

    /// dereference
    ScreenArea& operator*() const {return **iter_;}
    /// dereference
    ScreenArea* operator->() const {return *iter_;}

   private:
    friend class Box;
    iterator(const ChildList::iterator& iter) : iter_(iter) {}

    ChildList::iterator iter_;
  };

  /// the beginning of the box's list of children
  iterator begin() {return children_.begin();}
  /// the end of the box's list of children
  iterator end() {return children_.end();}

  /// add a child to the box (better to use packFront() and packBack())
  void pack(ScreenArea*, const iterator&);
  /// remove a child from the box
  void erase(const iterator&);

  /** Add a child ScreenArea to the front of the box.
   *
   * Boxes have a notion of Orientation, which defines the order 
   * and direction that contained Widgets are drawn.
   *
   * Screen areas at the 'back' of the container are drawn first, with
   * areas at the 'front' of the container following, in the 
   * direction specified by the Orientation of this Box.
   *
   * Screen areas contained in this box do not overlap.
   */
  void packFront(ScreenArea* sa) {pack(sa, begin());}
  /** add a child ScreenArea to the back of the box
   *
   * Boxes have a notion of Orientation, which defines the order 
   * and direction that contained Widgets are drawn.
   *
   * Screen areas at the 'back' of the container are drawn first, with
   * areas at the 'front' of the container following, in the 
   * direction specified by the Orientation of this Box.
   *
   * Screen areas contained in this box do not overlap.
   */
  void packBack(ScreenArea* sa) {pack(sa, end());}
  /// remove all children from the box
  void clear();

  /// find a child in the box (O(N), but N won't be too large in any reasonable case)
  iterator find(ScreenArea&);

 protected:
  ///
  virtual void setPackingInfo();
  ///
  virtual void handleResize(Uint16 w, Uint16 h);
 private:
  Orientation orientation_;
  // This is different from ScreenArea's list. ScreenArea keeps
  // front-to-back info, this keeps left-to-right (or whatever).
  ChildList children_;

  /// weights for expanding/contracting children
  PackingInfo::Weights weights_;
};

} // namespace wftk

#endif // _WFTK_BOX_H
