/*
  libwftk - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 2002-2003 Malcolm Walker
  Based on code copyright  (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
#include "application.h"
#include "multilineedit.h"
#include "pushbutton.h"
#include "resources.h"
#include "debug.h"
#include "surface.h"
#include "rotozoom.h"
#include "screensurface.h"
#include "box.h"
#include "filler.h"

#include "windowedroot.h"

using namespace wftk;

#include <sigc++/object_slot.h>

#include <iostream>

static void
do_lock(SDL_Surface* sdlSurface_)
{

  //lock surface
  if(SDL_MUSTLOCK(sdlSurface_))
  {
    if(SDL_LockSurface(sdlSurface_) < 0)
    {
      SDL_Delay(10);
      if(SDL_LockSurface(sdlSurface_) < 0)
      {
        std::cerr << "Surface::lock on surface failed twice."  << std::endl;
        std::cerr << "  no handling here yet :-(" << std::endl;
        assert(false);
      }
    }
  }
}

static void do_unlock(SDL_Surface* sdlSurface_)
{
  //unlock
  if(SDL_MUSTLOCK(sdlSurface_))
  {
    SDL_UnlockSurface(sdlSurface_);
  }
}

static SDL_Surface* clone_surf(SDL_Surface* surf)
{
  SDL_Surface* new_surf
    = SDL_CreateRGBSurface(surf->flags,
        surf->w,
        surf->h,
        surf->format->BitsPerPixel,
        surf->format->Rmask,
        surf->format->Gmask,
        surf->format->Bmask,
        surf->format->Amask);

  if(!new_surf)
    return 0;

  assert(new_surf->h == surf->h);
  assert(new_surf->pitch == surf->pitch);

  do_lock(new_surf);
  do_lock(surf);

  memcpy(new_surf->pixels, surf->pixels, surf->h * surf->pitch);

  do_unlock(surf);
  do_unlock(new_surf);

  return new_surf;
}

class SmokeSurface : public Surface, public SigC::Object
{
  public:
    
    SDL_Surface * tex;
    SDL_Surface * backup;
    float angle;
    float zoom;
    float dz, da;

    SmokeSurface::SmokeSurface(Surface s) : Surface(s), angle(0.0f), zoom(1.0f), timer(10) {
//      Surface::Surface(* s);
tex = SDL_CreateRGBSurface(SDL_SWSURFACE, width(), height(), 32, 0x000000ff, 0x0000ff00, 0x00ff0000, 0xff000000);
      backup = SDL_CreateRGBSurface(SDL_SWSURFACE, width() * 2, height() * 2, 32,
          0x000000ff, 0x0000ff00, 0x00ff0000, 0xff000000);
      SDL_LockSurface(sdlSurface_);
      SDL_LockSurface(tex);
      SDL_LockSurface(backup);
      SDL_SetAlpha(sdlSurface_, SDL_SRCALPHA | SDL_RLEACCEL, 255);
      SDL_SetAlpha(tex, SDL_SRCALPHA | SDL_RLEACCEL, 255);
      SDL_SetAlpha(backup, SDL_SRCALPHA | SDL_RLEACCEL, 255);  
      SDL_UnlockSurface(backup);
      SDL_UnlockSurface(tex);
      SDL_UnlockSurface(sdlSurface_);
      tex = clone_surf(sdlSurface_);

      // This timer rotates the smoke texture
      timer.alarm.connect(SigC::slot(*this, &SmokeSurface::rotatezoom));
    }
    
  private:
    Timer timer;

    void rotatezoom() {
      if (!this->sdlSurface_){
        std::cerr << " no SDL surface, bailing\n";
        return;
      }

      SDL_FreeSurface(sdlSurface_);

       sdlSurface_ = rotozoomSurface(tex, angle, zoom, SMOOTHING_OFF);
       

    }

  
};

class SmokeApp : public Application
{
  
protected:
  WindowedRoot* back;

  Surface image32;
  SmokeSurface* smoker;

  Rect smokeRect[10];
  int count;

  Timer appTimer;
  
public:

  SmokeApp(int argc, char** argv) : 
    Application(argc, argv), appTimer(30)
    {
      std::cout << "starting up ..." << std::endl;

      count = 0;

      //(these fonts are used by several gui elements,
      // so you should slways load text_font and button_font
      Font::registry.load("text_font","wf_opal.ttf, 18, 0xF0F0F0FF, 0x808080FF");
      Font::registry.load("button_font","wf_opal.ttf, 16, 0xF0F0F0, 0x808080");

      Surface::registry.load("button170","button.png");
      Surface::registry.load("button170_pressed","button_pressed.png");

      Surface::registry.load("bg", "background.png");

      if (!image32.readFromFile("smoke.png")) {
	      std::cout << "readFromFile(\"smoke.png\") failed!\n";
	      std::cout << "Substituting the WFTK logo.\n";
	      image32 = *Surface::registry.find("wftk_logo");
      }
      image32.setColorKey("black");

      smokeRect[0] = Rect(0, 45, 20, 20);
      smokeRect[1] = Rect(5, 40, 40, 40);
      smokeRect[2] = Rect(10, 35, 60, 60);
      smokeRect[3] = Rect(15, 30, 80, 80);
      smokeRect[4] = Rect(20, 25, 100, 100);
      smokeRect[5] = Rect(25, 20, 120, 120);
      smokeRect[6] = Rect(30, 15, 125, 125);
      smokeRect[7] = Rect(35, 10, 130, 130);
      smokeRect[8] = Rect(40, 5, 140, 140);
      smokeRect[9] = Rect(45, 0, 150, 150);

      Rect window(190, 215, 210, 235);

      back = new WindowedRoot(400, 320);

      RootWindow::instance()->setTitle("LIBWFTK Smoke Demo");

      back->setBackground("bg");

      Box* main_box = new Box(Box::LEFT_TO_RIGHT);
      back->pack(main_box);

      Box* left_box = new Box(Box::TOP_TO_BOTTOM);
      main_box->packBack(left_box);

      left_box->packBack(new MultiLineEdit("Rotozoom demo\nby Malcolm Walker, February 2003"));
      left_box->packBack(back->widget());

      Box* right_box = new Box(Box::TOP_TO_BOTTOM);
      main_box->packBack(right_box);

      right_box->packBack(new Filler(Filler::VERTICAL));

      //a pushbutton
      PushButton* quit_button = new PushButton("Quit");
      //quit_button->setColor(Color(32,32,128));
      right_box->packBack(quit_button);

      //connect this button's click event (SIGNAL) with the application's
      //quit action (SLOT)
      quit_button->clicked.connect(quitSlot());

      smoker = new SmokeSurface(image32);
      smoker->da = 15.0f;
      smoker->dz = 0.15;
      smoker->zoom = 0.1;
      smoker->setColorKey("black");
      smoker->setAlpha(18);

      // This timer draws the smoke texture
      appTimer.alarm.connect(SigC::slot(*this, &SmokeApp::tick));

      // draw the background of the drawing window
      back->clear();
    }

    virtual ~SmokeApp()
    {
      std::cout << "shutting down ..." << std::endl;
      //do your cleaning up here
      delete smoker;
      
      //all widgets are autodeleted ...
    }

    void tick()
    {

      // start modifying the surface
  
      count++;
      // iterate through list of textures
      if (count % 11) {
        // Apply new transformation factors to this texture
        smoker->angle = (float)((int)(smoker->angle + smoker->da) % 360);
        smoker->zoom += smoker->dz;
  
        RootWindow::instance()->screen()->mutex.grab();

        // Apply new tranformation to this texture and blit
        // according to preset location
        smoker->blit(back->target(), smokeRect[count % 10]);

        back->update();

        RootWindow::instance()->screen()->mutex.release();

        if (smoker->zoom > 1.5) {
          smoker->zoom = 0.15;
        }

      } // end of tick()
  
    }

};  // end of class declaration for SmokeApp
  
int main (int argc, char **argv)
{
  Debug::init(Debug::GENERIC);

  return SmokeApp(argc, argv).exec();
}

