// commands.c - TI-99 Commands
//
// Copyright (c) 2002, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "ti99/defs.h"
#include "emu/socket.h"

extern SOCKET *ts10_Stdout;

#ifdef DEBUG

// Dump contents into terminal from memory area
inline void ti99_Dump(TI99_CPU *ti99cpu, SOCKET *dbg,
	uint32 *Addr, uint32 eAddr, uint32 sw)
{
	int     idx;
	char    ascBuffer[17];
	char    *pasc;
	uint32  data;
	uint8   ch;

	while (*Addr <= eAddr) {
		SockPrintf(dbg, "%08X: ", *Addr);
		pasc = ascBuffer;
#if 0
		for (idx = 0; (idx < 16) && (*Addr <= eAddr); idx += 2) {
			data = ReadW(*Addr);
			*Addr += 2;

			SockPrintf(dbg, "%04X%c", data, (idx == 6) ? '-' : ' ');

			ch = data >> 8;
			*pasc++ = ((ch >= 32) && (ch < 127)) ? ch : '.';
			ch = data & 0xFF;
			*pasc++ = ((ch >= 32) && (ch < 127)) ? ch : '.';
		}
#endif
		for (idx = 0; (idx < 16) && (*Addr <= eAddr); idx++) {
			data = ReadB((*Addr)++);
			SockPrintf(dbg, "%02X%c", data, (idx == 7) ? '-' : ' ');
			*pasc++ = ((data >= 32) && (data < 127)) ? data : '.';
		}
		*pasc = '\0';
		SockPrintf(dbg, " |%-16s|\n", ascBuffer);
	}
}

int ti99_CmdAsm(void *dptr, int argc, char **argv)
{
	TI99_CPU *ti99cpu = ((TI99_SYSTEM *)dptr)->Processor;

	printf("Not implemented yet.\n");
	return EMU_OK;
}

int ti99_CmdDisasm(void *dptr, int argc, char **argv)
{
	TI99_CPU *ti99cpu = ((TI99_SYSTEM *)dptr)->Processor;
	char     *strAddr;
	static   uint32 sAddr = 0; // Start address (Initially 0)
	uint32   eAddr = -1;
	int32    count = -1;

	if (argc > 1) {
		sscanf(argv[1], "%x", &sAddr);
		if (strAddr = strchr(argv[1], '-'))
			sscanf(strAddr+1, "%x", &eAddr);
	}

	if (argc > 2)
		sscanf(argv[2], "%d", &count);
	else if (eAddr == -1)
		count = 20;

	// Display disassembly listing.
	if (count > 0) {
		while (count--)
			d99_Disasm(ti99cpu, ts10_Stdout, &sAddr, SWMASK('v'));
	} else {
		while (sAddr < eAddr)
			d99_Disasm(ti99cpu, ts10_Stdout, &sAddr, SWMASK('v'));
	}

	// Allow this command to be repeated with advancing addresses.
	strcpy(ts10_LastCmd, argv[0]);
	ts10_LastCmdFlag = 1;

	return EMU_OK;
}

int ti99_CmdDump(void *dptr, int argc, char **argv)
{
	TI99_CPU *ti99cpu = ((TI99_SYSTEM *)dptr)->Processor;
	static uint32 sAddr = 0;  // Start address (Initially 0)
	uint32        eAddr = -1; // End Address
	char          *strAddr;
	static uint32 sw = SWMASK('v'); // Memory Access Mode (Initially Virtual)

	if (argc > 1) {
		sscanf(argv[1], "%x", &sAddr);
		if (strAddr = strchr(argv[1], '-'))
			sscanf(strAddr+1, "%x", &eAddr);
	}

	if (argc > 2) {
		sscanf(argv[2], "%x", &eAddr);
		eAddr = sAddr + eAddr - 1;
	} else if (eAddr == -1)
		eAddr = sAddr + 0x140 - 1;

	ti99_Dump(ti99cpu, ts10_Stdout, &sAddr, eAddr, sw);

	// Allow this command to be repeated with advancing addresses.
	strcpy(ts10_LastCmd, argv[0]);
	ts10_LastCmdFlag = 1;

	return EMU_OK;
}

#endif /* DEBUG */


COMMAND ti99_Commands[] = {
#ifdef DEBUG
	{ "asm",     "{Not Implemented Yet}",   ti99_CmdAsm     },
	{ "disasm",  "[start[-end]] [count]",   ti99_CmdDisasm  },
	{ "dump",    "[srart[-end]] [length]",  ti99_CmdDump    },
#endif /* DEBUG */
	{ NULL,      NULL,                      NULL           },
};

COMMAND ti99_SetCommands[] = {
	{ NULL, NULL, NULL }
};

COMMAND ti99_ShowCommands[] = {
	{ NULL,   NULL, NULL }
};

