#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "sg_include.h"
#include "sg_err.h"

/* Test code for D. Gilbert's extensions to the Linux OS SCSI generic ("sg")
   device driver.
*  Copyright (C) 1999 D. Gilbert
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2, or (at your option)
*  any later version.

   This program simply "reads" the given file and tests if it is a SCSI
   generic device. If so it outputs some information about it. If a
   "-rc" flag is given then it sends the SCSI "Read Capacity" command to 
   it and outputs the result. If the "-h" flag is given host (adapter)
   information is output.

   Version 0.85 (20010115)

6 byte commands [READ: 0x08, WRITE: 0x0a]:
[cmd ][had|lu][midAdd][lowAdd][count ][flags ]
10 byte commands [EREAD: 0x28, EWRITE: 0x2a, READ_CAPACITY 0x25]:
[cmd ][   |lu][hiAddr][hmAddr][lmAddr][lowAdd][      ][hiCnt ][lowCnt][flags ]
12 byte commands [LREAD: 0xd8, LWRITE: 0xda]:
[cmd ][   |lu][hiAddr][hmAddr][lmAddr][lowAdd][hiCnt ][hmCnt ][lmCnt ][lowCnt]
 ... [      ][flags ]
*/
        
#ifndef SG_GET_VERSION_NUM
#define SG_GET_VERSION_NUM 0x2282
#endif

#define OFF sizeof(struct sg_header)
#define MAX_REPLY 512
#define READ_CAP_DATA_LEN 8
#define HBUFF_SZ 160


int main(int argc, char * argv[])
{
    int sg_fd, res, k, to, sg_tablesize, ver_num;
    unsigned char rcCmdBlk [10] = {0x25, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    unsigned char rcBuff[OFF + sizeof(rcCmdBlk) + MAX_REPLY];
    int rcInLen = OFF + sizeof(rcCmdBlk);
    int rcOutLen;
    unsigned char * buffp = rcBuff + OFF;
    struct sg_header * rsghp = (struct sg_header *)rcBuff;
    struct sg_scsi_id sg_id;
    int num_blocks = 0;
    int sec_size = 0;
    int do_read_capacity = 0;
    int do_host = 0;
    char * file_name = 0;
    char hbuff[HBUFF_SZ];
    struct stat a_st;
    int block_dev = 0;

    for (k = 1; k < argc; ++k) {
        if (0 == strcmp("-rc", argv[k]))
            do_read_capacity = 1;
        else if (0 == strcmp("-h", argv[k]))
            do_host = 1;
        else if (*argv[k] != '-')
            file_name = argv[k];
        else {
            file_name = 0;
            printf("Unknown argument: %s\n", argv[k]);
            break;
        }
    }
    if (0 == file_name) {
        printf("Usage: 'sg_whoami [-rc] [-h] <generic_device>'\n");
        printf("    For example: sg_whoami -rc /dev/sg0\n");
        return 1;
    }
    
    sg_fd = open(file_name, O_RDONLY);
    if (sg_fd < 0) {
        perror("sg_whoami: open error");
        return 1;
    }
    if (fstat(sg_fd, &a_st) < 0) {
        fprintf(stderr, "could do fstat() on fd ??\n");
        close(sg_fd);
        return 1;
    }
    if (S_ISBLK(a_st.st_mode))
        block_dev = 1;
    /* Just to be safe, check we have a sg device by trying an ioctl */
    if (block_dev || (ioctl(sg_fd, SG_GET_TIMEOUT, 0) < 0)) {
        /* perror("ioctl on generic device, error"); */
        printf("sg_whoami: not a sg device, or wrong driver\n");
        return 1;
    }
    printf("scsi%d, channel=%d, device=%d, lun=%d,  scsi_type=%d\n", 
           sg_id.host_no, sg_id.channel, sg_id.scsi_id, sg_id.lun,
           sg_id.scsi_type);

    res = ioctl(sg_fd, SG_GET_TIMEOUT, &k);
    if (res < 0) {
        perror("sg_whoami: ioctl (2) error");
        return 1;
    }
    to = res;

    res = ioctl(sg_fd, SG_GET_SG_TABLESIZE, &sg_tablesize);
    if (res < 0) {
        perror("sg_whoami: ioctl (3) error");
        return 1;
    }

    res = ioctl(sg_fd, SG_GET_VERSION_NUM, &ver_num);
    if (res < 0)
        printf("SG sg_tablesize=%d, timeout=%d\n", sg_tablesize, to);
    else
        printf("SG sg_tablesize=%d, timeout=%d, version_num=%d\n", 
               sg_tablesize, to, ver_num);

    res = close(sg_fd);
    if (res < 0) {
        perror("sg_whoami: close error");
        return 1;
    }
    if ((0 == do_read_capacity) && (0 == do_host))
        return 0;

    sg_fd = open(file_name, O_RDWR);
    if (sg_fd < 0) {
        perror("sg_whoami: can't open RDWR for read capacity");
        return 1;
    }
    if (do_read_capacity) {
        rcOutLen = OFF + READ_CAP_DATA_LEN;
        rsghp->pack_len = 0;                /* don't care */
        rsghp->pack_id = k;
        rsghp->reply_len = rcOutLen;
        rsghp->twelve_byte = 0;
        rsghp->result = 0;
        memcpy(rcBuff + OFF, rcCmdBlk, sizeof(rcCmdBlk));

        res = write(sg_fd, rcBuff, rcInLen);
        if (res < 0) {
            perror("sg_whoami: write (rc) error");
            return 1;
        }
        if (res < rcInLen) {
            printf("sg_whoami: wrote less (rc), ask=%d, got=%d", rcInLen, res);
            return 1;
        }
        
        memset(buffp, 0, 8);
        res = read(sg_fd, rcBuff, rcOutLen);
        if (res < 0) {
            perror("sg_whoami: read (rc) error");
            return 1;
        }
        if (res < rcOutLen) {
            printf("sg_whoami: read less (rc), ask=%d, got=%d", rcOutLen, res);
            return 1;
        }
        if (sg_chk_n_print("after read(rc)", rsghp->target_status,
                           rsghp->host_status, rsghp->driver_status,
                           rsghp->sense_buffer, SG_MAX_SENSE)) {
            num_blocks = 1 + ((buffp[0] << 24) | (buffp[1] << 16) |
                              (buffp[2] << 8) | buffp[3]);
            sec_size = (buffp[4] << 24) | (buffp[5] << 16) | 
                       (buffp[6] << 8) | buffp[7];
            printf("number of sectors=%d, sector size=%d\n", num_blocks,
                   sec_size);
        }
    }
    if (do_host) {
        *(int *)hbuff = sizeof(hbuff) - 1; 
        hbuff[HBUFF_SZ - 1] = '\0';
        res = ioctl(sg_fd, SCSI_IOCTL_PROBE_HOST, hbuff);
        if (res < 0) {
            perror("sg_whoami: SCSI_IOCTL_PROBE_HOST");
            return 1;
        }
        else 
            printf(" hosts present=%d\n  %s\n", res, hbuff);
    }

    res = close(sg_fd);
    if (res < 0) {
        perror("sg_whoami: read capacity close error");
        return 1;
    }
    return 0;
}
