
// Copyright (c) 1996-1999 The University of Cincinnati.
// All rights reserved. 

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Author: Dale E. Martin          dmartin@ece.uc.edu

//---------------------------------------------------------------------------

#include "error_func.hh"
#include "VHDLToken.hh"
#include "IIRBase.hh"
#include "IIR_Identifier.hh"
#include "IIR_TypeDefinition.hh"
#include "IIR_SelectedName.hh"
#include "IIR_Declaration.hh"
#include "set.hh"
#include "sstream-wrap.hh"


extern bool parse_error;

void 
report_error( const char *message, 
	      int line_number, 
	      IIR_Identifier *file_name, 
	      severity how_serious ){

  if( how_serious == FATAL ){
    parse_error = true;
  }

  if( file_name != NULL ){
    cerr << *file_name << ":";
  }
  
  if( line_number != -1 ){
    cerr << line_number << ": ";
  }

  if( how_serious == WARNING ){
    cerr << "warning: ";
  }

  cerr << message << endl;

}

void 
report_error( const string &message,
	      int line_number, 
	      IIR_Identifier *file_name, 
	      severity how_serious ){
  if( how_serious == FATAL ){
    parse_error = true;
  }
  
  report_error( message.c_str(), line_number, file_name, how_serious );
}

void 
report_error (ANTLRToken *tok, const string &message, severity how_serious ) {
  if( how_serious == FATAL ){
    parse_error = true;
  }

  IIR_Identifier *foo = IIR_Identifier::get( tok->getFileName(),
					strlen( tok->getFileName() ));
 
  report_error( message, tok->getLine(), foo, how_serious );

  foo->release();
}

void 
report_error (ANTLRToken *tok, char *msg, severity how_serious) {
  if( how_serious == FATAL ){
    parse_error = true;
  }
 
  IIR_Identifier *foo = IIR_Identifier::get( tok->getFileName(),
					     strlen( tok->getFileName() ));
 
  report_error(msg, tok->getLine(), foo, how_serious );

  foo->release();
}

void 
report_error( IIRBase *if_node, 
	      const string &message,
	      severity how_serious ){
  if( how_serious == FATAL ){
    parse_error = true;
  }

  report_error( message, if_node->get_line_number(), if_node->get_file_name(), how_serious );
}

void 
report_error( IIRBase *if_node, char *msg, severity how_serious ) {
  if( how_serious == FATAL ){
    parse_error = true;
  }

  report_error(msg, if_node->get_line_number(), if_node->get_file_name());
}

void 
report_undefined_symbol( IIR *name ){
  ostringstream err;
  err << "|" << *name << "| is undefined in this scope.";
  report_error( name, err.str() );
}

void 
report_undefined_symbol( IIR_Declaration *prefix, IIR_Name *suffix ){
  ostringstream err;
  err << "|" << *prefix << "." << *suffix << "| is undefined in this scope.";
  report_error( suffix, err.str() );
}

void 
report_undefined_symbol( set<IIR_TypeDefinition> *type_def_set, IIR *name ){
  if( type_def_set == NULL || type_def_set->num_elements() == 0 ){
    ostringstream err;
    err << "|" << *name << "| is undefined in this scope.";
    report_error( name, err.str() );
  }
  // else no error
}


void 
report_ambiguous_error( IIR *error_info, set<IIR_Declaration> *error_set ){
  ostringstream err;

  ASSERT( error_set != NULL );
  ASSERT( error_set->num_elements() > 1 );
  ASSERT( error_set->get_element() != NULL );

  err << "Can't disambiguate symbol |" 
      << *error_set->get_element()->get_declarator() 
      << "| from the following choices:";

  int i;
  IIR_Declaration *current_decl = error_set->get_element();
  for( i = 0; i < error_set->num_elements(); i++ ){
    if( current_decl != NULL ){
      err << "\n|" << *current_decl <<"|";
      
      if(  current_decl->get_file_name() != NULL ){
	err << ", declared in file |" << *current_decl->get_file_name() << "|";
      }

      if( current_decl->get_line_number() != -1 ){
	err << ", line " <<  current_decl->get_line_number();
      }
      err << ".";
    }
    current_decl = error_set->get_next_element();
  }
  
  report_error( error_info, err.str() );
}

void 
report_ambiguous_error( IIR *error_info, set<IIR_TypeDefinition> *error_set ){
  IIR_TypeDefinition *current;
  
  ostringstream err;
  err << "Cannot determine unique type for |" << *error_info << "|.  Choices include:\n";

  current = error_set->get_element();
  while( current != NULL ){
    // I broke this up to make it possible to debug it.
    err << "Type |" << *current->_get_declarator();
    err << "|, with type <" << *current << ">"; 
    if( current->get_file_name() != NULL ){
      err << ", declared in file |" << *current->get_file_name() << "|";
    }

    if( current->get_line_number() != -1 ){
      err << ", line " <<  current->get_line_number();
    }

    err << ".";

    current = error_set->get_next_element();
    if( current != NULL ){
      err << "\n";
    }
  }

  report_error( error_info, err.str() );
}


void 
report_ambiguous_error( IIR_Declaration *prefix, 
			IIR_Name *suffix, 
			set<IIR_Declaration> *error_set ){

  IIR_SelectedName *error_info = new IIR_SelectedName();
  error_info->set_prefix( prefix );
  error_info->set_suffix( suffix );
  
  // This should copy only the line number info over.
  ((IIRScram *)error_info)->_clone( suffix );
  report_ambiguous_error( error_info, error_set );
  delete error_info;
}
