
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer
//          Malolan Chetlur    
//          Umesh Kumar V. Rajasekaran
//          Radharamanan Radhakrishnan
//          Narayanan Thondugulam

#include "IIR_VariableAssignmentStatement.hh"
#include "IIR_AboveAttribute.hh"
#include "IIR_Label.hh"
#include "error_func.hh"
#include "set.hh"
#include "resolution_func.hh"
#include "savant.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"

IIRScram_VariableAssignmentStatement::~IIRScram_VariableAssignmentStatement(){}

void
IIRScram_VariableAssignmentStatement::_publish_vhdl(ostream &_vhdl_out) {

  _publish_vhdl_stmt_label(_vhdl_out);
    
  get_target()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " := ";
  if (get_expression()->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION) {
    _vhdl_out << "(";
  }
  get_expression()->_publish_vhdl(_vhdl_out);
  if (get_expression()->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION) {
    _vhdl_out << ")";
  }
}

void 
IIRScram_VariableAssignmentStatement::_publish_cc( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_::VariableAssignmentStatement_publish_cc" );
  if (get_label() != NULL) {
    // Leaving this hard return here so that the label sits on the left
    // edge of the source.
    _cc_out << "\n";
    get_label()->_publish_cc_lvalue( _cc_out );
    _cc_out << ":" << NL();
  }

  if( get_target()->get_kind() == IIR_AGGREGATE ){
    // Create a temporary aggregate and use that in the variable assignment
    ASSERT ( get_target()->get_subtype() != NULL );
    _cc_out << OS("{") <<
      get_target()->get_subtype()->_get_cc_type_name() <<
      " tempAggregate(";
    if (get_target()->get_subtype()->_is_record_type() == TRUE) {
      _cc_out << "true," << NL();
    }
    
    get_target()->_publish_cc_constructor_args( _cc_out );
    get_target()->_publish_cc_initialization_value( _cc_out );
    _cc_out << ");" << NL();
  }

  SCRAM_CC_REF( _cc_out, "IIRScram_::VariableAssignmentStatement_publish_cc" );  
  _cc_out << OS("assignVariable(");
  if (get_target()->get_kind() != IIR_AGGREGATE) {
    get_target()->_publish_cc_lvalue( _cc_out );
  }
  else {
    _cc_out << " tempAggregate";
  }
  
  _cc_out << ", " << NL();
  get_expression()->_publish_cc_rvalue( _cc_out );
  _cc_out << ", " << NL();
  
  // If the target or expression is an interface declaration and we are
  // currently publishing a subprogram then pick up the ranges of the
  // target and expression from the interface objects rather than from
  // the actual type definitions because the ranges of the parameters passed
  // to the subprogram can be different from those of the actual type
  // definitions.

  // However, the target and expression must have their own limits set in their
  // type_info structures. We can look at the values in their type_info
  // structures and figure it out. So publish just nullInfo here..

  //    if ((_is_currently_publishing_subprogram() == TRUE) &&
  //        (get_target()->_is_interface() == TRUE)) {
  //      get_target()->_publish_cc_bounds( _cc_out );
  //    }
  //    else {
  _cc_out << "nullInfo," << NL() << "nullInfo";
  
  //    if ((_is_currently_publishing_subprogram() == TRUE)
  //        && (get_expression()->_is_interface() == TRUE)) {  
  //      get_expression()->_publish_cc_bounds( _cc_out );
  //    }
  //    else {
  _cc_out << CS(");");

  if (get_target()->get_kind() == IIR_AGGREGATE) {
    _cc_out << CS("}");
  }
}

// This type checking should make sure that any functions are being with
// legal arguments and such...
void 
IIRScram_VariableAssignmentStatement::_type_check(){
  set<IIR_TypeDefinition> *target_lvals = get_target()->_get_rval_set();
  if( target_lvals == NULL ){
    report_undefined_symbol( get_target() );
    return;
  }
  set_expression( get_expression()->_semantic_transform( target_lvals ));

  set<IIR_TypeDefinition> *expression_rvals = get_expression()->_get_rval_set();
  if( expression_rvals == NULL ){
    report_undefined_symbol( get_expression() );
    return;
  }
  set_target( get_target()->_semantic_transform( expression_rvals ));

  get_target()->_type_check( expression_rvals );
  
  // Need to recalculate AFTER the transformation...
  delete target_lvals;
  target_lvals = get_target()->_get_rval_set();
  get_expression()->_type_check( target_lvals );

  delete expression_rvals;
  delete target_lvals;

  target_lvals = get_target()->_get_rval_set();
  expression_rvals = get_expression()->_get_rval_set();

  reconcile_sets( target_lvals, expression_rvals );

  switch( target_lvals->num_elements() ){
  case 0:{
    ostringstream err;

    err << "Incompatible types in assignment: " << *get_target()
	<< " := " << *get_expression();

    report_error( this, err.str() );
    break;
  }
  case 1:{
    IIR_TypeDefinition *type = target_lvals->get_element(); 
    set_expression( get_expression()->_rval_to_decl( type ));
    
    set_target( get_target()->_rval_to_decl( type ) );

    if( get_expression()->_is_readable() == FALSE ){
      ostringstream err;
      err << "Expression |" << *get_expression() << "| cannot be read from.";
      report_error( this, err.str() );
    }

    if( get_target()->_is_writable() == FALSE ){
      ostringstream err;
      err << "Target |" << *get_target() << "| is not writable and cannot be assigned to.";
      report_error( this, err.str() );
    }
    
    if( get_target()->_is_variable() == FALSE ){
      ostringstream err;
      err << "Target |" << *get_target() << "| is not a variable and cannot be assigned"
	  << " through a variable assignment statement.";
      report_error( this, err.str() );
    }

    break;
  }
  default:
    ostringstream err;
    err << "Ambigous assignment - cannot resolve types";
    report_error( this, err.str() );
    break;    
  }

  delete target_lvals;
  delete expression_rvals;
}

void
IIRScram_VariableAssignmentStatement::_get_list_of_input_signals( set<IIR> *list ){
  get_expression()->_get_list_of_input_signals(list);
}

IIR_Boolean
IIRScram_VariableAssignmentStatement::_is_above_attribute_found() {
  ASSERT(get_expression() != NULL);
  return get_expression()->_is_above_attribute_found();
}

void
IIRScram_VariableAssignmentStatement::
_build_above_attribute_set(set<IIR_AboveAttribute> *to_build) {
  ASSERT(get_expression() != NULL);
  get_expression()->_build_above_attribute_set(to_build);
}

IIR*
IIRScram_VariableAssignmentStatement::_clone() {
  IIR *exprclone;
  IIR_VariableAssignmentStatement *clone = new IIR_VariableAssignmentStatement;
  IIR_SequentialStatement::_clone(clone);

  clone->set_target(get_target()->_clone());
  exprclone = get_expression()->_clone();
  clone->set_expression(exprclone);

  return clone;
}

visitor_return_type *IIRScram_VariableAssignmentStatement::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_VariableAssignmentStatement(this, arg);
};
