// Copyright (c) 1996-2000 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//	    Magnus Danielson	cfmd@swipnet.se

//---------------------------------------------------------------------------
#include "IIRScram_StringLiteral.hh"
#include "IIR_ScalarTypeDefinition.hh"
#include "IIR_AccessSubtypeDefinition.hh"
#include "StandardPackage.hh"
#include "savant.hh"
#include "set.hh"
#include "published_file.hh"
#include <cctype>
using std::cerr;

class IIR_AccessTypeDefinition;

IIRScram_StringLiteral::IIRScram_StringLiteral(){
}


IIRScram_StringLiteral::~IIRScram_StringLiteral(){
}


void
IIRScram_StringLiteral::_publish_cc_string( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc_string" );
  IIRScram_TextLiteral::_publish_cc_lvalue( _cc_out );
}


void
IIRScram_StringLiteral::_publish_cc_lvalue( published_file &_cc_out ) {
  IIR_TypeDefinition *type = get_subtype();

  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc" );
  
  if(type != NULL) {
    ASSERT(type->_is_resolved() == TRUE);
    
    if ( type->_is_anonymous() == TRUE && 
	 type != StandardPackage::get_savant_null_type_definition() &&
	 type->_get_base_type() != NULL ){
      type = type->_get_base_type();
    }
    else {
      // Just ensuring other conditions that should be to be okay.
      if ( type->_get_type_mark() != NULL ) {
	// Atta boy, Dale, thanks for the type_mark
	// Just make sure this is not anonymous!!
	ASSERT ( type->_get_type_mark()->_is_anonymous() == FALSE );
	type = type->_get_type_mark();
      }
    }
    
    // _cc_out << "_get_cc_type_name()" can now handle the null type definition
    // case in addition to the old cases.
    _cc_out << type->_get_cc_type_name();
    
    if( type->_is_access_type() == FALSE ){
      _cc_out << OS("(ObjectBase::VARIABLE,");

      if (type->_is_array_type() == TRUE) {
	type->_publish_cc_object_type_info( _cc_out, FALSE, "", TRUE );
	_cc_out << "," << NL() << "-1," << NL();
      }
      
      if ((type->_is_anonymous() == TRUE) || (type->_is_unconstrained_array_type() == TRUE)) {
	_publish_cc_range( _cc_out );
	_cc_out << "," << NL();	
      }
      
      _publish_cc_string( _cc_out );
      
      _cc_out << CS(")");
    }
    else {
      //      ASSERT(IIR_TextLiteral::_cmp(this, "null") == 0);
      _cc_out << "()";
    }
  }  
  else {
    _publish_cc_string( _cc_out );
  }
}
  

void
IIRScram_StringLiteral::_publish_cc_range( published_file &_cc_out ) {
  int i;
  IIR_Int32 length = get_text_length();
  int funkyChars = 0;
  IIR_TypeDefinition *index_type = get_subtype()->_get_index_subtype();

  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc_range" );

  if (index_type->_is_scalar_type() == TRUE) {
    index_type->_get_base_type_left()->_publish_cc_value( _cc_out );
    _cc_out << "," << NL();
  }
  //   _cc_out << "1, to, ";
  // A string can be confined within quotes or percetages but if we 
  // have percentages as delimiters of strings we cannot have quotes inside
  if ((*this)[0] != '%') {
    for( i = 1; i < length-1; i++ ){
      // Check for two consecutive double quotes.
      if (((*this)[i] == '\"') && ((*this)[(i + 1)] == '\"')) {	//"
	i++;
	funkyChars++;
      }
    }
  } else {
    // A string whose delimiters are percentages
    for( i = 1; i < length-1; i++ ){
      // Check for two consecutive double quotes.
      if (((*this)[i] == '%') && ((*this)[(i + 1)] == '%')) {	
	i++;
	funkyChars++;
      }
    }
  }

  if (index_type->_get_base_type_direction()->_is_ascending_range() == TRUE) {
    _cc_out << "ArrayInfo::to," << NL();
    index_type->_get_base_type_left()->_publish_cc_value( _cc_out );
    _cc_out << " + ";
  }
  else {
    _cc_out << " ArrayInfo::downto," << NL();
    index_type->_get_base_type_left()->_publish_cc_value( _cc_out );
    _cc_out << " - ";
  }
  _cc_out << (length - funkyChars - 3); //omit quotes
}


void 
IIRScram_StringLiteral::_publish_cc_state_object_init( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc_state_object_init" );

  _cc_out << " ";
  if(get_subtype()->_is_iir_access_type_definition() != TRUE) {
    _cc_out << "*";
  }
  _cc_out << "(new ";
  _publish_cc_lvalue( _cc_out );
  _cc_out << ")";
}


//This function is used to creat objects of corresponding type 
//when string literals are compared with objects of VHDLTypes
void 
IIRScram_StringLiteral::_publish_cc_condition( published_file &_cc_out ) {
  ASSERT( _get_current_publish_node() != NULL );
  //_current_publish_node is set during conditions in case statements
  IIR_TypeDefinition* type = _get_current_publish_node()->get_subtype();

  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc_condition" );

  ASSERT(type != NULL);
  _cc_out << type->_get_cc_type_name();
  _cc_out << OS("(ObjectBase::VARIABLE,");
  type->_publish_cc_object_type_info(_cc_out, FALSE);
  _cc_out << "," << NL();
  type->_publish_cc_resolution_function_id(_cc_out);
  _cc_out << "," << NL();
  
  if(type->_is_anonymous() == TRUE) {
    _publish_cc_range( _cc_out );
    _cc_out << "," << NL();
  }
  _publish_cc_lvalue( _cc_out );
  _cc_out << CS(")");
}


void 
IIRScram_StringLiteral::_publish_cc_initialization_value( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc_initialization_value" );
  _publish_cc_string( _cc_out );
}


void 
IIRScram_StringLiteral::_publish_cc_without_quotes( published_file &_cc_out ) {
  IIR_Int32 length = get_text_length();
  register int i;

  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc_without_quotes" );

  if((*this)[0] == '\'') {
    i = 1;
  } else {
    i = 0;
  }

  for (; i < length - 1; i++) { 
    _cc_out << (*this)[i];
  }

  if((*this)[length - 1] != '\'') {
    _cc_out << (*this)[length-1];
  }
}


void 
IIRScram_StringLiteral::_publish_cc_function_name( published_file &_cc_out ) {
  IIR_Int32 length = get_text_length();
  IIR_Char c;

  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc_function_name" );

  _cc_out << "savant";
  for (int i = 1; i < length-1; i++) { 
    c = (*this)[i];
    if(isalnum(c)) {
      _cc_out << c;
    } 
    else {
      switch(c) {
      case '=':	
	_cc_out << "Equal"; break;
      case '/':	
	if(i+1 < length-1 && (*this)[i+1] == '=') {
	  _cc_out << "NotEqual"; i++; 
	} else {
	  _cc_out << "Divide";
	}	  
	break;
      case '<':	
	if(i+1 < length-1 && (*this)[i+1] == '=') {
	  _cc_out << "LessThanOrEqual"; i++; 
	} else {
	  _cc_out << "LessThan";
	}	  
	break;
      case '>':	
	if(i+1 < length-1 && (*this)[i+1] == '=') {
	  _cc_out << "GreaterThanOrEqual"; i++; 
	} else {
	  _cc_out << "GreaterThan";
	}	  
	break;
      case '+':	_cc_out << "Plus"; break;
      case '-':	_cc_out << "Minus"; break;
      case '&':	_cc_out << "Concatenate"; break;
      case '*':	
	if(i+1 < length-1 && (*this)[i+1] == '*') {
	  _cc_out << "Pow"; i++; 
	} else {
	  _cc_out << "Multiply";
	}	  
	break;
      default:
	cerr << "IIRScram_StringLiteral::_publish_cc_function_name( published_file &_cc_out )::"
	     << "Unknown operator `" << c << "\'\n";
	abort();
      }	// switch
    } // else 
  } // for
}


const string
IIRScram_StringLiteral::_convert_function_name( IIR_Boolean _is_unary_operator ){
  int length = get_text_length();
  char c;

  string retval = "savant";
  for( int i = 1; i < length-1; i++ ){ 
    c = (*this)[i];
    if (i == 1) {
      c = toupper(c);
    }
    
    if(isalnum(c)) {
      retval += c;
    } 
    else {
      switch(c) {
      case '=':	
	retval += "Equal"; break;
      case '/':	
	if(i+1 < length-1 && (*this)[i+1] == '=') {
	  retval += "NotEqual"; i++; 
	} 
	else {
	  retval += "Divide";
	}	  
	break;
      case '<':	
	if(i+1 < length-1 && (*this)[i+1] == '=') {
	  retval += "LessThanOrEqual"; i++; 
	} 
	else {
	  retval += "LessThan";
	}	  
	break;
      case '>':	
	if(i+1 < length-1 && (*this)[i+1] == '=') {
	  retval += "GreaterThanOrEqual"; i++; 
	} 
	else {
	  retval += "GreaterThan";
	}	  
	break;
      case '+':
	if (_is_unary_operator == TRUE)  {
	  retval += "Unary";
	}
	retval += "Plus";
	break;
      case '-':
	if (_is_unary_operator == TRUE)  {
	  retval += "Unary";
	}
	retval += "Minus";
	break;
      case '&':	retval += "Concatenate"; break;
      case '*':	
	if(i+1 < length-1 && (*this)[i+1] == '*') {
	  retval += "Pow"; i++; 
	} 
	else {
	  retval += "Multiply";
	}	  
	break;
      default:
	cerr << "IIRScram_StringLiteral::_publish_cc_function_name( published_file &_cc_out )::"
	     << "Unknown operator `" << c << "\'\n";
	abort();
      }	// switch
    } // else 
  } // for
  return retval;
}


void 
IIRScram_StringLiteral::_publish_cc_universal_value( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc_universal_value" );

  _publish_cc_string( _cc_out );
}


void
IIRScram_StringLiteral::_publish_cc_bounds( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc_bounds" );

  _cc_out << "nullInfo";
}


void 
IIRScram_StringLiteral::_publish_cc_headers( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc_headers" );
  if(get_subtype() != NULL) {
    get_subtype()->_publish_cc_headers( _cc_out );
  }
}


void 
IIRScram_StringLiteral::_publish_cc_variable_name( published_file &_cc_out ) {
  IIR_Int32 length = get_text_length();
  register int i;

  SCRAM_CC_REF( _cc_out, "IIRScram_StringLiteral::_publish_cc_variable_name" );

  if((*this)[0] == '\'') {
    _cc_out << "_";
    i = 1;
  } else {
    i = 0;
  }

  for (; i < length-1; i++) { 
    if(isalnum((*this)[i])) {
      _cc_out << (*this)[i];
    } else {
      _cc_out << (unsigned int)(*this)[i];
    }
  }

  if((*this)[length-1] != '\'') {
    if(isalnum((*this)[i])) {
      _cc_out << (*this)[length-1];
    } else {
      _cc_out << (unsigned int)(*this)[length-1];
    }
  }
  else {
    _cc_out << "_";
  }
}


void 
IIRScram_StringLiteral::_type_check( set<IIR_TypeDefinition> * ){
}


set<IIR_TypeDefinition> *
IIRScram_StringLiteral::_get_rval_set(IIR_Boolean (IIR::*constraint_function)() ){
  if( _cmp( this, "null" ) == 0 ){
    return new set<IIR_TypeDefinition>( (IIR_TypeDefinition *)StandardPackage::get_savant_null_type_definition());
  }
  else{
    return _get_rval_set_for_string();
  }
}

IIR*
IIRScram_StringLiteral::_clone() {
  return this;
}

visitor_return_type *IIRScram_StringLiteral::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_StringLiteral(this, arg);
};

