// Copyright (c) 1996-2000 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu

//---------------------------------------------------------------------------

#include "IIRScram_PhysicalLiteral.hh"
#include "IIR_PhysicalUnit.hh"
#include "IIR_Identifier.hh"
#include "resolution_func.hh"
#include "error_func.hh"
#include "set.hh"
#include "IIR_PhysicalTypeDefinition.hh"
#include "savant.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"

using std::cerr;

IIRScram_PhysicalLiteral::~IIRScram_PhysicalLiteral() {}

void 
IIRScram_PhysicalLiteral::_publish_vhdl(ostream &_vhdl_out) {
  ASSERT(get_subtype() != NULL);
  if(get_abstract_literal() != NULL) {
    get_abstract_literal()->_publish_vhdl(_vhdl_out);
  }
  _vhdl_out << " ";
  if( get_unit_name() != NULL ){
    get_unit_name()->get_declarator()->_publish_vhdl(_vhdl_out);
  }
}


ostream & 
IIRScram_PhysicalLiteral::_print( ostream &os ) {
  if(get_abstract_literal() != NULL) {
    os << *get_abstract_literal();
  }
  os << " ";
  os << *get_unit_name()->get_declarator();

  return os;
}


void 
IIRScram_PhysicalLiteral::_publish_cc_lvalue( published_file &_cc_out ) {
  if (get_unit_name()->_get_physical_type()->_is_scalar_type() == TRUE)  {
    _cc_out << get_unit_name()->_get_physical_type()->_get_cc_kernel_type();
  }
  else {
    _cc_out << get_unit_name()->_get_physical_type()->_get_cc_type_name();
  }
  _cc_out << OS("(ObjectBase::VARIABLE,") 
	  << "UniversalLongLongInteger";
  _publish_cc_value( _cc_out );
  _cc_out << CS(")");
}

void
IIRScram_PhysicalLiteral::_publish_cc_universal_type( published_file &_cc_out ) {
  _cc_out << "UniversalLongLongInteger";
}

void
IIRScram_PhysicalLiteral::_publish_cc_universal_value( published_file &_cc_out ) {
  if (get_abstract_literal() != NULL) {
    _cc_out << "longlongint(";
    get_abstract_literal()->_publish_cc_universal_value( _cc_out );
    _cc_out << ")";
  }
  else {
    _cc_out << "(int) 1";
  }
  
  _cc_out << " * ";
  
  ASSERT (get_unit_name() != NULL);

  _cc_out << "longlongint(";
  get_unit_name()->get_multiplier()->_publish_cc_universal_value( _cc_out );
  _cc_out << ")";
}

void 
IIRScram_PhysicalLiteral::_publish_cc_state_object_init( published_file &_cc_out ) {
  _cc_out << "new ";
  _publish_cc_lvalue( _cc_out );
}

void
IIRScram_PhysicalLiteral::_publish_cc_elaborate( published_file &_cc_out ){
  _cc_out << "UniversalLongLongInteger";
  _publish_cc_value( _cc_out );
}

void 
IIRScram_PhysicalLiteral::_publish_cc_wait_data( published_file &_cc_out ) {
  _publish_cc_lvalue( _cc_out );
}

void 
IIRScram_PhysicalLiteral::_publish_cc_value( published_file &_cc_out ) {
  int counter = 0;

  ASSERT ( get_unit_name() != NULL );
  ASSERT ( get_unit_name()->_get_physical_type() != NULL );
  
  IIR_PhysicalUnit *node = get_unit_name()->_get_physical_type()->get_primary_unit();
  _cc_out << "((LONG) (";
  
  if(get_abstract_literal() != NULL) {
    get_abstract_literal()->_publish_cc_value( _cc_out );
  }
  else {
    _cc_out << "(int) 1";
  }
  
  _cc_out << " * longlongint( ((const PhysicalTypeInfo &) ";

  get_unit_name()->_get_physical_type()->_publish_cc_object_type_info( _cc_out, FALSE );
  _cc_out << ").get_scale_info()[";
  if (node != get_unit_name()){
    counter++;
    node = get_unit_name()->_get_physical_type()->units.first();
    while(node != get_unit_name()){
      counter++;
      node = get_unit_name()->_get_physical_type()->units.successor(node);
    }
  }
  _cc_out << counter << "]";
  //get_unit_name()->_get_declarator()->_publish_cc_lvalue( _cc_out ); 
  _cc_out << ")))";
}

void 
IIRScram_PhysicalLiteral::_publish_cc_unit_name( published_file &_cc_out ) {
  get_unit_name()->_get_declarator()->_publish_cc_lvalue( _cc_out );
}

void
IIRScram_PhysicalLiteral::_publish_cc_initialization_value( published_file &_cc_out ) {
  _publish_cc_value( _cc_out );
}

void
IIRScram_PhysicalLiteral::_publish_cc_headers( published_file &_cc_out ) {
  get_unit_name()->_get_physical_type()->_publish_cc_headers( _cc_out );
}

void
IIRScram_PhysicalLiteral::_get_list_of_input_signals( set<IIR> * ){
  // Nothing to be done here.  This needs to be here since
  // IIR_PhysicalLiteral is derived from IIR_Expression.  This merely
  // serves to suppress the message generated by
  // _report_undefined_scram_fn()
}

void 
IIRScram_PhysicalLiteral::_type_check( set<IIR_TypeDefinition> *rval_set ){
  set<IIR_Declaration> unit_decls( get_unit_name() );
  reconcile_sets( &unit_decls, rval_set );
  switch( unit_decls.num_elements() ){
  case 0:{
    ostringstream err;
    err << "Incompatible types in usage";
    report_error( this, err.str() );
    break;
  }

  case 1:{
    // Good they're actually compatible...
    ASSERT( unit_decls.get_element() != NULL );
    ASSERT( unit_decls.get_element()->get_kind() == IIR_PHYSICAL_UNIT );

    set_unit_name( (IIR_PhysicalUnit*)(unit_decls.get_element()) );
    break;
  }

  default:{
    // We built a set of one - how could this be?
    cerr << "Internal error in IIRScram_PhysicalLiteral::_type_check";
    abort();
  }
  }
}

set<IIR_TypeDefinition> *
IIRScram_PhysicalLiteral::_get_rval_set(IIR_Boolean (IIR::*constraint_function)() ){
  set<IIR_TypeDefinition> *retval;
  IIR_PhysicalUnit *my_unit = get_unit_name();
  ASSERT( my_unit != NULL );
  ASSERT( my_unit->_is_iir_declaration() == TRUE );

  retval = new set<IIR_TypeDefinition>( my_unit->_get_physical_type() );

  return retval;
}

IIR_TypeDefinition *
IIRScram_PhysicalLiteral::get_subtype(){
  IIR_PhysicalUnit *my_unit = get_unit_name();
  ASSERT( my_unit != NULL );
  ASSERT( my_unit->_is_iir_declaration() == TRUE );

  return my_unit->_get_physical_type();
}

IIR*
IIRScram_PhysicalLiteral::_clone() {
  return this;
}

IIR_Boolean
IIRScram_PhysicalLiteral::_is_resolved(){
  if( get_subtype() == NULL ){
    return FALSE;
  }
  else{
    return TRUE;
  }
}

visitor_return_type *
IIRScram_PhysicalLiteral::_accept_visitor( node_visitor *visitor, 
					   visitor_argument_type *arg ){
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_PhysicalLiteral(this, arg);
}
